"use client";
import { useEffect, useState, useCallback } from "react";
import Select from "react-select";
import Input from "@/components/shared/input/Input";
import { VALIDATOR_REQUIRE, VALIDATOR_MINLENGTH } from "@/utils/validators";
import { useRouter } from "next/navigation";
import { useForm } from "@/hooks/custom-form-hook";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

const FilingProperties = ({
  partnerCompanies,
  ownCompanies,
  currencies,
  onCreateSuccess,
}) => {
  const router = useRouter();

  const [formState, handleInput] = useForm({}, false);
  const [ownCompanySelected, setOwnCompanySelected] = useState(null);
  const [partnerCompanySelected, setPartnerCompanySelected] = useState(null);
  const [selectedCurrency, setSelectedCurrency] = useState(null);

  useEffect(() => {
    if (!currencies.length) return;
    // Set default currency to HUF if not already selected
    const defaultCurrency =
      currencies.find((c) => c.label === "HUF") || currencies[0] || null;
    setSelectedCurrency((prev) => prev || defaultCurrency);
    // Optionally populate the formState
    if (!formState.inputs.currency_id?.value) {
      handleInput("currency_id", defaultCurrency?.value || "", true);
    }
  }, [currencies]);

  // When own company is selected, get default reg number
  useEffect(() => {
    const generateRegNumber = async () => {
      if (!ownCompanySelected) return;
      try {
        const res = await ApiCall.filing.getNextRegNumber({
          ownCompanyId: ownCompanySelected.value,
        });

        if (res.status === 200) {
          const generated = res.data.registrationNumber;
          handleInput("document_registration_number", generated, true);
        }
      } catch (err) {
        console.error("Iktatószám generálás sikertelen:", err);
        toast.error("Nem sikerült iktatószámot generálni");
      }
    };

    generateRegNumber();
  }, [ownCompanySelected]);

  const handleSubmit = useCallback(async () => {
    const payload = Object.fromEntries(
      Object.entries(formState.inputs).map(([k, v]) => [k, v.value])
    );

    payload.own_company_id = ownCompanySelected?.value;
    payload.partner_company_id = partnerCompanySelected?.value;
    payload.currency_id = selectedCurrency?.value;

    try {
      const res = await ApiCall.filing.postCreate(payload);
      if (res.status === 200) {
        toast.success(res.data.message || "Bejegyzés mentve");
        onCreateSuccess(res.data.id);
        // router.push("/admin/filing/list");
      } else {
        toast.error(res.data.error || "Hiba történt mentéskor");
      }
    } catch (err) {
      toast.error("Mentés sikertelen");
    }
  }, [formState, ownCompanySelected, partnerCompanySelected]);
  return (
    <div>
      {" "}
      {/* <pre className="text-xs text-gray-400">
        {JSON.stringify(formState, null, 2)}
      </pre> */}
      {/* Step 1: Own company */}
      <div className="mb-4">
        <label className="text-sm font-medium">Saját cég</label>
        <Select
          options={ownCompanies}
          placeholder="Válassz saját céget"
          value={ownCompanySelected}
          onChange={(selected) => {
            setOwnCompanySelected(selected);
            handleInput("own_company_id", selected?.value || "", true);
            setPartnerCompanySelected(null); // reset later steps
          }}
        />
      </div>
      {/* Step 2: Partner company */}
      {ownCompanySelected && (
        <div className="mb-4">
          <label className="text-sm font-medium">Partner cég</label>
          <Select
            options={partnerCompanies}
            placeholder="Válassz partnert"
            value={partnerCompanySelected}
            onChange={(selected) => {
              setPartnerCompanySelected(selected);
              handleInput("partner_company_id", selected?.value || "", true);
            }}
          />
        </div>
      )}
      {/* Step 3: Remaining fields */}
      {ownCompanySelected && partnerCompanySelected && (
        <>
          <Input
            id="document_arrived_at"
            name="document_arrived_at"
            element="input"
            type="date"
            label="Dokumentum beérkezett"
            onInput={handleInput}
            validators={[VALIDATOR_REQUIRE()]}
            errorText="Kötelező dátum"
            initialValid={true}
            value={
              formState?.inputs?.document_arrived_at?.value ||
              new Date().toISOString().split("T")[0]
            }
          />

          <Input
            id="invoice_serial_number"
            type="text"
            name="invoice_serial_number"
            element="input"
            label="Számla sorszám"
            onInput={handleInput}
            validators={[VALIDATOR_REQUIRE(), VALIDATOR_MINLENGTH(2)]}
            errorText="Kötelező mező"
            initialValid={false}
            value={formState?.inputs?.invoice_serial_number?.value || ""}
          />
          {formState?.inputs?.invoice_serial_number?.isValid && (
            <>
              <Input
                id="document_registration_number"
                name="document_registration_number"
                type="text"
                element="input"
                label="Iktatószám"
                onInput={handleInput}
                value={
                  formState.inputs.document_registration_number?.value || ""
                }
                readOnly // auto-generated
                initialValid={true}
              />
              {formState.inputs.document_registration_number?.isValid && (
                <>
                  <Input
                    id="payment_deadline"
                    name="payment_deadline"
                    element="input"
                    type="date"
                    label="Fizetési határidő"
                    onInput={handleInput}
                    initialValid={false}
                    value={formState?.inputs?.payment_deadline?.value}
                  />
                </>
              )}
              {formState?.inputs?.payment_deadline?.isValid && (
                <>
                  <Select
                    options={currencies}
                    placeholder="Válassz pénznemet"
                    value={selectedCurrency}
                    onChange={(selected) => {
                      setSelectedCurrency(selected);
                      handleInput("currency_id", selected?.value || "", true);
                    }}
                  />
                  {selectedCurrency && (
                    <>
                      <Input
                        id="invoice_value"
                        name="invoice_value"
                        element="input"
                        type="text"
                        label="Számla érték"
                        onInput={handleInput}
                        initialValid={false}
                        value={formState?.inputs?.invoice_value?.value}
                      />
                      {selectedCurrency &&
                        formState.inputs.invoice_value?.value && (
                          <div className="mt-1 text-gray-600 text-sm">
                            {new Intl.NumberFormat("hu-HU", {
                              style: "currency",
                              currency: selectedCurrency.label, // assuming label holds 'HUF', 'EUR', etc.
                              minimumFractionDigits: 2,
                            }).format(
                              Number(formState.inputs.invoice_value.value) || 0
                            )}
                          </div>
                        )}
                      <button
                        className={`mt-4 px-4 py-2 ${
                          !formState.isValid
                            ? "!bg-gray-400"
                            : "bg-green-600 hover:bg-green-700"
                        } text-white rounded`}
                        onClick={handleSubmit}
                        disabled={!formState.isValid}>
                        Mentés
                      </button>
                    </>
                  )}
                </>
              )}
            </>
          )}
        </>
      )}
    </div>
  );
};

export default FilingProperties;
