// src/components/hr/HrPersonDocumentClient.tsx
"use client";

import { useEffect, useState } from "react";
import toast from "react-hot-toast";

import ApiCall from "@/api/api-calls";
import PersonDocument from "@/components/hr/PersonDocument";
import { useMyApprovals } from "@/hooks/useMyApprovalsBadge";

type TemplateExportSettings = {
  pageSize?: "A4" | "A3" | "A5";
  orientation?: "portrait" | "landscape";
  font?: string;
  fontSizePt?: number;
  lang?: string;
  margins?: {
    topCm?: number;
    bottomCm?: number;
    leftCm?: number;
    rightCm?: number;
    headerCm?: number;
    footerCm?: number;
  };
  table?: {
    cantSplitRows?: boolean;
    addSpacingAfter?: boolean;
  };
};

type FormField = {
  name: string;
  label: string;
  type: "text" | "textarea" | "checkbox" | "date" | "separator";
  required?: boolean;
  defaultValue?: any;
  description?: string | null;
};

type PersonDocMeta = {
  id: string;
  person_id: string;
  title: string;
  status: "draft" | "pending_approval" | "approved" | "rejected" | string;
  form_values?: Record<string, any> | null;
  template_document_id?: string | null;
  company_id?: string | null;
  recipients?: any;
  created_at?: string;
  updated_at?: string;
  created_by?: string;
  updated_by?: string | null;
};

type Approver = {
  id: string;
  approver_user_id: string;
  approver_name: string;
  approver_role?: string | null;
  status: "pending" | "approved" | "rejected" | string;
  decided_at?: string | null;
  comment?: string | null;
  // optionally you can have a boolean from backend:
  // is_me?: boolean;
};

type HrPersonDocumentClientProps = {
  id: string;
  initialHtml: string;
};

export default function HrPersonDocumentClient({
  id,
  initialHtml,
}: HrPersonDocumentClientProps) {
  const [meta, setMeta] = useState<PersonDocMeta | null>(null);
  const [schema, setSchema] = useState<FormField[]>([]);
  const [values, setValues] = useState<Record<string, any>>({});
  const [html, setHtml] = useState(initialHtml);
  const [approvers, setApprovers] = useState<Approver[]>([]);
  const { refreshSummary, loadList } = useMyApprovals();

  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [deciding, setDeciding] = useState(false);
  const [submitting, setSubmitting] = useState(false);

  console.log("approvers", approvers);
  // Load meta + schema + approvers
  useEffect(() => {
    (async () => {
      try {
        setLoading(true);

        const [metaRes, apprRes] = await Promise.all([
          ApiCall.hr.getPersonDocumentMeta(id),
          ApiCall.hr.getPersonDocumentApprovers(id).catch(() => ({
            data: [],
          })), // in case endpoint not present yet
        ]);

        const m = metaRes.data as PersonDocMeta;
        setMeta(m);

        setApprovers((apprRes.data ?? []) as Approver[]);

        setValues((m.form_values ?? {}) as Record<string, any>);

        if (m.template_document_id) {
          const tplRes = await ApiCall.templateDocuments.getItem(
            m.template_document_id
          );
          const tpl = tplRes.data as {
            id: string;
            form_schema?: FormField[] | null;
          };
          setSchema((tpl.form_schema ?? []) as FormField[]);
        } else {
          setSchema([]);
        }
      } catch (e: any) {
        console.error(e);
        toast.error(
          e?.response?.data?.error ??
            "HR dokumentum metaadatainak betöltése sikertelen"
        );
      } finally {
        setLoading(false);
      }
    })();
  }, [id]);

  const reloadMetaAndApprovers = async () => {
    try {
      const [metaRes, apprRes] = await Promise.all([
        ApiCall.hr.getPersonDocumentMeta(id),
        ApiCall.hr.getPersonDocumentApprovers(id).catch(() => ({
          data: [],
        })),
      ]);
      setMeta(metaRes.data as PersonDocMeta);
      setApprovers((apprRes.data ?? []) as Approver[]);
    } catch (e) {
      console.error(e);
    }
  };

  const handleFieldChange = (name: string, value: any) => {
    setValues((prev) => ({ ...prev, [name]: value }));
  };

  const handleSaveValues = async () => {
    if (!meta) return;
    try {
      setSaving(true);
      await ApiCall.hr.updatePersonDocumentValues(id, values);
      toast.success("Mezők mentve");

      // reload HTML preview
      const htmlRes = await ApiCall.hr.getPersonDocument(id);
      setHtml(htmlRes.data as string);

      // reload meta + approvers (status might change)
      await reloadMetaAndApprovers();
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Mezők mentése közben hiba történt"
      );
    } finally {
      setSaving(false);
    }
  };

  const handleDecision = async (decision: "approved" | "rejected") => {
    if (!meta) return;

    const label = decision === "approved" ? "jóváhagyod" : "elutasítod";
    const confirmMsg = `Biztosan ${label} a dokumentumot?`;

    if (!confirm(confirmMsg)) return;

    const comment =
      decision === "rejected"
        ? prompt("Elutasítás oka (opcionális):") ?? undefined
        : undefined;

    try {
      setDeciding(true);
      await ApiCall.hr.decidePersonDocument(id, decision, comment);
      toast.success(
        decision === "approved" ? "Jóváhagyás rögzítve" : "Elutasítás rögzítve"
      );
      await Promise.all([refreshSummary(), loadList()]);

      await reloadMetaAndApprovers();
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ??
          (decision === "approved"
            ? "Jóváhagyás közben hiba történt"
            : "Elutasítás közben hiba történt")
      );
    } finally {
      setDeciding(false);
    }
  };

  const handleSubmitForApproval = async () => {
    if (!meta) return;
    if (!confirm("Biztosan jóváhagyásra küldöd a dokumentumot?")) return;

    try {
      setSubmitting(true);
      await ApiCall.hr.submitPersonDocumentForApproval(id);
      toast.success("Dokumentum jóváhagyásra küldve");
      await Promise.all([refreshSummary(), loadList()]);

      await reloadMetaAndApprovers();
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Jóváhagyásra küldés közben hiba történt"
      );
    } finally {
      setSubmitting(false);
    }
  };

  if (loading) {
    return (
      <div className="flex-1 flex items-center justify-center text-sm text-gray-600">
        Betöltés…
      </div>
    );
  }

  if (!meta) {
    return (
      <div className="flex-1 flex items-center justify-center text-sm text-red-600">
        Dokumentum nem található
      </div>
    );
  }

  const isPending = meta.status === "pending_approval";
  const isDraftLike =
    meta.status === "draft" ||
    meta.status === "rejected" ||
    meta.status === "approved";

  const statusLabelMap: Record<string, string> = {
    draft: "Piszkozat",
    pending_approval: "Jóváhagyásra vár",
    approved: "Jóváhagyva",
    rejected: "Elutasítva",
  };
  const statusLabel = statusLabelMap[meta.status] ?? meta.status;

  const statusColorMap: Record<string, string> = {
    draft: "bg-gray-100 text-gray-700",
    pending_approval: "bg-amber-100 text-amber-800",
    approved: "bg-emerald-100 text-emerald-800",
    rejected: "bg-red-100 text-red-800",
  };
  const statusColor =
    statusColorMap[meta.status] ?? "bg-gray-100 text-gray-700";

  const approverStatusBadge = (status: string) => {
    const base = "px-2 py-0.5 rounded-full text-[11px] font-medium";
    switch (status) {
      case "approved":
        return `${base} bg-emerald-100 text-emerald-800`;
      case "rejected":
        return `${base} bg-red-100 text-red-800`;
      case "pending":
      default:
        return `${base} bg-gray-100 text-gray-700`;
    }
  };

  return (
    <>
      <header className="px-4 py-2 border-b bg-white flex items-center justify-between gap-4">
        <div className="flex flex-col gap-1">
          <h1 className="text-sm font-semibold truncate">
            HR dokumentum – {meta.title}
          </h1>
          <div className="flex items-center gap-2 text-xs text-gray-500">
            <span
              className={`px-2 py-0.5 rounded-full font-medium ${statusColor}`}>
              {statusLabel}
            </span>
            {meta.person_id && <span>Dolgozó ID: {meta.person_id}</span>}
          </div>
        </div>

        <div className="flex items-center gap-2">
          <button
            type="button"
            onClick={handleSaveValues}
            disabled={saving}
            className="px-3 py-1 rounded border border-emerald-500 text-xs text-emerald-700 hover:bg-emerald-50 disabled:opacity-60">
            {saving ? "Mentés…" : "Mezők mentése"}
          </button>

          {isDraftLike && (
            <button
              type="button"
              onClick={handleSubmitForApproval}
              disabled={submitting}
              className="px-3 py-1 rounded border border-amber-500 text-xs text-amber-700 hover:bg-amber-50 disabled:opacity-60">
              {submitting ? "Küldés…" : "Jóváhagyásra küldés"}
            </button>
          )}

          {isPending && (
            <>
              <button
                type="button"
                onClick={() => handleDecision("rejected")}
                disabled={deciding}
                className="px-3 py-1 rounded border border-red-400 text-xs text-red-700 hover:bg-red-50 disabled:opacity-60">
                {deciding ? "Feldolgozás…" : "Elutasítás"}
              </button>
              <button
                type="button"
                onClick={() => handleDecision("approved")}
                disabled={deciding}
                className="px-3 py-1 rounded bg-emerald-600/80 text-xs text-white hover:bg-emerald-600 disabled:bg-emerald-300">
                {deciding ? "Feldolgozás…" : "Jóváhagyás"}
              </button>
            </>
          )}
        </div>
      </header>

      {/* Approvers list */}
      {approvers.length > 0 && (
        <section className="px-4 pt-3 pb-1 bg-white border-b">
          <h2 className="text-xs font-semibold text-gray-700 mb-1">
            Jóváhagyók
          </h2>
          <div className="flex flex-wrap gap-2">
            {approvers.map((a) => (
              <div
                key={a.id}
                className="border border-gray-200 rounded px-2 py-1 flex items-center gap-2 text-[11px] bg-gray-50">
                <span className="font-medium text-gray-800">
                  {a.approver_name}
                </span>
                {a.approver_role && (
                  <span className="text-gray-500">({a.approver_role})</span>
                )}
                <span className={approverStatusBadge(a.status)}>
                  {a.status === "approved"
                    ? "Jóváhagyva"
                    : a.status === "rejected"
                    ? "Elutasítva"
                    : "Függőben"}
                </span>
                {a.comment && (
                  <span className="text-gray-500 italic truncate max-w-[200px]">
                    „{a.comment}”
                  </span>
                )}
              </div>
            ))}
          </div>
        </section>
      )}

      <section className="flex-1 flex flex-col md:flex-row gap-4 p-4 overflow-hidden bg-gray-50">
        {/* LEFT: form fields */}
        <div className="w-full md:w-1/2 flex flex-col gap-3 overflow-auto">
          <div className="border border-gray-200 rounded bg-white p-3">
            <h2 className="text-sm font-semibold mb-2">
              Űrlap mezők (HR szerkesztés)
            </h2>

            {!schema.length && (
              <p className="text-xs text-gray-500">
                Ehhez a dokumentumhoz nincs mentett <code>form_schema</code>.
              </p>
            )}

            <div className="grid grid-cols-1 gap-3">
              {schema.map((field, idx) => {
                const key = field.name || `field_${idx}`;
                const value = values[field.name];

                if (field.type === "separator") {
                  return (
                    <div
                      key={key}
                      className="pt-2 mt-2 border-t border-gray-200">
                      <h3 className="text-xs font-semibold text-gray-800">
                        {field.label || "Szekció"}
                      </h3>
                      {field.description && (
                        <p className="text-[11px] text-gray-500 mt-0.5">
                          {field.description}
                        </p>
                      )}
                    </div>
                  );
                }

                if (field.type === "checkbox") {
                  return (
                    <label
                      key={key}
                      className="flex items-center gap-2 text-xs border rounded px-3 py-2 bg-gray-50">
                      <input
                        type="checkbox"
                        className="w-4 h-4"
                        checked={!!value}
                        onChange={(e) =>
                          handleFieldChange(field.name, e.target.checked)
                        }
                      />
                      <div className="flex flex-col">
                        <span className="text-sm">{field.label}</span>
                        {field.description && (
                          <span className="text-[11px] text-gray-500">
                            {field.description}
                          </span>
                        )}
                      </div>
                    </label>
                  );
                }

                if (field.type === "textarea") {
                  return (
                    <div
                      key={key}
                      className="flex flex-col text-xs text-gray-700">
                      <span className="mb-1">
                        {field.label}
                        {field.required && (
                          <span className="text-red-500"> *</span>
                        )}
                      </span>
                      {field.description && (
                        <p className="text-[11px] text-gray-500 mb-1">
                          {field.description}
                        </p>
                      )}
                      <textarea
                        className="border rounded px-2 py-1 text-sm min-h-[80px]"
                        value={value ?? ""}
                        onChange={(e) =>
                          handleFieldChange(field.name, e.target.value)
                        }
                      />
                    </div>
                  );
                }

                // text / date
                return (
                  <div
                    key={key}
                    className="flex flex-col text-xs text-gray-700">
                    <span className="mb-1">
                      {field.label}
                      {field.required && (
                        <span className="text-red-500"> *</span>
                      )}
                    </span>
                    {field.description && (
                      <p className="text-[11px] text-gray-500 mb-1">
                        {field.description}
                      </p>
                    )}
                    <input
                      type={field.type === "date" ? "date" : "text"}
                      className="border rounded px-2 py-1 text-sm"
                      value={value ?? ""}
                      onChange={(e) =>
                        handleFieldChange(field.name, e.target.value)
                      }
                    />
                  </div>
                );
              })}
            </div>
          </div>
        </div>

        {/* RIGHT: preview */}
        <div className="w-full md:w-1/2 flex flex-col border border-gray-200 rounded bg-white overflow-hidden">
          <div className="px-3 py-2 border-b text-xs text-gray-600">
            Generált dokumentum előnézet
          </div>
          <div className="flex-1 min-h-[300px]">
            <PersonDocument content={html} />
          </div>
        </div>
      </section>
    </>
  );
}
