// src/components/hr/MyApprovalsClient.tsx
"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import { useMyApprovals, MyApprovalItem } from "@/hooks/useMyApprovalsBadge";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";

type ApproverRow = {
  id: string;
  person_document_id: string;
  approver_user_id: string;
  approver_name: string;
  approver_role?: string | null;
  status: "pending" | "approved" | "rejected";
  decided_at?: string | null;
  comment?: string | null;
};

function formatDateTime(iso?: string | null) {
  if (!iso) return "";
  const d = new Date(iso);
  return d.toLocaleString("hu-HU", {
    year: "numeric",
    month: "2-digit",
    day: "2-digit",
    hour: "2-digit",
    minute: "2-digit",
  });
}

function formatDate(iso?: string | null) {
  if (!iso) return "—";
  const d = new Date(iso);
  return d.toLocaleDateString("hu-HU", {
    year: "numeric",
    month: "2-digit",
    day: "2-digit",
  });
}

function ApproverStatusBadge({ status }: { status: string }) {
  const base = "px-2 py-0.5 rounded-full text-[11px] font-medium inline-block";

  if (status === "approved") {
    return (
      <span className={`${base} bg-emerald-100 text-emerald-800`}>
        Jóváhagyva
      </span>
    );
  }
  if (status === "rejected") {
    return (
      <span className={`${base} bg-red-100 text-red-800`}>Elutasítva</span>
    );
  }
  return (
    <span className={`${base} bg-amber-100 text-amber-800`}>Függőben</span>
  );
}

function DocStatusBadge({ status }: { status: string }) {
  const base = "px-2 py-0.5 rounded-full text-[11px] font-medium inline-block";
  switch (status) {
    case "approved":
      return (
        <span className={`${base} bg-emerald-100 text-emerald-800`}>
          Jóváhagyva
        </span>
      );
    case "rejected":
      return (
        <span className={`${base} bg-red-100 text-red-800`}>Elutasítva</span>
      );
    case "pending_approval":
      return (
        <span className={`${base} bg-amber-100 text-amber-800`}>
          Jóváhagyásra vár
        </span>
      );
    default:
      return (
        <span className={`${base} bg-gray-100 text-gray-700`}>{status}</span>
      );
  }
}

/**
 * Expanded content under each document row: list of all approvers
 */
function ApproversForDocument({ documentId }: { documentId: string }) {
  const [rows, setRows] = useState<ApproverRow[]>([]);
  const [loading, setLoading] = useState(true);

  const fetchApprovers = useCallback(async () => {
    setLoading(true);
    try {
      // 🔹 Adjust this call to match your ApiCall structure
      // Endpoint: GET /api/admin/hr/person-documents/:id/approvers
      const res = await ApiCall.hr.getPersonDocumentApprovers(documentId);
      const list: ApproverRow[] = res.data ?? res;
      setRows(list);
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.message ||
          "Jóváhagyók listájának betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  }, [documentId]);

  useEffect(() => {
    fetchApprovers();
  }, [fetchApprovers]);

  if (loading) {
    return (
      <div className="p-3 text-xs text-gray-500">Jóváhagyók betöltése…</div>
    );
  }

  if (!rows.length) {
    return (
      <div className="p-3 text-xs text-gray-500">
        Ehhez a dokumentumhoz nincsenek jóváhagyók beállítva.
      </div>
    );
  }

  return (
    <div className="p-3">
      <div className="text-[11px] text-gray-500 mb-2 font-semibold">
        Jóváhagyók
      </div>
      <table className="min-w-full text-xs border border-gray-100 rounded">
        <thead className="bg-gray-50">
          <tr>
            <th className="px-2 py-1 text-left font-medium text-gray-600">
              Név
            </th>
            <th className="px-2 py-1 text-left font-medium text-gray-600">
              Szerep
            </th>
            <th className="px-2 py-1 text-left font-medium text-gray-600">
              Státusz
            </th>
            <th className="px-2 py-1 text-left font-medium text-gray-600">
              Döntés ideje
            </th>
            <th className="px-2 py-1 text-left font-medium text-gray-600">
              Megjegyzés
            </th>
          </tr>
        </thead>
        <tbody>
          {rows.map((a) => (
            <tr key={a.id} className="border-t border-gray-100">
              <td className="px-2 py-1 text-gray-800">{a.approver_name}</td>
              <td className="px-2 py-1 text-gray-700">
                {a.approver_role ?? "—"}
              </td>
              <td className="px-2 py-1">
                <ApproverStatusBadge status={a.status} />
              </td>
              <td className="px-2 py-1 text-gray-700">
                {a.decided_at ? formatDateTime(a.decided_at) : "—"}
              </td>
              <td className="px-2 py-1 text-gray-700">
                {a.comment?.trim() || "—"}
              </td>
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  );
}

export default function MyApprovalsClient() {
  const router = useRouter();
  const { items, loadingList, loadList } = useMyApprovals();
  const [initialLoaded, setInitialLoaded] = useState(false);
  const [expandedRowId, setExpandedRowId] = useState<string | null>(null);

  useEffect(() => {
    (async () => {
      await loadList();
      setInitialLoaded(true);
    })();
  }, [loadList]);

  const handleOpenDoc = (id: string) => {
    router.push(`/admin/hr/person-documents/${id}`);
  };

  const columns = useMemo(
    () => [
      {
        id: "title",
        name: "Dokumentum címe",
        selector: (r: MyApprovalItem) => r.title,
        cell: (r: MyApprovalItem) => (
          <button
            type="button"
            onClick={() => handleOpenDoc(r.document_id)}
            className="text-xs text-emerald-700 hover:underline text-left"
            title={r.title}>
            {r.title}
          </button>
        ),
        sortable: true,
        grow: 3,
      },
      {
        id: "person_name",
        name: "Dolgozó",
        selector: (r: MyApprovalItem) => r.person_name ?? "",
        cell: (r: MyApprovalItem) => (
          <span className="text-xs text-gray-800">{r.person_name ?? "—"}</span>
        ),
        sortable: true,
        grow: 2,
      },
      {
        id: "approver_role",
        name: "Szerep",
        selector: (r: MyApprovalItem) => r.approver_role ?? "",
        cell: (r: MyApprovalItem) => (
          <span className="text-xs text-gray-700">
            {r.approver_role ?? "—"}
          </span>
        ),
        sortable: true,
        grow: 2,
      },
      {
        id: "status",
        name: "Dokumentum státusz",
        selector: (r: MyApprovalItem) => r.status,
        cell: (r: MyApprovalItem) => <DocStatusBadge status={r.status} />,
        sortable: true,
        grow: 2,
      },
      {
        id: "approver_status",
        name: "Jóváhagyói státuszod",
        selector: (r: MyApprovalItem) => r.approver_status,
        cell: (r: MyApprovalItem) => (
          <ApproverStatusBadge status={r.approver_status} />
        ),
        sortable: true,
        grow: 2,
      },
      {
        id: "created_at",
        name: "Létrehozva",
        selector: (r: MyApprovalItem) => r.created_at ?? "",
        cell: (r: MyApprovalItem) => (
          <div style={{ textAlign: "right" }} className="text-xs text-gray-700">
            {formatDate(r.created_at)}
          </div>
        ),
        sortable: true,
        width: "130px",
      },
      {
        id: "actions",
        name: "",
        selector: () => "",
        cell: (r: MyApprovalItem) => (
          <button
            type="button"
            onClick={() => handleOpenDoc(r.document_id)}
            className="text-[11px] text-emerald-700 hover:underline">
            Dokumentum megnyitása
          </button>
        ),
        ignoreRowClick: true,
        allowOverflow: true,
        button: true,
        width: "160px",
      },
    ],
    [router]
  );

  const renderExpanded = useCallback(
    (row: MyApprovalItem) => {
      if (!expandedRowId || row.document_id !== expandedRowId) return null;
      return <ApproversForDocument documentId={row.document_id} />;
    },
    [expandedRowId]
  );

  // If you eventually want server-side paging/sorting for approvals,
  // you can wire it up like PeopleListPage. For now we keep it simple.
  return (
    <div className="p-3">
      <div className="flex justify-between items-center mb-2">
        <button
          type="button"
          onClick={loadList}
          className="text-[11px] text-emerald-700 hover:underline">
          Frissítés
        </button>
      </div>

      {!initialLoaded && loadingList ? (
        <div className="p-4 text-xs text-gray-500">Betöltés…</div>
      ) : items.length === 0 ? (
        <div className="p-4 text-xs text-gray-500 border rounded bg-white">
          Jelenleg nincs jóváhagyásra váró dokumentumod.
        </div>
      ) : (
        <DataTableWrapper
          tableTitle=""
          data={items}
          columns={columns as any}
          progressPending={loadingList}
          // client-side paging/sorting for now
          pagination
          paginationPerPage={25}
          keyField="approver_id"
          expandableRows
          expandableRowExpanded={(r: MyApprovalItem) =>
            !!expandedRowId && r.document_id === expandedRowId
          }
          onRowExpandToggled={(open: boolean, row: MyApprovalItem) => {
            if (open) {
              setExpandedRowId(row.document_id);
            } else {
              setExpandedRowId((prev) =>
                prev === row.document_id ? null : prev
              );
            }
          }}
          renderExpandableRows={(row: MyApprovalItem) => renderExpanded(row)}
        />
      )}
    </div>
  );
}

// // src/components/hr/MyApprovalsClient.tsx
// "use client";

// import { useEffect, useState } from "react";
// import { useRouter } from "next/navigation";
// import ApiCall from "@/api/api-calls";
// import { useMyApprovals, MyApprovalItem } from "@/hooks/useMyApprovalsBadge";

// export default function MyApprovalsClient() {
//   const router = useRouter();
//   const { items, loadingList, loadList } = useMyApprovals();
//   const [initialLoaded, setInitialLoaded] = useState(false);

//   useEffect(() => {
//     (async () => {
//       await loadList();
//       setInitialLoaded(true);
//     })();
//   }, [loadList]);

//   const handleOpenDoc = (id: string) => {
//     router.push(`/admin/hr/person-documents/${id}`);
//   };

//   const formatDate = (iso?: string | null) => {
//     if (!iso) return "";
//     const d = new Date(iso);
//     return d.toLocaleString("hu-HU", {
//       year: "numeric",
//       month: "2-digit",
//       day: "2-digit",
//       hour: "2-digit",
//       minute: "2-digit",
//     });
//   };
//   console.log(items);
//   return (
//     <div className="bg-white border border-gray-200 rounded shadow-sm h-full flex flex-col">
//       <div className="px-4 py-2 border-b flex justify-between items-center">
//         <button
//           type="button"
//           onClick={loadList}
//           className="text-[11px] text-emerald-700 hover:underline">
//           Frissítés
//         </button>
//       </div>

//       <div className="flex-1 overflow-auto">
//         {loadingList && !initialLoaded ? (
//           <div className="p-4 text-xs text-gray-500">Betöltés…</div>
//         ) : items.length === 0 ? (
//           <div className="p-4 text-xs text-gray-500">
//             Jelenleg nincs jóváhagyásra váró dokumentumod.
//           </div>
//         ) : (
//           <table className="min-w-full text-xs">
//             <thead className="bg-gray-50 border-b">
//               <tr>
//                 <th className="px-3 py-2 text-left font-medium text-gray-600">
//                   Dokumentum címe
//                 </th>
//                 <th className="px-3 py-2 text-left font-medium text-gray-600">
//                   Dolgozó
//                 </th>
//                 <th className="px-3 py-2 text-left font-medium text-gray-600">
//                   Szerep
//                 </th>
//                 <th className="px-3 py-2 text-left font-medium text-gray-600">
//                   Dokumentum státusz
//                 </th>
//                 <th className="px-3 py-2 text-left font-medium text-gray-600">
//                   Jóváhagyói státuszod
//                 </th>
//                 <th className="px-3 py-2 text-left font-medium text-gray-600">
//                   Létrehozva
//                 </th>
//                 <th className="px-3 py-2"></th>
//               </tr>
//             </thead>
//             <tbody>
//               {items.map((item: MyApprovalItem) => (
//                 <tr
//                   key={item.approver_id + item.document_id}
//                   className="border-b">
//                   <td className="px-3 py-2 align-top">
//                     <div className="font-semibold text-gray-800">
//                       {item.title}
//                     </div>
//                   </td>
//                   <td className="px-3 py-2 align-top text-gray-700">
//                     {item.person_name ?? "-"}
//                   </td>
//                   <td className="px-3 py-2 align-top text-gray-700">
//                     {item.approver_role ?? "-"}
//                   </td>
//                   <td className="px-3 py-2 align-top text-gray-700">
//                     {item.status}
//                   </td>
//                   <td className="px-3 py-2 align-top">
//                     <span
//                       className={`px-2 py-0.5 rounded-full text-[11px] font-medium ${
//                         item.approver_status === "approved"
//                           ? "bg-emerald-100 text-emerald-800"
//                           : item.approver_status === "rejected"
//                           ? "bg-red-100 text-red-800"
//                           : "bg-amber-100 text-amber-800"
//                       }`}>
//                       {item.approver_status === "approved"
//                         ? "Jóváhagyva"
//                         : item.approver_status === "rejected"
//                         ? "Elutasítva"
//                         : "Függőben"}
//                     </span>
//                   </td>
//                   <td className="px-3 py-2 align-top text-gray-700">
//                     {formatDate(item.created_at)}
//                   </td>
//                   <td className="px-3 py-2 align-top text-right">
//                     <button
//                       type="button"
//                       onClick={() => handleOpenDoc(item.document_id)}
//                       className="text-[11px] text-emerald-700 hover:underline">
//                       Dokumentum megnyitása
//                     </button>
//                   </td>
//                 </tr>
//               ))}
//             </tbody>
//           </table>
//         )}
//       </div>
//     </div>
//   );
// }
