"use client";

import { useEffect, useState } from "react";
import Select from "react-select";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

type CategoryOption = { value: string; label: string; code?: string };
type DocTemplateOption = { value: string; label: string; code: string };

export type TaskTemplateFormProps = {
  id?: string; // if provided, we are editing
  onSaved?: () => void;
};

export default function TaskTemplateForm({
  id,
  onSaved,
}: TaskTemplateFormProps) {
  const [loading, setLoading] = useState(false);
  const [categories, setCategories] = useState<CategoryOption[]>([]);
  const [docTemplates, setDocTemplates] = useState<DocTemplateOption[]>([]);

  // form state
  const [code, setCode] = useState("");
  const [title, setTitle] = useState("");
  const [description, setDescription] = useState("");
  const [categoryId, setCategoryId] = useState<string | null>(null);
  const [defaultOffsetDays, setDefaultOffsetDays] = useState<string>("");
  const [defaultRole, setDefaultRole] = useState("");
  const [documentCode, setDocumentCode] = useState<string | null>(null);
  const [sortOrder, setSortOrder] = useState<string>("");
  const [isActive, setIsActive] = useState(true);

  const resetForm = () => {
    setCode("");
    setTitle("");
    setDescription("");
    setCategoryId(null);
    setDefaultOffsetDays("");
    setDefaultRole("");
    setDocumentCode(null);
    setSortOrder("");
    setIsActive(true);
  };

  const loadMeta = async () => {
    try {
      const [catRes, docRes] = await Promise.all([
        ApiCall.hr.getTaskCategories(),
        ApiCall.hr.getDocTemplates(),
      ]);

      const catOpts: CategoryOption[] = (catRes.data?.items ?? []).map(
        (c: any) => ({
          value: c.id,
          label: `${c.name} (${c.code})`,
          code: c.code,
        })
      );
      setCategories(catOpts);

      const docOpts: DocTemplateOption[] = (docRes.data?.items ?? []).map(
        (d: any) => ({
          value: d.code,
          label: d.name,
          code: d.code,
        })
      );
      setDocTemplates(docOpts);
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message ||
          "Segédlisták betöltése sikertelen (kategóriák / dokumentum sablonok)"
      );
    }
  };

  const loadExisting = async (templateId: string) => {
    // you’ll need a small API helper for this:
    // ApiCall.hr.getTaskTemplate(templateId)
    try {
      const res = await ApiCall.hr.getTaskTemplate(templateId);
      const row = res.data;

      setCode(row.code);
      setTitle(row.title);
      setDescription(row.description ?? "");
      setCategoryId(row.category_id ?? null);
      setDefaultOffsetDays(
        row.default_offset_days !== null &&
          row.default_offset_days !== undefined
          ? String(row.default_offset_days)
          : ""
      );
      setDefaultRole(row.default_role ?? "");
      setDocumentCode(row.document_code ?? null);
      setSortOrder(
        row.sort_order !== null && row.sort_order !== undefined
          ? String(row.sort_order)
          : ""
      );
      setIsActive(row.is_active);
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message || "HR feladat sablon betöltése sikertelen"
      );
    }
  };

  useEffect(() => {
    loadMeta();
  }, []);

  useEffect(() => {
    if (id) {
      loadExisting(id);
    } else {
      resetForm();
    }
  }, [id]);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!code.trim() || !title.trim()) {
      toast.error("Kód és megnevezés kötelező");
      return;
    }

    setLoading(true);
    try {
      await ApiCall.hr.upsertTaskTemplate({
        id: id ?? undefined,
        code: code.trim(),
        title: title.trim(),
        description: description.trim() || null,
        category_id: categoryId ?? null,
        default_offset_days: defaultOffsetDays
          ? Number(defaultOffsetDays)
          : null,
        default_role: defaultRole.trim() || null,
        document_code: documentCode ?? null,
        sort_order: sortOrder ? Number(sortOrder) : null,
        is_active: isActive,
      });

      toast.success(id ? "Sablon frissítve" : "Sablon létrehozva");
      if (!id) resetForm();
      onSaved?.();
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message || "HR feladat sablon mentése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  };

  return (
    <section className="mt-4 border rounded p-4 bg-gray-50 space-y-3">
      <h2 className="font-semibold text-sm">
        {id ? "Sablon szerkesztése" : "Új sablon létrehozása"}
      </h2>

      <form
        onSubmit={handleSubmit}
        className="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">
        <div>
          <label className="block text-xs font-semibold mb-1">Kód</label>
          <input
            className="border rounded px-2 py-1 w-full"
            value={code}
            onChange={(e) => setCode(e.target.value)}
            placeholder="pl. return_assets"
          />
        </div>

        <div>
          <label className="block text-xs font-semibold mb-1">Megnevezés</label>
          <input
            className="border rounded px-2 py-1 w-full"
            value={title}
            onChange={(e) => setTitle(e.target.value)}
            placeholder="pl. Leltári eszközök visszavétele"
          />
        </div>

        <div className="md:col-span-2">
          <label className="block text-xs font-semibold mb-1">Leírás</label>
          <textarea
            className="border rounded px-2 py-1 w-full min-h-[60px]"
            value={description}
            onChange={(e) => setDescription(e.target.value)}
          />
        </div>

        <div>
          <label className="block text-xs font-semibold mb-1">Kategória</label>
          <Select
            classNamePrefix="hr-cat"
            options={categories}
            value={
              categoryId
                ? categories.find((c) => c.value === categoryId) ?? null
                : null
            }
            onChange={(v) => setCategoryId((v as any)?.value ?? null)}
            isClearable
            placeholder="Válassz kategóriát"
          />
        </div>

        <div>
          <label className="block text-xs font-semibold mb-1">
            Alapértelmezett offset nap (belépés / kilépés napjához képest)
          </label>
          <input
            type="number"
            className="border rounded px-2 py-1 w-full"
            value={defaultOffsetDays}
            onChange={(e) => setDefaultOffsetDays(e.target.value)}
            placeholder="pl. 0, 7, -1"
          />
        </div>

        <div>
          <label className="block text-xs font-semibold mb-1">
            Alapértelmezett felelős szerep
          </label>
          <input
            className="border rounded px-2 py-1 w-full"
            value={defaultRole}
            onChange={(e) => setDefaultRole(e.target.value)}
            placeholder="pl. it, hr, finance"
          />
        </div>

        <div>
          <label className="block text-xs font-semibold mb-1">
            Dokumentum sablon (opcionális)
          </label>
          <Select
            classNamePrefix="hr-doc"
            options={docTemplates}
            value={
              documentCode
                ? docTemplates.find((d) => d.value === documentCode) ?? null
                : null
            }
            onChange={(v) => setDocumentCode((v as any)?.value ?? null)}
            isClearable
            placeholder="Válassz dokumentumot"
          />
        </div>

        <div>
          <label className="block text-xs font-semibold mb-1">Sorrend</label>
          <input
            type="number"
            className="border rounded px-2 py-1 w-full"
            value={sortOrder}
            onChange={(e) => setSortOrder(e.target.value)}
            placeholder="Alapértelmezett: 100"
          />
        </div>

        <div className="flex items-center gap-2 mt-2">
          <input
            id="tpl-active"
            type="checkbox"
            className="h-4 w-4"
            checked={isActive}
            onChange={(e) => setIsActive(e.target.checked)}
          />
          <label htmlFor="tpl-active" className="text-xs">
            Aktív sablon
          </label>
        </div>

        <div className="md:col-span-2 flex justify-end gap-2 mt-2">
          <button
            type="button"
            onClick={resetForm}
            className="px-3 py-1.5 rounded border text-xs">
            Mégse
          </button>
          <button
            type="submit"
            disabled={loading}
            className="px-3 py-1.5 rounded bg-emerald-600 text-white text-xs">
            {id ? "Sablon frissítése" : "Sablon létrehozása"}
          </button>
        </div>
      </form>
    </section>
  );
}
