// components/InventoryRegDocument.jsx
import React, { useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import { toast } from "react-hot-toast";
import { useModalManager } from "@/hooks/useModalManager";
import Modal from "../shared/modal/Modal";
import MediaPreview from "../shared/mediaobject/MediaPreview";
import Select from "react-select";
import apiClient from "@/lib/apiClient";

const InventoryLeaseDocument = ({ assetId, documentTypeId }) => {
  const [docs, setDocs] = useState([]);
  const [filteredDocs, setFilteredDocs] = useState([]);
  const [documentTypes, setDocumentTypes] = useState([]);
  const [selectedDocumentType, setSelectedDocumentType] = useState(null);

  const [file, setFile] = useState(null);
  const [loading, setLoading] = useState(false);

  const { toggleModal, isModalOpen } = useModalManager();
  const [selectedDoc, setSelectedDoc] = useState(null);

  // Blob URL for preview (PDF only)
  const [selectedDocURL, setSelectedDocURL] = useState(null);

  const fetchDocuments = async () => {
    try {
      const res = await ApiCall.inventory.getAssetDocuments(assetId);
      const types = await ApiCall.documents.getDocumentTypes();
      setDocs(res.data);
      setDocumentTypes(types.data);
    } catch (err) {
      toast.error("Failed to load documents");
    }
  };

  useEffect(() => {
    if (assetId) fetchDocuments();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [assetId]);

  useEffect(() => {
    if (docs?.length) {
      const filtered = docs.filter(
        (doc) => doc.document_type_id === documentTypeId
      );
      setFilteredDocs(filtered);
    } else {
      setFilteredDocs([]);
    }
  }, [docs, documentTypeId]);

  useEffect(() => {
    if (selectedDoc && documentTypes.length) {
      const match = documentTypes.find(
        (opt) => opt.id === selectedDoc.document_type_id
      );
      setSelectedDocumentType(match || null);
    } else {
      setSelectedDocumentType(null);
    }
  }, [selectedDoc, documentTypes]);

  const handleUpload = async () => {
    if (!file) return;
    setLoading(true);
    try {
      const formData = new FormData();
      formData.append("file", file);
      formData.append("document_type_id", documentTypeId);
      formData.append("description", "Lízing");
      formData.append("target_column", "device_is_leased");
      await ApiCall.inventory.attachDocumentToAsset(assetId, formData);
      toast.success("Sikeresen hozzáadva.");
      setFile(null);
      fetchDocuments();
    } catch (err) {
      toast.error("Sikertelen feltöltés");
    } finally {
      setLoading(false);
    }
  };

  const handleDelete = async (docId) => {
    try {
      await ApiCall.documents.removeAttachedDocument(docId);
      toast.success("Deleted attachment");
      toggleModal("showDeleteModal");
      setSelectedDoc(null);
      fetchDocuments();
    } catch (err) {
      toast.error("Delete failed");
    }
  };

  // Download with proper filename
  const openDoc = async (doc) => {
    try {
      const url = ApiCall.documents.getDocumentDownloadUrl(doc.id);
      const res = await apiClient.get(url, { responseType: "blob" });

      const cd = res.headers["content-disposition"] || "";
      let filename = doc.file_name || "download";
      const m = /filename\*?=(?:UTF-8''|")?([^;"']+)/i.exec(cd);
      if (m) filename = decodeURIComponent(m[1].replace(/"/g, ""));

      const blob = new Blob([res.data], {
        // keep original type for downloads (not required but fine)
        type: res.headers["content-type"] || "application/octet-stream",
      });
      const href = URL.createObjectURL(blob);

      const a = document.createElement("a");
      a.href = href;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      a.remove();
      URL.revokeObjectURL(href);
    } catch (e) {
      toast.error("Letöltés sikertelen");
    }
  };

  // Build a blob URL for preview when a doc is selected
  useEffect(() => {
    let revoke;
    const loadPreview = async () => {
      if (!selectedDoc) return;
      try {
        // Only attempt preview for PDFs; others will show a message + download button
        if (selectedDoc.mime_type !== "application/pdf") {
          setSelectedDocURL(null);
          return;
        }

        const url = ApiCall.documents.getDocumentDownloadUrl(selectedDoc.id);
        const res = await apiClient.get(url, { responseType: "blob" });

        // Critically: force the blob to the known MIME so <object> renders it
        const blob = new Blob([res.data], { type: selectedDoc.mime_type });
        const href = URL.createObjectURL(blob);
        console.log("blob", blob, "href", href);
        setSelectedDocURL(href);
        revoke = () => URL.revokeObjectURL(href);
      } catch (e) {
        setSelectedDocURL(null);
        toast.error("Előnézet nem tölthető be");
      }
    };

    loadPreview();
    return () => {
      if (revoke) revoke();
      setSelectedDocURL(null);
    };
  }, [selectedDoc]);

  return (
    <div className="w-full p-4 border rounded">
      <h2 className="font-bold mb-2">Lízing dokumentumok</h2>

      <div
        className="border-2 border-dashed border-gray-400 p-6 rounded-lg text-center cursor-pointer hover:border-blue-500 transition-colors"
        onDragOver={(e) => e.preventDefault()}
        onDrop={(e) => {
          e.preventDefault();
          if (e.dataTransfer.files.length > 0) {
            setFile(e.dataTransfer.files[0]);
          }
        }}
        onClick={() => document.getElementById("fileInput").click()}>
        <p className="text-gray-600">
          Húzza ide a fájlt vagy kattintson a kiválasztáshoz
        </p>
        {file && <p className="mt-2 font-medium text-sm">{file.name}</p>}
        <input
          type="file"
          id="fileInput"
          className="hidden"
          onChange={(e) => setFile(e.target.files[0])}
        />
      </div>

      <button
        className="w-full bg-green-500 text-white px-4 py-2 mt-2 rounded disabled:opacity-50 disabled:bg-gray-500"
        onClick={handleUpload}
        disabled={!file || loading}>
        Feltöltés
      </button>

      <ul className="list-disc pl-6 py-2">
        {filteredDocs.map((doc) => (
          <li key={doc.id} className="w-full flex items-center gap-3 mb-1">
            <span className="truncate">{doc.file_name}</span>
            <span className="text-gray-500">
              ({(doc.file_size / 1024).toFixed(1)} KB)
            </span>
            <button
              className="ml-4 text-gray-600 underline"
              onClick={() => {
                setSelectedDoc(doc);
                toggleModal("showDetailsModal");
              }}>
              Részletek
            </button>
            <button
              className="ml-auto text-red-500 underline"
              onClick={() => {
                setSelectedDoc(doc);
                toggleModal("showDeleteModal");
              }}>
              Csatolmány eltávolítása
            </button>
          </li>
        ))}
      </ul>

      {/* Delete modal */}
      <Modal
        title={(selectedDoc?.file_name || "") + " eltávolítása"}
        show={isModalOpen("showDeleteModal")}
        onClose={() => {
          toggleModal("showDeleteModal");
          setSelectedDoc(null);
        }}
        height="auto"
        width="max-content">
        <div className="p-4 w-full flex justify-evenly items-center">
          <button
            className="mr-2 text-red-500 underline"
            onClick={() => selectedDoc && handleDelete(selectedDoc.id)}>
            Csatolt fájl eltávolítása
          </button>
          <button
            className="ml-2 text-gray-500 underline"
            onClick={() => {
              toggleModal("showDeleteModal");
              setSelectedDoc(null);
            }}>
            Mégse
          </button>
        </div>
      </Modal>

      {/* Details modal */}
      <Modal
        title="Dokumentum részletei"
        show={isModalOpen("showDetailsModal")}
        onClose={() => {
          toggleModal("showDetailsModal");
          setSelectedDoc(null);
        }}
        height="auto">
        {selectedDoc ? (
          <div className="space-y-2 text-sm">
            <div className="w-full grid grid-cols-4 gap-4 mx-2">
              {/* PDF inline preview via blob URL */}
              {selectedDoc?.mime_type === "application/pdf" &&
                selectedDocURL && (
                  <MediaPreview
                    data={selectedDoc}
                    index={0}
                    preview={selectedDocURL}
                    className="w-full h-auto min-h-[800px] col-span-2"
                  />
                )}

              {/* Non-PDF fallback */}
              {selectedDoc?.mime_type !== "application/pdf" && (
                <div className="col-span-2 p-4 rounded border bg-gray-50 text-gray-700">
                  Ehhez a fájltípushoz nincs beágyazott előnézet.
                  <div className="mt-2">
                    <button
                      onClick={() => openDoc(selectedDoc)}
                      className="underline text-blue-600">
                      Letöltés
                    </button>
                  </div>
                </div>
              )}

              {/* Right-side details & actions */}
              <div className="w-full col-span-2 px-4 py-2">
                <p className="py-2">
                  <strong>Fájlnév:</strong> {selectedDoc.file_name}
                </p>
                <p className="py-2">
                  <strong>Feltöltve:</strong>{" "}
                  {new Date(selectedDoc.uploaded_at).toLocaleString()}
                </p>
                <p className="py-2">
                  <strong>Feltöltő személy:</strong>{" "}
                  {selectedDoc.uploaded_by_username}
                </p>

                {selectedDoc?.updated_at !== selectedDoc.uploaded_at && (
                  <p className="py-2">
                    <strong>Frissítve:</strong>{" "}
                    {new Date(selectedDoc.updated_at).toLocaleString()}
                  </p>
                )}
                {selectedDoc?.modified_by_username && (
                  <p className="py-2">
                    <strong>Utoljára frissítette:</strong>{" "}
                    {selectedDoc.modified_by_username}
                  </p>
                )}
                <p className="py-2">
                  <strong>Méret:</strong>{" "}
                  {(selectedDoc.file_size / 1024).toFixed(1)} KB
                </p>
                <p className="py-2">
                  <strong>Típus:</strong> {selectedDoc.mime_type}
                </p>

                <Select
                  options={documentTypes}
                  className="w-full"
                  menuPosition="fixed"
                  placeholder="Válassza ki, hogy mi legyen a dokumentum típusa."
                  backspaceRemovesValue={false}
                  name="documentType"
                  value={selectedDocumentType}
                  onChange={(val) => setSelectedDocumentType(val)}
                  required
                />

                <div className="py-2">
                  <label className="font-semibold block mb-1">Leírás:</label>
                  <textarea
                    className="w-full border p-2 rounded"
                    rows={2}
                    value={selectedDoc.description || ""}
                    onChange={(e) =>
                      setSelectedDoc((prev) => ({
                        ...prev,
                        description: e.target.value,
                      }))
                    }
                  />
                </div>

                <div className="flex gap-3 mt-2">
                  <button
                    onClick={() => openDoc(selectedDoc)}
                    className="underline text-blue-600">
                    Letöltés
                  </button>
                </div>

                <button
                  onClick={async () => {
                    try {
                      await ApiCall.documents.updateDocumentMetadata(
                        selectedDoc.id,
                        {
                          document_type_id:
                            selectedDocumentType?.id ||
                            selectedDoc.document_type_id,
                          description: selectedDoc.description,
                        }
                      );
                      toast.success("Dokumentum frissítve");
                      fetchDocuments();
                      toggleModal("showDetailsModal");
                    } catch (err) {
                      toast.error("Frissítés sikertelen");
                    }
                  }}
                  className="w-full cursor-pointer mt-4 bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700">
                  Mentés
                </button>
              </div>
            </div>
          </div>
        ) : (
          <p>Nincs kiválasztott dokumentum.</p>
        )}
      </Modal>
    </div>
  );
};

export default InventoryLeaseDocument;
