"use client";

import { useEffect, useMemo, useState } from "react";
import Link from "next/link";
import AnimateHeight from "react-animate-height";
import { FaChevronDown } from "react-icons/fa";

import * as FaIcons from "react-icons/fa";
import * as MdIcons from "react-icons/md";
import * as HiIcons from "react-icons/hi";
import * as BsIcons from "react-icons/bs";
import * as AiIcons from "react-icons/ai";
import clsx from "clsx";

const LIBS = [FaIcons, MdIcons, HiIcons, BsIcons, AiIcons] as const;

const resolveIcon = (raw: unknown) => {
  const fallback = FaIcons.FaRegQuestionCircle;
  if (typeof raw !== "string") return fallback;

  const name = raw.trim();

  for (const lib of LIBS) {
    // @ts-ignore dynamic lookup
    if (lib[name]) return lib[name];
  }

  const [, maybeBase] = name.includes(":") ? name.split(":") : [null, name];
  const pascal = maybeBase
    .replace(/[^a-zA-Z0-9]+/g, " ")
    .trim()
    .split(" ")
    .filter(Boolean)
    .map((w) => w[0].toUpperCase() + w.slice(1))
    .join("");

  for (const prefix of ["Fa", "Md", "Hi", "Bs", "Ai"]) {
    const candidate = `${prefix}${pascal}`;
    for (const lib of LIBS) {
      // @ts-ignore
      if (lib[candidate]) return lib[candidate];
    }
  }

  return fallback;
};

type MenuItem = {
  id: string | number;
  adminmenutitle: string;
  adminmenulink?: string | null;
  icon?: string | null;
  children?: MenuItem[];
};

type Props = {
  item: MenuItem;
  level?: number;
  isSidebarOpen: boolean;
  currentPath: string;
};

const RenderMenuItem = ({
  item,
  level = 0,
  isSidebarOpen,
  currentPath,
}: Props) => {
  const hasChildren = !!(item.children && item.children.length);
  const Icon = resolveIcon(item?.icon);

  const isSelfActive = !!(
    item.adminmenulink && item.adminmenulink === currentPath
  );
  const isDescActive = useMemo(() => {
    const walk = (n?: MenuItem): boolean => {
      if (!n) return false;
      if (n.adminmenulink === currentPath) return true;
      if (!n.children) return false;
      return n.children.some(walk);
    };
    return walk(item);
  }, [item, currentPath]);

  const [isOpen, setIsOpen] = useState<boolean>(isDescActive);
  useEffect(() => {
    if (isDescActive) setIsOpen(true);
  }, [isDescActive]);

  // indent when open only (labels hidden when collapsed)
  const levelIndent = Math.min(level, 3);
  const indentOpen =
    levelIndent === 0 ? "pl-0" : levelIndent === 1 ? "pl-2" : "pl-4";

  if (hasChildren) {
    return (
      <li className="menu nav-item">
        <button
          type="button"
          title={item.adminmenutitle}
          className={clsx(
            "nav-link group w-full no-before-icon flex items-center justify-between cursor-pointer hover:bg-blue-100 mb-[0.25rem] p-[0.625rem]",
            "overflow-hidden",
            isOpen && isSidebarOpen ? "bg-gray-100" : ""
          )}
          onClick={() => isSidebarOpen && setIsOpen((s) => !s)} // disabled toggle when collapsed
          aria-expanded={isSidebarOpen ? isOpen : undefined}>
          <div
            className={clsx(
              "flex items-center min-w-0",
              isSidebarOpen ? indentOpen : "pl-0"
            )}>
            <Icon className="shrink-0 group-hover:!text-primary text-[0.85rem]" />

            {/* label that collapses cleanly */}
            <span
              className={clsx(
                "text-black text-[0.7rem] font-semibold dark:text-[#506690] dark:group-hover:text-white-dark",
                "transition-all duration-200 origin-left whitespace-nowrap overflow-hidden",
                isSidebarOpen
                  ? "ml-3 max-w-[170px] opacity-100"
                  : "ml-0 max-w-0 opacity-0 pointer-events-none"
              )}>
              {item.adminmenutitle}
            </span>
          </div>

          {/* caret hidden when collapsed */}
          {isSidebarOpen && (
            <div
              className={!isOpen ? "-rotate-90 rtl:rotate-90" : ""}
              aria-hidden>
              <FaChevronDown className="h-3 w-3" />
            </div>
          )}
        </button>

        {/* sub menu only renders (height>0) when sidebar is open */}
        <AnimateHeight
          duration={300}
          height={isSidebarOpen && isOpen ? "auto" : 0}>
          <ul
            className={clsx(
              "sub-menu text-gray-500",
              isSidebarOpen ? indentOpen : ""
            )}>
            {item.children!.map((child) => (
              <RenderMenuItem
                key={child.id}
                item={child}
                level={level + 1}
                isSidebarOpen={isSidebarOpen}
                currentPath={currentPath}
              />
            ))}
          </ul>
        </AnimateHeight>
      </li>
    );
  }

  // leaf
  return (
    <li className="menu nav-item">
      <Link
        href={item.adminmenulink || "#"}
        title={item.adminmenutitle}
        className={clsx(
          "nav-link group w-full flex items-center justify-between no-before-icon cursor-pointer mb-[0.25rem] p-[0.625rem] hover:bg-blue-100",
          "overflow-hidden",
          isSelfActive ? "active bg-gray-100" : ""
        )}
        aria-current={isSelfActive ? "page" : undefined}>
        <div
          className={clsx(
            "flex items-center min-w-0",
            isSidebarOpen ? indentOpen : "pl-0"
          )}>
          <Icon className="text-[0.85rem] shrink-0 group-hover:!text-primary text-lg" />
          <span
            className={clsx(
              "text-black text-[0.7rem] dark:text-[#506690] dark:group-hover:text-white-dark",
              "transition-all duration-200 origin-left whitespace-nowrap overflow-hidden",
              isSidebarOpen
                ? "ml-3 max-w-[170px] opacity-100"
                : "ml-0 max-w-0 opacity-0 pointer-events-none"
            )}>
            {item.adminmenutitle}
          </span>
        </div>
      </Link>
    </li>
  );
};

export default RenderMenuItem;

// "use client";

// import { useState } from "react";
// import { FaChevronDown } from "react-icons/fa";
// import Link from "next/link";
// import * as FaIcons from "react-icons/fa";
// import * as MdIcons from "react-icons/md";
// import * as HiIcons from "react-icons/hi";
// import * as BsIcons from "react-icons/bs";
// import * as AiIcons from "react-icons/ai";

// const iconLibs = {
//   fa: FaIcons,
//   md: MdIcons,
//   hi: HiIcons,
//   bs: BsIcons,
//   ai: AiIcons,
// };

// const RenderMenuItem = ({ item, level = 0, isSidebarOpen }) => {
//   const [isOpen, setIsOpen] = useState(false);
//   const hasChildren = item.children && item.children.length > 0;

//   const getIconComponent = (
//     iconName,
//     fallback = FaIcons.FaRegQuestionCircle
//   ) => {
//     if (typeof iconName !== "string") return fallback;
//     const prefix = iconName.slice(0, 2).toLowerCase();
//     const lib = iconLibs[prefix];
//     return lib?.[iconName] || fallback;
//   };

//   const Icon = getIconComponent(item.icon);

//   return (
//     <div className="group">
//       <div
//         onClick={() => (hasChildren ? setIsOpen((prev) => !prev) : null)}
//         className={`cursor-pointer hover:bg-gray-700 transition-all rounded-md flex items-center ${
//           isSidebarOpen
//             ? "px-3 py-2 gap-2 text-white"
//             : "p-3 justify-center text-teal-300"
//         }`}>
//         <Icon className="text-lg" />

//         {isSidebarOpen && (
//           <>
//             {hasChildren ? (
//               // If it has children, just display the text (no link here)
//               <span className="flex-1 truncate text-[0.9rem]">
//                 {item.adminmenutitle}
//               </span>
//             ) : (
//               // If it doesn't have children, make the whole row a link
//               <Link
//                 href={item.adminmenulink || "#"}
//                 className="flex-1 truncate text-[0.9rem]">
//                 {item.adminmenutitle}
//               </Link>
//             )}

//             {hasChildren && (
//               <FaChevronDown
//                 className={`transition-transform duration-300 ${
//                   isOpen ? "rotate-180" : "rotate-0"
//                 }`}
//               />
//             )}
//           </>
//         )}
//       </div>

//       {hasChildren && isOpen && (
//         <ul
//           className={`pl-${
//             (level + 1) * 4
//           } mt-2 space-y-1 border-l border-gray-700`}>
//           {item.children.map((child) => (
//             <RenderMenuItem
//               key={child.id}
//               item={child}
//               level={level + 1}
//               isSidebarOpen={isSidebarOpen}
//             />
//           ))}
//         </ul>
//       )}
//     </div>
//   );
// };

// export default RenderMenuItem;
