"use client";

import { useEffect, useMemo, useRef, useState } from "react";
import { Card, CardBody, Chip, TextInput } from "@/components/ui/CardBits";

export type LookupOption = {
  id: string;
  label: string;
  meta?: string;
};

function useDebounced<T>(value: T, delay = 250) {
  const [v, setV] = useState(value);
  useEffect(() => {
    const t = setTimeout(() => setV(value), delay);
    return () => clearTimeout(t);
  }, [value, delay]);
  return v;
}

type BaseProps = {
  label?: string;
  placeholder?: string;
  disabled?: boolean;
  minChars?: number;
  fetcher: (q: string) => Promise<LookupOption[]>;
};

export function LookupSelect({
  value,
  onChange,
  ...p
}: BaseProps & {
  value: LookupOption | null;
  onChange: (v: LookupOption | null) => void;
}) {
  const [q, setQ] = useState("");
  const dq = useDebounced(q);
  const [open, setOpen] = useState(false);
  const [items, setItems] = useState<LookupOption[]>([]);
  const [loading, setLoading] = useState(false);

  const boxRef = useRef<HTMLDivElement | null>(null);
  const minChars = p.minChars ?? 1;

  // ✅ fix: fetcher ref (ne legyen dependency loop)
  const fetcherRef = useRef(p.fetcher);
  useEffect(() => {
    fetcherRef.current = p.fetcher;
  }, [p.fetcher]);

  useEffect(() => {
    function onDocClick(e: MouseEvent) {
      if (!boxRef.current) return;
      if (!boxRef.current.contains(e.target as any)) setOpen(false);
    }
    document.addEventListener("mousedown", onDocClick);
    return () => document.removeEventListener("mousedown", onDocClick);
  }, []);

  useEffect(() => {
    let alive = true;

    async function run() {
      const term = dq.trim();

      if (term.length < minChars) {
        // ✅ ne renderelj feleslegesen
        setLoading(false);
        setItems((prev) => (prev.length ? [] : prev));
        return;
      }

      setLoading(true);
      try {
        const r = await fetcherRef.current(term);
        if (alive) setItems(r);
      } finally {
        if (alive) setLoading(false);
      }
    }

    void run();
    return () => {
      alive = false;
    };
    // ✅ csak dq + minChars (fetcherRef nem változik)
  }, [dq, minChars]);

  return (
    <div ref={boxRef} className="relative">
      {p.label ? (
        <div className="mb-1 text-xs font-medium text-slate-600">{p.label}</div>
      ) : null}

      <TextInput
        disabled={p.disabled}
        value={q}
        onChange={(e) => {
          setQ(e.target.value);
          setOpen(true);
        }}
        onFocus={() => setOpen(true)}
        placeholder={p.placeholder ?? "Keresés…"}
        style={{ width: "100%" }}
      />

      {value ? (
        <div className="mt-2 flex items-center gap-2">
          <Chip tone="info" dot={false}>
            {value.label}
          </Chip>
          <button
            className="text-xs text-slate-600 hover:text-slate-900"
            onClick={() => onChange(null)}
            type="button">
            törlés
          </button>
        </div>
      ) : null}

      {open ? (
        <div className="absolute z-50 mt-2 w-full">
          <Card>
            <CardBody className="p-2">
              <div className="mb-2 text-xs text-slate-500">
                {loading
                  ? "Keresés…"
                  : items.length
                  ? "Találatok"
                  : `Írj be legalább ${minChars} karaktert`}
              </div>

              <div className="max-h-64 overflow-auto">
                {items.map((it) => (
                  <button
                    key={it.id}
                    type="button"
                    className="w-full rounded-lg px-2 py-2 text-left hover:bg-slate-50"
                    onClick={() => {
                      onChange(it);
                      setOpen(false);
                      setQ("");
                    }}>
                    <div className="text-sm font-medium text-slate-900">
                      {it.label}
                    </div>
                    {it.meta ? (
                      <div className="text-xs text-slate-600">{it.meta}</div>
                    ) : null}
                  </button>
                ))}
              </div>
            </CardBody>
          </Card>
        </div>
      ) : null}
    </div>
  );
}

export function LookupMultiSelect({
  value,
  onChange,
  ...p
}: BaseProps & {
  value: LookupOption[];
  onChange: (v: LookupOption[]) => void;
}) {
  const [q, setQ] = useState("");
  const dq = useDebounced(q);
  const [open, setOpen] = useState(false);
  const [items, setItems] = useState<LookupOption[]>([]);
  const [loading, setLoading] = useState(false);

  const boxRef = useRef<HTMLDivElement | null>(null);
  const minChars = p.minChars ?? 1;

  const selectedIds = useMemo(() => new Set(value.map((v) => v.id)), [value]);

  // ✅ fix: fetcher ref
  const fetcherRef = useRef(p.fetcher);
  useEffect(() => {
    fetcherRef.current = p.fetcher;
  }, [p.fetcher]);

  useEffect(() => {
    function onDocClick(e: MouseEvent) {
      if (!boxRef.current) return;
      if (!boxRef.current.contains(e.target as any)) setOpen(false);
    }
    document.addEventListener("mousedown", onDocClick);
    return () => document.removeEventListener("mousedown", onDocClick);
  }, []);

  useEffect(() => {
    let alive = true;

    async function run() {
      const term = dq.trim();

      if (term.length < minChars) {
        setLoading(false);
        setItems((prev) => (prev.length ? [] : prev));
        return;
      }

      setLoading(true);
      try {
        const r = await fetcherRef.current(term);
        if (alive) setItems(r.filter((x) => !selectedIds.has(x.id)));
      } finally {
        if (alive) setLoading(false);
      }
    }

    void run();
    return () => {
      alive = false;
    };
  }, [dq, minChars, selectedIds]);

  return (
    <div ref={boxRef} className="relative">
      {p.label ? (
        <div className="mb-1 text-xs font-medium text-slate-600">{p.label}</div>
      ) : null}

      <TextInput
        disabled={p.disabled}
        value={q}
        onChange={(e) => {
          setQ(e.target.value);
          setOpen(true);
        }}
        onFocus={() => setOpen(true)}
        placeholder={p.placeholder ?? "Keresés…"}
        style={{ width: "100%" }}
      />

      {value.length ? (
        <div className="mt-2 flex flex-wrap gap-2">
          {value.map((v) => (
            <span key={v.id} className="inline-flex items-center gap-2">
              <Chip tone="info" dot={false}>
                {v.label}
              </Chip>
              <button
                type="button"
                className="text-xs text-slate-600 hover:text-slate-900"
                onClick={() => onChange(value.filter((x) => x.id !== v.id))}>
                ✕
              </button>
            </span>
          ))}
        </div>
      ) : null}

      {open ? (
        <div className="absolute z-50 mt-2 w-full">
          <Card>
            <CardBody className="p-2">
              <div className="mb-2 text-xs text-slate-500">
                {loading
                  ? "Keresés…"
                  : items.length
                  ? "Találatok"
                  : `Írj be legalább ${minChars} karaktert`}
              </div>

              <div className="max-h-64 overflow-auto">
                {items.map((it) => (
                  <button
                    key={it.id}
                    type="button"
                    className="w-full rounded-lg px-2 py-2 text-left hover:bg-slate-50"
                    onClick={() => {
                      onChange([...value, it]);
                      setQ("");
                    }}>
                    <div className="text-sm font-medium text-slate-900">
                      {it.label}
                    </div>
                    {it.meta ? (
                      <div className="text-xs text-slate-600">{it.meta}</div>
                    ) : null}
                  </button>
                ))}
              </div>
            </CardBody>
          </Card>
        </div>
      ) : null}
    </div>
  );
}

// "use client";

// import { useEffect, useMemo, useRef, useState } from "react";
// import { Chip, TextInput, Card, CardBody } from "@/components/ui/CardBits";

// export type LookupOption = {
//   id: string;
//   label: string;
//   meta?: string;
// };

// function useDebounced<T>(value: T, delay = 250) {
//   const [v, setV] = useState(value);
//   useEffect(() => {
//     const t = setTimeout(() => setV(value), delay);
//     return () => clearTimeout(t);
//   }, [value, delay]);
//   return v;
// }

// type BaseProps = {
//   label?: string;
//   placeholder?: string;
//   disabled?: boolean;
//   minChars?: number;
//   fetcher: (q: string) => Promise<LookupOption[]>;
// };

// export function LookupSelect({
//   value,
//   onChange,
//   ...p
// }: BaseProps & {
//   value: LookupOption | null;
//   onChange: (v: LookupOption | null) => void;
// }) {
//   const [q, setQ] = useState("");
//   const dq = useDebounced(q);
//   const [open, setOpen] = useState(false);
//   const [items, setItems] = useState<LookupOption[]>([]);
//   const [loading, setLoading] = useState(false);

//   const boxRef = useRef<HTMLDivElement | null>(null);
//   const minChars = p.minChars ?? 1;

//   useEffect(() => {
//     function onDocClick(e: MouseEvent) {
//       if (!boxRef.current) return;
//       if (!boxRef.current.contains(e.target as any)) setOpen(false);
//     }
//     document.addEventListener("mousedown", onDocClick);
//     return () => document.removeEventListener("mousedown", onDocClick);
//   }, []);

//   useEffect(() => {
//     let alive = true;
//     async function run() {
//       const term = dq.trim();
//       if (term.length < minChars) {
//         setItems([]);
//         return;
//       }
//       setLoading(true);
//       try {
//         const r = await p.fetcher(term);
//         if (alive) setItems(r);
//       } finally {
//         if (alive) setLoading(false);
//       }
//     }
//     void run();
//     return () => {
//       alive = false;
//     };
//   }, [dq, minChars, p]);

//   return (
//     <div ref={boxRef} className="relative">
//       {p.label ? (
//         <div className="mb-1 text-xs font-medium text-slate-600">{p.label}</div>
//       ) : null}

//       <TextInput
//         disabled={p.disabled}
//         value={q}
//         onChange={(e) => {
//           setQ(e.target.value);
//           setOpen(true);
//         }}
//         onFocus={() => setOpen(true)}
//         placeholder={p.placeholder ?? "Keresés…"}
//         style={{ width: "100%" }}
//       />

//       {value ? (
//         <div className="mt-2 flex items-center gap-2">
//           <Chip tone="info" dot={false}>
//             {value.label}
//           </Chip>
//           <button
//             className="text-xs text-slate-600 hover:text-slate-900"
//             onClick={() => onChange(null)}
//             type="button">
//             törlés
//           </button>
//         </div>
//       ) : null}

//       {open ? (
//         <div className="absolute z-50 mt-2 w-full">
//           <Card>
//             <CardBody className="p-2">
//               <div className="mb-2 text-xs text-slate-500">
//                 {loading
//                   ? "Keresés…"
//                   : items.length
//                   ? "Találatok"
//                   : `Írj be legalább ${minChars} karaktert`}
//               </div>

//               <div className="max-h-64 overflow-auto">
//                 {items.map((it) => (
//                   <button
//                     key={it.id}
//                     type="button"
//                     className="w-full rounded-lg px-2 py-2 text-left hover:bg-slate-50"
//                     onClick={() => {
//                       onChange(it);
//                       setOpen(false);
//                       setQ("");
//                     }}>
//                     <div className="text-sm font-medium text-slate-900">
//                       {it.label}
//                     </div>
//                     {it.meta ? (
//                       <div className="text-xs text-slate-600">{it.meta}</div>
//                     ) : null}
//                   </button>
//                 ))}
//               </div>
//             </CardBody>
//           </Card>
//         </div>
//       ) : null}
//     </div>
//   );
// }

// export function LookupMultiSelect({
//   value,
//   onChange,
//   ...p
// }: BaseProps & {
//   value: LookupOption[];
//   onChange: (v: LookupOption[]) => void;
// }) {
//   const [q, setQ] = useState("");
//   const dq = useDebounced(q);
//   const [open, setOpen] = useState(false);
//   const [items, setItems] = useState<LookupOption[]>([]);
//   const [loading, setLoading] = useState(false);

//   const boxRef = useRef<HTMLDivElement | null>(null);
//   const minChars = p.minChars ?? 1;

//   const selectedIds = useMemo(() => new Set(value.map((v) => v.id)), [value]);

//   useEffect(() => {
//     function onDocClick(e: MouseEvent) {
//       if (!boxRef.current) return;
//       if (!boxRef.current.contains(e.target as any)) setOpen(false);
//     }
//     document.addEventListener("mousedown", onDocClick);
//     return () => document.removeEventListener("mousedown", onDocClick);
//   }, []);

//   useEffect(() => {
//     let alive = true;
//     async function run() {
//       const term = dq.trim();
//       if (term.length < minChars) {
//         setItems([]);
//         return;
//       }
//       setLoading(true);
//       try {
//         const r = await p.fetcher(term);
//         if (alive) setItems(r.filter((x) => !selectedIds.has(x.id)));
//       } finally {
//         if (alive) setLoading(false);
//       }
//     }
//     void run();
//     return () => {
//       alive = false;
//     };
//   }, [dq, minChars, p, selectedIds]);

//   return (
//     <div ref={boxRef} className="relative">
//       {p.label ? (
//         <div className="mb-1 text-xs font-medium text-slate-600">{p.label}</div>
//       ) : null}

//       <TextInput
//         disabled={p.disabled}
//         value={q}
//         onChange={(e) => {
//           setQ(e.target.value);
//           setOpen(true);
//         }}
//         onFocus={() => setOpen(true)}
//         placeholder={p.placeholder ?? "Keresés…"}
//         style={{ width: "100%" }}
//       />

//       {value.length ? (
//         <div className="mt-2 flex flex-wrap gap-2">
//           {value.map((v) => (
//             <span key={v.id} className="inline-flex items-center gap-2">
//               <Chip tone="info" dot={false}>
//                 {v.label}
//               </Chip>
//               <button
//                 type="button"
//                 className="text-xs text-slate-600 hover:text-slate-900"
//                 onClick={() => onChange(value.filter((x) => x.id !== v.id))}>
//                 ✕
//               </button>
//             </span>
//           ))}
//         </div>
//       ) : null}

//       {open ? (
//         <div className="absolute z-50 mt-2 w-full">
//           <Card>
//             <CardBody className="p-2">
//               <div className="mb-2 text-xs text-slate-500">
//                 {loading
//                   ? "Keresés…"
//                   : items.length
//                   ? "Találatok"
//                   : `Írj be legalább ${minChars} karaktert`}
//               </div>

//               <div className="max-h-64 overflow-auto">
//                 {items.map((it) => (
//                   <button
//                     key={it.id}
//                     type="button"
//                     className="w-full rounded-lg px-2 py-2 text-left hover:bg-slate-50"
//                     onClick={() => {
//                       onChange([...value, it]);
//                       setQ("");
//                     }}>
//                     <div className="text-sm font-medium text-slate-900">
//                       {it.label}
//                     </div>
//                     {it.meta ? (
//                       <div className="text-xs text-slate-600">{it.meta}</div>
//                     ) : null}
//                   </button>
//                 ))}
//               </div>
//             </CardBody>
//           </Card>
//         </div>
//       ) : null}
//     </div>
//   );
// }
