"use client";
import { useRef, useState } from "react";
import toast from "react-hot-toast";

const MAX_FILE_SIZE_MB = 20;

const FileUploader = ({
  onUpload,
  onUploadSuccess,
  acceptedTypes = "*",
  maxSizeMB = MAX_FILE_SIZE_MB,
  multiple = true,
  uploadLabel = "Minden feltöltése",
  clearLabel = "Törlés",
}) => {
  const fileInputRef = useRef(null);
  const [files, setFiles] = useState([]);
  const [uploading, setUploading] = useState(false);
  const [progress, setProgress] = useState(0);

  const validateFiles = (fileList) => {
    const valid = [];
    const allowedTypes =
      acceptedTypes === "*"
        ? null
        : acceptedTypes.split(",").map((t) => t.trim());

    for (let file of fileList) {
      const sizeMB = file.size / 1024 / 1024;
      if (sizeMB > maxSizeMB) {
        toast.error(`${file.name} exceeds ${maxSizeMB}MB`);
        continue;
      }

      if (
        allowedTypes &&
        !allowedTypes.some((type) => {
          if (type.endsWith("/*")) {
            // e.g., "image/*" matches "image/jpeg"
            return file.type.startsWith(type.split("/")[0] + "/");
          }
          return file.type === type;
        })
      ) {
        toast.error(`${file.name} type not allowed`);
        continue;
      }

      valid.push(file);
    }

    return valid;
  };

  const handleFileChange = (e) => {
    const selected = Array.from(e.target.files);
    setFiles((prev) => [...prev, ...validateFiles(selected)]);
  };

  const handleDrop = (e) => {
    e.preventDefault();
    const dropped = Array.from(e.dataTransfer.files);
    setFiles((prev) => [...prev, ...validateFiles(dropped)]);
  };

  const removeFile = (idx) => {
    setFiles((prev) => prev.filter((_, i) => i !== idx));
  };

  const handleUpload = async () => {
    if (!files.length) return toast.error("No valid files to upload");
    if (!onUpload) return toast.error("Upload handler missing");

    const formData = new FormData();
    files.forEach((file) => formData.append("file", file));

    setUploading(true);
    setProgress(0);

    try {
      await onUpload(formData, (e) => {
        const percent = Math.round((e.loaded * 100) / e.total);
        setProgress(percent);
      });
      setFiles([]);
      setProgress(0);
      onUploadSuccess?.();
    } catch (err) {
      toast.error("Sikertelen feltöltés");
    } finally {
      setUploading(false);
    }
  };

  return (
    <div className="border border-gray-300 rounded p-4">
      <div
        onDrop={handleDrop}
        onDragOver={(e) => e.preventDefault()}
        className="border-2 border-dashed border-gray-400 rounded p-6 text-center">
        <p className="mb-4">Húzza ide, vagy kattintson a kiválasztáshoz</p>
        <button
          type="button"
          className="bg-blue-600 text-white px-4 py-2 rounded"
          onClick={() => fileInputRef.current.click()}>
          Fájlok tallózása
        </button>
        <input
          ref={fileInputRef}
          type="file"
          multiple={multiple}
          accept={acceptedTypes}
          onChange={handleFileChange}
          className="hidden"
        />
      </div>

      {files.length > 0 && (
        <>
          <div className="mt-4 space-y-2">
            <h4 className="font-semibold">Kiválasztott fájlok:</h4>
            <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
              {files.map((file, idx) => {
                const size = (file.size / 1024 / 1024).toFixed(2);
                return (
                  <div
                    key={idx}
                    className="border p-2 rounded relative bg-white shadow-sm text-sm">
                    <p className="truncate">{file.name}</p>
                    <p className="text-gray-500">{size} MB</p>
                    <button
                      onClick={() => removeFile(idx)}
                      className="absolute top-1 right-1 text-xs text-red-600 hover:text-red-800">
                      ✕
                    </button>
                  </div>
                );
              })}
            </div>
          </div>

          {uploading && (
            <div className="mt-4">
              <p className="text-sm mb-1">
                Feltöltés folyamatban... {progress}%
              </p>
              <div className="w-full h-2 bg-gray-200 rounded">
                <div
                  className="h-2 bg-blue-600 rounded"
                  style={{ width: `${progress}%` }}
                />
              </div>
            </div>
          )}

          <div className="mt-4 flex gap-3">
            <button
              onClick={handleUpload}
              disabled={uploading}
              className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 disabled:opacity-50">
              {uploadLabel}
            </button>
            <button
              onClick={() => setFiles([])}
              disabled={uploading}
              className="bg-gray-400 text-black px-4 py-2 rounded hover:bg-gray-500 disabled:opacity-50">
              {clearLabel}
            </button>
          </div>
        </>
      )}
    </div>
  );
};

export default FileUploader;

// // components/Media/FileUploader.jsx
// "use client";
// import { useState, useRef } from "react";
// import toast from "react-hot-toast";

// const MAX_FILE_SIZE_MB = 10;

// const FileUploader = ({
//   onUpload,
//   onUploadSuccess,
//   acceptedTypes = "image/*",
//   maxSizeMB = MAX_FILE_SIZE_MB,
//   multiple = true,
//   uploadLabel = "Upload All",
//   clearLabel = "Clear",
// }) => {
//   const fileInputRef = useRef(null);
//   const [files, setFiles] = useState([]);
//   const [uploading, setUploading] = useState(false);
//   const [progress, setProgress] = useState(0);
//   const [uploaded, setUploaded] = useState(false);

//   const validateFiles = (fileList) => {
//     const validFiles = [];
//     for (let file of fileList) {
//       const fileSizeMB = file.size / 1024 / 1024;
//       if (fileSizeMB > maxSizeMB) {
//         toast.error(`${file.name} exceeds ${maxSizeMB}MB limit`);
//         continue;
//       }
//       if (acceptedTypes !== "*" && !file.type.match(acceptedTypes)) {
//         toast.error(`${file.name} type not allowed`);
//         continue;
//       }
//       validFiles.push(file);
//     }
//     return validFiles;
//   };

//   const handleFileChange = (e) => {
//     const selectedFiles = Array.from(e.target.files);
//     const valid = validateFiles(selectedFiles);
//     setFiles((prev) => [...prev, ...valid]);
//   };

//   const handleDrop = (e) => {
//     e.preventDefault();
//     const droppedFiles = Array.from(e.dataTransfer.files);
//     const valid = validateFiles(droppedFiles);
//     setFiles((prev) => [...prev, ...valid]);
//   };

//   const handleRemoveFile = (index) => {
//     setFiles((prev) => prev.filter((_, i) => i !== index));
//   };

//   const handleUpload = async () => {
//     if (!files.length) return toast.error("No valid files to upload");
//     if (!onUpload || typeof onUpload !== "function")
//       return toast.error("Upload handler missing");

//     const formData = new FormData();
//     files.forEach((file) => formData.append("file", file)); // ✅ Correct key

//     setUploading(true);
//     setUploaded(false);
//     setProgress(0);

//     try {
//       await onUpload(formData, (event) => {
//         const percent = Math.round((event.loaded * 100) / event.total);
//         setProgress(percent);
//       });

//       toast.success("Upload successful ✅");
//       setUploaded(true);
//       setFiles([]);
//       setProgress(0);
//       onUploadSuccess?.(); // 🔁 optional gallery reload
//     } catch (err) {
//       console.error("❌ Upload error:", err);
//       toast.error("Upload failed");
//     } finally {
//       setUploading(false);
//     }
//   };

//   const renderPreview = (file, idx) => {
//     const isImage = file.type.startsWith("image/");
//     const sizeMB = (file.size / 1024 / 1024).toFixed(2);

//     return (
//       <div
//         key={idx}
//         className="relative flex flex-col items-center border p-2 rounded shadow-sm w-28">
//         {isImage ? (
//           <img
//             src={URL.createObjectURL(file)}
//             alt={file.name}
//             className="w-20 h-20 object-cover rounded"
//           />
//         ) : (
//           <div className="w-20 h-20 flex items-center justify-center text-xs text-gray-500 border rounded bg-gray-100">
//             {file.name}
//           </div>
//         )}
//         <div className="text-xs mt-1 text-center truncate w-full">
//           {file.name}
//         </div>
//         <div className="text-xs text-gray-500">{sizeMB} MB</div>
//         <button
//           onClick={() => handleRemoveFile(idx)}
//           className="absolute top-0 right-0 bg-red-600 text-white text-xs px-1 rounded-bl"
//           title="Remove">
//           ✕
//         </button>
//       </div>
//     );
//   };

//   return (
//     <div className="p-4 border rounded shadow-sm">
//       <div
//         onDrop={handleDrop}
//         onDragOver={(e) => e.preventDefault()}
//         className="border-2 border-dashed border-gray-400 rounded p-6 text-center">
//         <p className="mb-4">Drag and drop files here</p>
//         <button
//           type="button"
//           className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
//           onClick={() => fileInputRef.current?.click()}>
//           Browse Files
//         </button>
//         <input
//           ref={fileInputRef}
//           type="file"
//           accept={acceptedTypes}
//           multiple={multiple}
//           onChange={handleFileChange}
//           className="hidden"
//         />
//       </div>

//       {files.length > 0 && (
//         <div className="mt-4">
//           <h4 className="font-semibold mb-2">Files to Upload:</h4>
//           <div className="flex flex-wrap gap-3 mb-2">
//             {files.map((file, idx) => renderPreview(file, idx))}
//           </div>

//           {uploading && (
//             <div className="text-sm text-blue-600 mb-2">
//               Uploading... {progress}%
//               <div className="w-full bg-gray-200 h-2 rounded mt-1">
//                 <div
//                   className="h-2 bg-blue-600 rounded"
//                   style={{ width: `${progress}%` }}
//                 />
//               </div>
//             </div>
//           )}

//           {uploaded && !uploading && (
//             <div className="text-green-600 text-sm mt-1">
//               ✔ Files uploaded successfully
//             </div>
//           )}

//           <div className="mt-4 flex gap-3">
//             <button
//               onClick={handleUpload}
//               disabled={uploading}
//               className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 disabled:opacity-50">
//               {uploadLabel}
//             </button>
//             <button
//               onClick={() => setFiles([])}
//               disabled={uploading}
//               className="bg-gray-400 text-black px-4 py-2 rounded hover:bg-gray-500 disabled:opacity-50">
//               {clearLabel}
//             </button>
//           </div>
//         </div>
//       )}
//     </div>
//   );
// };

// export default FileUploader;
