"use client";

import { useEffect, useMemo, useState, useCallback } from "react";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";
import Select from "react-select";

export default function OrgNodeForm({ mode, type, context, onClose, onSaved }) {
  // mode: 'create' | 'edit'
  // type: 'site' | 'department' | 'location'
  // context: for create -> { site_id? , department_id? } ; for edit -> full node (id, name, site_id?, department_id?)
  const [saving, setSaving] = useState(false);

  const [name, setName] = useState(context?.name || "");
  const [siteId, setSiteId] = useState(context?.site_id || null);
  const [departmentId, setDepartmentId] = useState(
    context?.department_id || null
  );

  const [sites, setSites] = useState([]);
  const [departments, setDepartments] = useState([]);

  const isCreate = mode === "create";

  // Close on Esc
  const handleKeyDown = useCallback(
    (e) => {
      if (e.key === "Escape") onClose?.();
    },
    [onClose]
  );

  useEffect(() => {
    document.addEventListener("keydown", handleKeyDown);
    return () => document.removeEventListener("keydown", handleKeyDown);
  }, [handleKeyDown]);

  useEffect(() => {
    const loadSites = async () => {
      try {
        const res = await ApiCall.orgApi.listSites();
        setSites(res.data || []);
      } catch {
        toast.error("Telephelyek betöltése sikertelen");
      }
    };
    loadSites();
  }, []);

  useEffect(() => {
    const loadDepartments = async () => {
      if (!siteId) {
        setDepartments([]);
        setDepartmentId(null);
        return;
      }
      try {
        const res = await ApiCall.orgApi.listDepartmentsBySite(siteId);
        setDepartments(res.data || []);
        if (departmentId && !res.data?.some((d) => d.id === departmentId)) {
          setDepartmentId(null);
        }
      } catch {
        toast.error("Szervezeti egységek betöltése sikertelen");
      }
    };
    loadDepartments();
  }, [siteId]);

  const siteOptions = useMemo(
    () => sites.map((s) => ({ value: s.id, label: s.name })),
    [sites]
  );
  const departmentOptions = useMemo(
    () => departments.map((d) => ({ value: d.id, label: d.name })),
    [departments]
  );

  const title = isCreate
    ? type === "site"
      ? "Telephely létrehozása"
      : type === "department"
      ? "Szervezeti egység létrehozása"
      : "Hely létrehozása"
    : type === "site"
    ? "Telephely szerkesztése"
    : type === "department"
    ? "Szervezeti egység szerkesztése"
    : "Hely szerkesztése";

  const canSave = () => {
    if (!name.trim()) return false;
    if (type === "department" && !siteId) return false;
    if (type === "location" && !departmentId) return false;
    return true;
  };

  const onSubmit = async (e) => {
    e.preventDefault();
    if (!canSave()) return;

    setSaving(true);
    try {
      if (isCreate) {
        if (type === "site")
          await ApiCall.orgApi.createSite({ name: name.trim() });
        if (type === "department")
          await ApiCall.orgApi.createDepartment({
            name: name.trim(),
            site_id: siteId,
          });
        if (type === "location")
          await ApiCall.orgApi.createLocation({
            name: name.trim(),
            department_id: departmentId,
          });
      } else {
        const id = context.id;
        if (type === "site")
          await ApiCall.orgApi.updateSite(id, { name: name.trim() });
        if (type === "department")
          await ApiCall.orgApi.updateDepartment(id, {
            name: name.trim(),
            site_id: siteId || undefined,
          });
        if (type === "location")
          await ApiCall.orgApi.updateLocation(id, {
            name: name.trim(),
            department_id: departmentId || undefined,
          });
      }

      toast.success("Mentve");
      onSaved?.();
    } catch (e) {
      toast.error(e?.response?.data?.error || "Mentés sikertelen");
    } finally {
      setSaving(false);
    }
  };

  // Prefill selects in edit mode
  useEffect(() => {
    if (mode === "edit" && type !== "site") {
      if (type === "department") setSiteId(context.site_id || null);
      if (type === "location") {
        setSiteId(context.site_id || null);
        setDepartmentId(context.department_id || null);
      }
    }
  }, [mode, type]);

  return (
    <div className="fixed inset-0 z-50 flex" role="dialog" aria-modal="true">
      {/* Backdrop */}
      <div className="absolute inset-0 bg-black/30 z-40" onClick={onClose} />
      {/* Panel */}
      <div
        className="relative z-50 ml-auto h-full w-full max-w-md bg-white shadow-xl p-4 flex flex-col"
        onClick={(e) => e.stopPropagation()}>
        <div className="flex items-center justify-between mb-4">
          <h2 className="text-lg font-semibold">{title}</h2>
          <button
            onClick={onClose}
            className="text-sm text-gray-500 hover:text-gray-700">
            Bezár
          </button>
        </div>

        <form onSubmit={onSubmit} className="space-y-3">
          <div>
            <label className="block text-sm font-medium mb-1">Név</label>
            <input
              type="text"
              className="w-full rounded border px-3 py-2"
              value={name}
              onChange={(e) => setName(e.target.value)}
              placeholder="Név"
              autoFocus
            />
          </div>

          {type !== "site" && (
            <div>
              <label className="block text-sm font-medium mb-1">
                Telephely
              </label>
              <Select
                options={siteOptions}
                value={siteOptions.find((o) => o.value === siteId) || null}
                onChange={(opt) => setSiteId(opt?.value || null)}
                isClearable
              />
            </div>
          )}

          {type === "location" && (
            <div>
              <label className="block text-sm font-medium mb-1">
                Szervezeti egység
              </label>
              <Select
                options={departmentOptions}
                value={
                  departmentOptions.find((o) => o.value === departmentId) ||
                  null
                }
                onChange={(opt) => setDepartmentId(opt?.value || null)}
                isClearable
                isDisabled={!siteId}
              />
            </div>
          )}

          <div className="pt-2 flex items-center justify-end gap-2">
            <button
              type="button"
              onClick={onClose}
              className="rounded border px-3 py-2 text-sm">
              Mégse
            </button>
            <button
              type="submit"
              disabled={!canSave() || saving}
              className="rounded bg-blue-600 px-3 py-2 text-sm text-white disabled:opacity-50">
              {saving ? "Mentés…" : "Mentés"}
            </button>
          </div>
        </form>

        <div className="mt-3 text-xs text-gray-500">
          Tipp: Szerkesztéskor telephely (department) vagy szervezeti egység
          (location) módosítása “áthelyezést” eredményez.
        </div>
      </div>
    </div>
  );
}
