// components/org-structure/OrgTree.tsx
"use client";

import { useMemo, useState } from "react";
import {
  ChevronDown,
  ChevronRight,
  Pencil,
  Trash2,
  Plus,
  X,
} from "lucide-react";
import {
  DndContext,
  MouseSensor,
  TouchSensor,
  useSensor,
  useSensors,
  DragEndEvent,
  DragStartEvent,
  DragOverEvent,
} from "@dnd-kit/core";
import {
  SortableContext,
  verticalListSortingStrategy,
  useSortable,
  arrayMove,
} from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";

type NodeType = "site" | "department" | "location";

type Props = {
  data: any[];
  loading: boolean;
  onCreate: (type: NodeType, ctx?: any) => void;
  onEdit: (type: NodeType, node: any) => void;
  onDelete: (type: NodeType, node: any) => void;
  // NEW:
  onMoveDepartment: (
    departmentId: string,
    toSiteId: string,
    beforeId?: string
  ) => void;
  onMoveLocation: (
    locationId: string,
    toDepartmentId: string,
    beforeId?: string
  ) => void;
  onReorderDepartments: (siteId: string, orderedIds: string[]) => void;
  onReorderLocations: (departmentId: string, orderedIds: string[]) => void;
};

function Row({
  indent = 0,
  title,
  right,
  children,
  defaultOpen = true,
  leading,
}: {
  indent?: number;
  title: React.ReactNode;
  right?: React.ReactNode;
  children?: React.ReactNode;
  defaultOpen?: boolean;
  /** Element rendered before the chevron (e.g., a checkbox) */
  leading?: React.ReactNode;
}) {
  const [open, setOpen] = useState(defaultOpen);
  const hasChildren = !!children;

  return (
    <div className="w-full">
      <div className="flex items-center justify-between py-2 border-b">
        <div className="flex items-center gap-2">
          <div style={{ width: indent * 16 }} />
          {hasChildren ? (
            <button
              onClick={() => setOpen((o) => !o)}
              className="p-1 rounded hover:bg-gray-100"
              aria-label="toggle">
              {open ? (
                <ChevronDown className="w-4 h-4" />
              ) : (
                <ChevronRight className="w-4 h-4" />
              )}
            </button>
          ) : (
            <div style={{ width: 24 }} />
          )}
          {leading}
          <span className="font-medium">{title}</span>
        </div>
        <div className="flex items-center gap-2">{right}</div>
      </div>

      {hasChildren && open && <div>{children}</div>}
    </div>
  );
}

/** Sortable draggable item wrapper */
function DraggableRow({
  id,
  children,
}: {
  id: string; // e.g. dep:uuid or loc:uuid
  children: React.ReactNode;
}) {
  const {
    attributes,
    listeners,
    setNodeRef,
    transform,
    transition,
    isDragging,
  } = useSortable({
    id,
  });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
    opacity: isDragging ? 0.6 : undefined,
    cursor: "grab",
  } as React.CSSProperties;

  return (
    <div ref={setNodeRef} style={style} {...attributes} {...listeners}>
      {children}
    </div>
  );
}

const depId = (id: string) => `dep:${id}`;
const locId = (id: string) => `loc:${id}`;
const siteIdKey = (id: string) => `site:${id}`;
const depListId = (siteId: string) => `dep-list:${siteId}`;
const locListId = (deptId: string) => `loc-list:${deptId}`;
const orphanListId = (siteId: string) => `orphan-list:${siteId}`;

type SelEntry = { type: NodeType; node: any };
type SelMap = Map<string, SelEntry>;

export default function OrgTree({
  data,
  loading,
  onCreate,
  onEdit,
  onDelete,
  onMoveDepartment,
  onMoveLocation,
  onReorderDepartments,
  onReorderLocations,
}: Props) {
  const sensors = useSensors(
    useSensor(MouseSensor, { activationConstraint: { distance: 6 } }),
    useSensor(TouchSensor, {
      activationConstraint: { delay: 120, tolerance: 8 },
    })
  );

  /** ---------------- Selection state ---------------- */
  const [selected, setSelected] = useState<SelMap>(new Map());
  const selectedCount = selected.size;

  const isChecked = (key: string) => selected.has(key);

  const toggleSelect = (
    key: string,
    type: NodeType,
    node: any,
    force?: boolean
  ) => {
    setSelected((prev) => {
      const next = new Map(prev);
      const has = next.has(key);
      const shouldAdd = typeof force === "boolean" ? force : !has;
      if (shouldAdd) next.set(key, { type, node });
      else next.delete(key);
      return next;
    });
  };

  const clearSelection = () => setSelected(new Map());

  const firstSelected = (): SelEntry | null => {
    for (const [, v] of selected.entries()) return v;
    return null;
  };

  const bulkDelete = () => {
    // naive sequential; your onDelete can show a confirm per node if needed
    for (const [, { type, node }] of selected.entries()) {
      onDelete(type, node);
    }
    clearSelection();
  };

  const singleEdit = () => {
    const entry = firstSelected();
    if (!entry) return;
    onEdit(entry.type, entry.node);
    // keep selection as-is (or clear if you prefer)
  };

  // Build quick lookups to resolve parents on drop
  const siteByDep = useMemo(() => {
    const m = new Map<string, string>(); // depId -> siteId
    data.forEach((s) =>
      (s.departments || []).forEach((d: any) =>
        m.set(String(d.id), String(s.id))
      )
    );
    return m;
  }, [data]);

  const depByLoc = useMemo(() => {
    const m = new Map<string, string>(); // locId -> departmentId
    data.forEach((s) =>
      (s.departments || []).forEach((d: any) =>
        (d.locations || []).forEach((l: any) =>
          m.set(String(l.id), String(d.id))
        )
      )
    );
    return m;
  }, [data]);

  const allDepIdsBySite = useMemo(() => {
    const m = new Map<string, string[]>(); // siteId -> dep ids[]
    data.forEach((s) => {
      m.set(
        String(s.id),
        (s.departments || []).map((d: any) => String(d.id))
      );
    });
    return m;
  }, [data]);

  const allLocIdsByDept = useMemo(() => {
    const m = new Map<string, string[]>(); // deptId -> loc ids[]
    data.forEach((s) =>
      (s.departments || []).forEach((d: any) => {
        m.set(
          String(d.id),
          (d.locations || []).map((l: any) => String(l.id))
        );
      })
    );
    return m;
  }, [data]);

  // DnD handlers
  const handleDragEnd = (e: DragEndEvent) => {
    const active = e.active?.id as string | undefined;
    const over = e.over?.id as string | undefined;
    if (!active || !over) return;

    const [activeType, activeUuid] = active.split(":"); // dep|loc
    const [overType, overKey] = over.split(":"); // dep|loc|dep-list|loc-list|orphan-list|site

    // Reorder within same list
    if (
      activeType === "dep" &&
      (overType === "dep" || overType === "dep-list" || overType === "site")
    ) {
      const fromSiteId = siteByDep.get(activeUuid);
      // if dropped over another dep, use that dep's site
      let toSiteId = fromSiteId || "";
      let beforeId: string | undefined = undefined;

      if (overType === "dep") {
        const targetDepId = overKey;
        toSiteId = siteByDep.get(targetDepId) || toSiteId;
        beforeId = targetDepId;
      } else if (overType === "dep-list" || overType === "site") {
        toSiteId = overKey;
      }

      // If parent changes OR position changes, call server
      const current = allDepIdsBySite.get(toSiteId) || [];
      const isSameParent = fromSiteId === toSiteId;

      if (isSameParent && overType === "dep") {
        // local reorder sequence
        const ids = current.slice();
        const fromIdx = ids.indexOf(activeUuid);
        const toIdx = ids.indexOf(overKey);
        if (fromIdx > -1 && toIdx > -1 && fromIdx !== toIdx) {
          const next = arrayMove(ids, fromIdx, toIdx);
          onReorderDepartments(toSiteId, next);
          return;
        }
      }

      // move (also works as append when beforeId is undefined)
      onMoveDepartment(activeUuid, toSiteId, beforeId);
      return;
    }

    if (
      activeType === "loc" &&
      (overType === "loc" || overType === "loc-list" || overType === "dep")
    ) {
      const fromDeptId = depByLoc.get(activeUuid);
      let toDeptId = fromDeptId || "";
      let beforeId: string | undefined = undefined;

      if (overType === "loc") {
        const targetLocId = overKey;
        toDeptId = depByLoc.get(targetLocId) || toDeptId;
        beforeId = targetLocId;
      } else if (overType === "loc-list" || overType === "dep") {
        toDeptId = overKey; // container carries department id
      }

      const current = allLocIdsByDept.get(toDeptId) || [];
      const isSameParent = fromDeptId === toDeptId;

      if (isSameParent && overType === "loc") {
        const ids = current.slice();
        const fromIdx = ids.indexOf(activeUuid);
        const toIdx = ids.indexOf(overKey);
        if (fromIdx > -1 && toIdx > -1 && fromIdx !== toIdx) {
          const next = arrayMove(ids, fromIdx, toIdx);
          onReorderLocations(toDeptId, next);
          return;
        }
      }
      console.log(
        "Moving location: ",
        activeUuid,
        "toDeptId:",
        toDeptId,
        "beforeId:",
        beforeId
      );

      if (!toDeptId) {
        console.error(
          "Cannot move orphaned location without a target department."
        );
        return;
      }
      onMoveLocation(activeUuid, toDeptId, beforeId);
      return;
    }
  };

  const handleDragStart = (_e: DragStartEvent) => {};
  const handleDragOver = (_e: DragOverEvent) => {};

  if (loading)
    return <div className="p-4 text-sm text-gray-500">Betöltés…</div>;
  if (!data?.length)
    return <div className="p-4 text-sm text-gray-500">Nincs adat</div>;

  return (
    <DndContext
      sensors={sensors}
      onDragEnd={handleDragEnd}
      onDragStart={handleDragStart}
      onDragOver={handleDragOver}>
      {/* Sticky multi-select action bar */}
      {selectedCount > 0 && (
        <div className="sticky top-0 z-10 mb-2 bg-white border rounded p-2 shadow-sm flex items-center justify-between">
          <div className="text-sm">
            <span className="font-medium">{selectedCount}</span>{" "}
            {selectedCount === 1 ? "elem kiválasztva" : "elem kiválasztva"}
          </div>
          <div className="flex items-center gap-2">
            {selectedCount === 1 && (
              <button
                className="inline-flex items-center gap-1 rounded bg-blue-600 px-2 py-1 text-white hover:bg-blue-700 text-xs"
                onClick={singleEdit}>
                <Pencil className="w-3 h-3" /> Szerkesztés
              </button>
            )}
            <button
              className="inline-flex items-center gap-1 rounded bg-red-600 px-2 py-1 text-white hover:bg-red-700 text-xs"
              onClick={bulkDelete}>
              <Trash2 className="w-3 h-3" /> Törlés
            </button>
            <button
              className="inline-flex items-center gap-1 rounded border px-2 py-1 text-xs"
              onClick={clearSelection}
              title="Kijelölés törlése">
              <X className="w-3 h-3" /> Törlés (kijelölés)
            </button>
          </div>
        </div>
      )}

      <div className="divide-y">
        {data.map((site) => {
          const depIds = (site.departments || []).map((d: any) => String(d.id));
          const siteKey = siteIdKey(String(site.id));

          const siteCheckbox = (
            <input
              type="checkbox"
              className="rounded border-gray-300"
              checked={isChecked(siteKey)}
              onClick={(e) => e.stopPropagation()}
              onChange={(e) =>
                toggleSelect(siteKey, "site", site, e.currentTarget.checked)
              }
            />
          );

          return (
            <Row
              key={site.id}
              leading={<div className="mr-1">{siteCheckbox}</div>}
              title={
                <>
                  {site.name}
                  {typeof site.assetCount === "number" && (
                    <span className="ml-2 text-xs text-gray-500">
                      ({site.assetCount} eszköz)
                    </span>
                  )}
                </>
              }
              right={
                <>
                  <button
                    className="inline-flex items-center gap-1 rounded bg-green-600 px-2 py-1 text-white hover:bg-green-700 text-xs"
                    onClick={() =>
                      onCreate("department", { site_id: site.id })
                    }>
                    <Plus className="w-3 h-3" /> Új szervezeti egység
                  </button>
                  <button
                    className="rounded bg-blue-600 px-2 py-1 text-white hover:bg-blue-700 text-xs"
                    onClick={() => onEdit("site", site)}>
                    <Pencil className="w-3 h-3" />
                  </button>
                  <button
                    className="rounded bg-red-600 px-2 py-1 text-white hover:bg-red-700 text-xs"
                    onClick={() => onDelete("site", site)}>
                    <Trash2 className="w-3 h-3" />
                  </button>
                </>
              }>
              {/* Departments list is a sortable context per site */}
              <SortableContext
                items={depIds.map(depId)}
                strategy={verticalListSortingStrategy}
                id={depListId(site.id)}>
                {(site.departments || []).map((dept: any) => {
                  const locIds = (dept.locations || []).map((l: any) =>
                    String(l.id)
                  );
                  const deptKey = depId(String(dept.id));

                  const deptCheckbox = (
                    <input
                      type="checkbox"
                      className="rounded border-gray-300"
                      checked={isChecked(deptKey)}
                      onClick={(e) => e.stopPropagation()}
                      onChange={(e) =>
                        toggleSelect(
                          deptKey,
                          "department",
                          dept,
                          e.currentTarget.checked
                        )
                      }
                    />
                  );

                  return (
                    <DraggableRow key={deptKey} id={deptKey}>
                      <Row
                        indent={1}
                        leading={<div className="mr-1">{deptCheckbox}</div>}
                        title={
                          <>
                            {dept.name}
                            {typeof dept.assetCount === "number" && (
                              <span className="ml-2 text-xs text-gray-500">
                                ({dept.assetCount} eszköz)
                              </span>
                            )}
                          </>
                        }
                        right={
                          <>
                            <button
                              className="inline-flex items-center gap-1 rounded bg-green-600 px-2 py-1 text-white hover:bg-green-700 text-xs"
                              onClick={() =>
                                onCreate("location", {
                                  site_id: site.id,
                                  department_id: dept.id,
                                })
                              }>
                              <Plus className="w-3 h-3" /> Új hely
                            </button>
                            <button
                              className="rounded bg-blue-600 px-2 py-1 text-white hover:bg-blue-700 text-xs"
                              onClick={() => onEdit("department", dept)}>
                              <Pencil className="w-3 h-3" />
                            </button>
                            <button
                              className="rounded bg-red-600 px-2 py-1 text-white hover:bg-red-700 text-xs"
                              onClick={() => onDelete("department", dept)}>
                              <Trash2 className="w-3 h-3" />
                            </button>
                          </>
                        }>
                        {/* Locations list per department */}
                        <SortableContext
                          items={locIds.map(locId)}
                          strategy={verticalListSortingStrategy}
                          id={locListId(dept.id)}>
                          {(dept.locations || []).map((loc: any) => {
                            const locKey = locId(String(loc.id));
                            const locCheckbox = (
                              <input
                                type="checkbox"
                                className="rounded border-gray-300"
                                checked={isChecked(locKey)}
                                onClick={(e) => e.stopPropagation()}
                                onChange={(e) =>
                                  toggleSelect(
                                    locKey,
                                    "location",
                                    loc,
                                    e.currentTarget.checked
                                  )
                                }
                              />
                            );

                            return (
                              <DraggableRow key={locKey} id={locKey}>
                                <Row
                                  indent={2}
                                  leading={
                                    <div className="mr-1">{locCheckbox}</div>
                                  }
                                  title={
                                    <>
                                      {loc.name}
                                      {typeof loc.assetCount === "number" && (
                                        <span className="ml-2 text-xs text-gray-500">
                                          ({loc.assetCount} eszköz)
                                        </span>
                                      )}
                                    </>
                                  }
                                  right={
                                    <>
                                      <button
                                        className="rounded bg-blue-600 px-2 py-1 text-white hover:bg-blue-700 text-xs"
                                        onClick={() => onEdit("location", loc)}>
                                        <Pencil className="w-3 h-3" />
                                      </button>
                                      <button
                                        className="rounded bg-red-600 px-2 py-1 text-white hover:bg-red-700 text-xs"
                                        onClick={() =>
                                          onDelete("location", loc)
                                        }>
                                        <Trash2 className="w-3 h-3" />
                                      </button>
                                    </>
                                  }
                                />
                              </DraggableRow>
                            );
                          })}
                        </SortableContext>
                      </Row>
                    </DraggableRow>
                  );
                })}
              </SortableContext>

              {/* Orphans (locations without department) under site */}
              {!!site.orphans?.length && (
                <div className="mt-2 ml-8 border-l pl-4">
                  <div className="text-xs uppercase text-gray-500 mb-1">
                    Árvák
                  </div>
                  <SortableContext
                    items={site.orphans.map((o: any) => locId(String(o.id)))}
                    strategy={verticalListSortingStrategy}
                    id={orphanListId(site.id)}>
                    {site.orphans.map((o: any) => {
                      const oKey = locId(String(o.id));
                      const orphanCheckbox = (
                        <input
                          type="checkbox"
                          className="rounded border-gray-300"
                          checked={isChecked(oKey)}
                          onClick={(e) => e.stopPropagation()}
                          onChange={(e) =>
                            toggleSelect(
                              oKey,
                              "location",
                              o,
                              e.currentTarget.checked
                            )
                          }
                        />
                      );

                      return (
                        <DraggableRow key={oKey} id={oKey}>
                          <Row
                            indent={2}
                            leading={
                              <div className="mr-1">{orphanCheckbox}</div>
                            }
                            title={
                              <>
                                {o.name}
                                {typeof o.assetCount === "number" && (
                                  <span className="ml-2 text-xs text-gray-500">
                                    ({o.assetCount} eszköz)
                                  </span>
                                )}
                              </>
                            }
                            right={
                              <>
                                <button
                                  className="rounded bg-blue-600 px-2 py-1 text-white hover:bg-blue-700 text-xs"
                                  onClick={() => onEdit("location", o)}>
                                  <Pencil className="w-3 h-3" />
                                </button>
                                <button
                                  className="rounded bg-red-600 px-2 py-1 text-white hover:bg-red-700 text-xs"
                                  onClick={() => onDelete("location", o)}>
                                  <Trash2 className="w-3 h-3" />
                                </button>
                              </>
                            }
                          />
                        </DraggableRow>
                      );
                    })}
                  </SortableContext>
                </div>
              )}
            </Row>
          );
        })}
      </div>
    </DndContext>
  );
}

// // components/org-structure/OrgTree.tsx
// "use client";

// import { useMemo, useState } from "react";
// import { ChevronDown, ChevronRight, Pencil, Trash2, Plus } from "lucide-react";
// import {
//   DndContext,
//   MouseSensor,
//   TouchSensor,
//   useSensor,
//   useSensors,
//   DragEndEvent,
//   DragStartEvent,
//   DragOverEvent,
// } from "@dnd-kit/core";
// import {
//   SortableContext,
//   verticalListSortingStrategy,
//   useSortable,
//   arrayMove,
// } from "@dnd-kit/sortable";
// import { CSS } from "@dnd-kit/utilities";

// type NodeType = "site" | "department" | "location";

// type Props = {
//   data: any[];
//   loading: boolean;
//   onCreate: (type: NodeType, ctx?: any) => void;
//   onEdit: (type: NodeType, node: any) => void;
//   onDelete: (type: NodeType, node: any) => void;
//   // NEW:
//   onMoveDepartment: (
//     departmentId: string,
//     toSiteId: string,
//     beforeId?: string
//   ) => void;
//   onMoveLocation: (
//     locationId: string,
//     toDepartmentId: string,
//     beforeId?: string
//   ) => void;
//   onReorderDepartments: (siteId: string, orderedIds: string[]) => void;
//   onReorderLocations: (departmentId: string, orderedIds: string[]) => void;
// };

// function Row({
//   indent = 0,
//   title,
//   right,
//   children,
//   defaultOpen = true,
// }: {
//   indent?: number;
//   title: React.ReactNode;
//   right?: React.ReactNode;
//   children?: React.ReactNode;
//   defaultOpen?: boolean;
// }) {
//   const [open, setOpen] = useState(defaultOpen);
//   const hasChildren = !!children;

//   return (
//     <div className="w-full">
//       <div className="flex items-center justify-between py-2 border-b">
//         <div className="flex items-center gap-2">
//           <div style={{ width: indent * 16 }} />
//           {hasChildren ? (
//             <button
//               onClick={() => setOpen((o) => !o)}
//               className="p-1 rounded hover:bg-gray-100"
//               aria-label="toggle">
//               {open ? (
//                 <ChevronDown className="w-4 h-4" />
//               ) : (
//                 <ChevronRight className="w-4 h-4" />
//               )}
//             </button>
//           ) : (
//             <div style={{ width: 24 }} />
//           )}
//           <span className="font-medium">{title}</span>
//         </div>
//         <div className="flex items-center gap-2">{right}</div>
//       </div>

//       {hasChildren && open && <div>{children}</div>}
//     </div>
//   );
// }

// /** Sortable draggable item wrapper */
// function DraggableRow({
//   id,
//   children,
// }: {
//   id: string; // e.g. dep:uuid or loc:uuid
//   children: React.ReactNode;
// }) {
//   const {
//     attributes,
//     listeners,
//     setNodeRef,
//     transform,
//     transition,
//     isDragging,
//   } = useSortable({
//     id,
//   });

//   const style = {
//     transform: CSS.Transform.toString(transform),
//     transition,
//     opacity: isDragging ? 0.6 : undefined,
//     cursor: "grab",
//   } as React.CSSProperties;

//   return (
//     <div ref={setNodeRef} style={style} {...attributes} {...listeners}>
//       {children}
//     </div>
//   );
// }

// const depId = (id: string) => `dep:${id}`;
// const locId = (id: string) => `loc:${id}`;
// const siteIdKey = (id: string) => `site:${id}`;
// const depListId = (siteId: string) => `dep-list:${siteId}`;
// const locListId = (deptId: string) => `loc-list:${deptId}`;
// const orphanListId = (siteId: string) => `orphan-list:${siteId}`;

// export default function OrgTree({
//   data,
//   loading,
//   onCreate,
//   onEdit,
//   onDelete,
//   onMoveDepartment,
//   onMoveLocation,
//   onReorderDepartments,
//   onReorderLocations,
// }: Props) {
//   const sensors = useSensors(
//     useSensor(MouseSensor, { activationConstraint: { distance: 6 } }),
//     useSensor(TouchSensor, {
//       activationConstraint: { delay: 120, tolerance: 8 },
//     })
//   );

//   // Build quick lookups to resolve parents on drop
//   const siteByDep = useMemo(() => {
//     const m = new Map<string, string>(); // depId -> siteId
//     data.forEach((s) =>
//       (s.departments || []).forEach((d: any) =>
//         m.set(String(d.id), String(s.id))
//       )
//     );
//     return m;
//   }, [data]);

//   const depByLoc = useMemo(() => {
//     const m = new Map<string, string>(); // locId -> departmentId
//     data.forEach((s) =>
//       (s.departments || []).forEach((d: any) =>
//         (d.locations || []).forEach((l: any) =>
//           m.set(String(l.id), String(d.id))
//         )
//       )
//     );
//     return m;
//   }, [data]);

//   const allDepIdsBySite = useMemo(() => {
//     const m = new Map<string, string[]>(); // siteId -> dep ids[]
//     data.forEach((s) => {
//       m.set(
//         String(s.id),
//         (s.departments || []).map((d: any) => String(d.id))
//       );
//     });
//     return m;
//   }, [data]);

//   const allLocIdsByDept = useMemo(() => {
//     const m = new Map<string, string[]>(); // deptId -> loc ids[]
//     data.forEach((s) =>
//       (s.departments || []).forEach((d: any) => {
//         m.set(
//           String(d.id),
//           (d.locations || []).map((l: any) => String(l.id))
//         );
//       })
//     );
//     return m;
//   }, [data]);

//   // DnD handlers
//   const handleDragEnd = (e: DragEndEvent) => {
//     const active = e.active?.id as string | undefined;
//     const over = e.over?.id as string | undefined;
//     if (!active || !over) return;

//     const [activeType, activeUuid] = active.split(":"); // dep|loc
//     const [overType, overKey] = over.split(":"); // dep|loc|dep-list|loc-list|orphan-list|site

//     // Reorder within same list
//     if (
//       activeType === "dep" &&
//       (overType === "dep" || overType === "dep-list" || overType === "site")
//     ) {
//       const fromSiteId = siteByDep.get(activeUuid);
//       // if dropped over another dep, use that dep's site
//       let toSiteId = fromSiteId || "";
//       let beforeId: string | undefined = undefined;

//       if (overType === "dep") {
//         const targetDepId = overKey;
//         toSiteId = siteByDep.get(targetDepId) || toSiteId;
//         beforeId = targetDepId;
//       } else if (overType === "dep-list" || overType === "site") {
//         toSiteId = overKey;
//       }

//       // If parent changes OR position changes, call server
//       const current = allDepIdsBySite.get(toSiteId) || [];
//       const isSameParent = fromSiteId === toSiteId;

//       if (isSameParent && overType === "dep") {
//         // local reorder sequence
//         const ids = current.slice();
//         const fromIdx = ids.indexOf(activeUuid);
//         const toIdx = ids.indexOf(overKey);
//         if (fromIdx > -1 && toIdx > -1 && fromIdx !== toIdx) {
//           const next = arrayMove(ids, fromIdx, toIdx);
//           onReorderDepartments(toSiteId, next);
//           return;
//         }
//       }

//       // move (also works as append when beforeId is undefined)
//       onMoveDepartment(activeUuid, toSiteId, beforeId);
//       return;
//     }

//     if (
//       activeType === "loc" &&
//       (overType === "loc" || overType === "loc-list" || overType === "dep")
//     ) {
//       const fromDeptId = depByLoc.get(activeUuid);
//       let toDeptId = fromDeptId || "";
//       let beforeId: string | undefined = undefined;

//       if (overType === "loc") {
//         const targetLocId = overKey;
//         toDeptId = depByLoc.get(targetLocId) || toDeptId;
//         beforeId = targetLocId;
//       } else if (overType === "loc-list" || overType === "dep") {
//         toDeptId = overKey; // container carries department id
//       }

//       const current = allLocIdsByDept.get(toDeptId) || [];
//       const isSameParent = fromDeptId === toDeptId;

//       if (isSameParent && overType === "loc") {
//         const ids = current.slice();
//         const fromIdx = ids.indexOf(activeUuid);
//         const toIdx = ids.indexOf(overKey);
//         if (fromIdx > -1 && toIdx > -1 && fromIdx !== toIdx) {
//           const next = arrayMove(ids, fromIdx, toIdx);
//           onReorderLocations(toDeptId, next);
//           return;
//         }
//       }
//       console.log(
//         "Moving location: ",
//         activeUuid,
//         "toDeptId:",
//         toDeptId,
//         "beforeId:",
//         beforeId
//       );

//       if (!toDeptId) {
//         console.error(
//           "Cannot move orphaned location without a target department."
//         );
//         return;
//       }
//       onMoveLocation(activeUuid, toDeptId, beforeId);
//       return;
//     }
//   };

//   const handleDragStart = (_e: DragStartEvent) => {};
//   const handleDragOver = (_e: DragOverEvent) => {};

//   if (loading)
//     return <div className="p-4 text-sm text-gray-500">Betöltés…</div>;
//   if (!data?.length)
//     return <div className="p-4 text-sm text-gray-500">Nincs adat</div>;

//   return (
//     <DndContext
//       sensors={sensors}
//       onDragEnd={handleDragEnd}
//       onDragStart={handleDragStart}
//       onDragOver={handleDragOver}>
//       <div className="divide-y">
//         {data.map((site) => {
//           const depIds = (site.departments || []).map((d: any) => String(d.id));
//           return (
//             <Row
//               key={site.id}
//               title={
//                 <>
//                   {site.name}
//                   {typeof site.assetCount === "number" && (
//                     <span className="ml-2 text-xs text-gray-500">
//                       ({site.assetCount} eszköz)
//                     </span>
//                   )}
//                 </>
//               }
//               right={
//                 <>
//                   <button
//                     className="inline-flex items-center gap-1 rounded bg-green-600 px-2 py-1 text-white hover:bg-green-700 text-xs"
//                     onClick={() =>
//                       onCreate("department", { site_id: site.id })
//                     }>
//                     <Plus className="w-3 h-3" /> Új szervezeti egység
//                   </button>
//                   <button
//                     className="rounded bg-blue-600 px-2 py-1 text-white hover:bg-blue-700 text-xs"
//                     onClick={() => onEdit("site", site)}>
//                     <Pencil className="w-3 h-3" />
//                   </button>
//                   <button
//                     className="rounded bg-red-600 px-2 py-1 text-white hover:bg-red-700 text-xs"
//                     onClick={() => onDelete("site", site)}>
//                     <Trash2 className="w-3 h-3" />
//                   </button>
//                 </>
//               }>
//               {/* Departments list is a sortable context per site */}
//               <SortableContext
//                 items={depIds.map(depId)}
//                 strategy={verticalListSortingStrategy}
//                 id={depListId(site.id)}>
//                 {(site.departments || []).map((dept: any) => {
//                   const locIds = (dept.locations || []).map((l: any) =>
//                     String(l.id)
//                   );
//                   return (
//                     <DraggableRow key={depId(dept.id)} id={depId(dept.id)}>
//                       <Row
//                         indent={1}
//                         title={
//                           <>
//                             {dept.name}
//                             {typeof dept.assetCount === "number" && (
//                               <span className="ml-2 text-xs text-gray-500">
//                                 ({dept.assetCount} eszköz)
//                               </span>
//                             )}
//                           </>
//                         }
//                         right={
//                           <>
//                             <button
//                               className="inline-flex items-center gap-1 rounded bg-green-600 px-2 py-1 text-white hover:bg-green-700 text-xs"
//                               onClick={() =>
//                                 onCreate("location", {
//                                   site_id: site.id,
//                                   department_id: dept.id,
//                                 })
//                               }>
//                               <Plus className="w-3 h-3" /> Új hely
//                             </button>
//                             <button
//                               className="rounded bg-blue-600 px-2 py-1 text-white hover:bg-blue-700 text-xs"
//                               onClick={() => onEdit("department", dept)}>
//                               <Pencil className="w-3 h-3" />
//                             </button>
//                             <button
//                               className="rounded bg-red-600 px-2 py-1 text-white hover:bg-red-700 text-xs"
//                               onClick={() => onDelete("department", dept)}>
//                               <Trash2 className="w-3 h-3" />
//                             </button>
//                           </>
//                         }>
//                         {/* Locations list per department */}
//                         <SortableContext
//                           items={locIds.map(locId)}
//                           strategy={verticalListSortingStrategy}
//                           id={locListId(dept.id)}>
//                           {(dept.locations || []).map((loc: any) => (
//                             <DraggableRow
//                               key={locId(loc.id)}
//                               id={locId(loc.id)}>
//                               <Row
//                                 indent={2}
//                                 title={
//                                   <>
//                                     {loc.name}
//                                     {typeof loc.assetCount === "number" && (
//                                       <span className="ml-2 text-xs text-gray-500">
//                                         ({loc.assetCount} eszköz)
//                                       </span>
//                                     )}
//                                   </>
//                                 }
//                                 right={
//                                   <>
//                                     <button
//                                       className="rounded bg-blue-600 px-2 py-1 text-white hover:bg-blue-700 text-xs"
//                                       onClick={() => onEdit("location", loc)}>
//                                       <Pencil className="w-3 h-3" />
//                                     </button>
//                                     <button
//                                       className="rounded bg-red-600 px-2 py-1 text-white hover:bg-red-700 text-xs"
//                                       onClick={() => onDelete("location", loc)}>
//                                       <Trash2 className="w-3 h-3" />
//                                     </button>
//                                   </>
//                                 }
//                               />
//                             </DraggableRow>
//                           ))}
//                         </SortableContext>
//                       </Row>
//                     </DraggableRow>
//                   );
//                 })}
//               </SortableContext>

//               {/* Orphans (locations without department) under site */}
//               {!!site.orphans?.length && (
//                 <div className="mt-2 ml-8 border-l pl-4">
//                   <div className="text-xs uppercase text-gray-500 mb-1">
//                     Árvák
//                   </div>
//                   <SortableContext
//                     items={site.orphans.map((o: any) => locId(String(o.id)))}
//                     strategy={verticalListSortingStrategy}
//                     id={orphanListId(site.id)}>
//                     {site.orphans.map((o: any) => (
//                       <DraggableRow key={locId(o.id)} id={locId(o.id)}>
//                         <Row
//                           indent={2}
//                           title={
//                             <>
//                               {o.name}
//                               {typeof o.assetCount === "number" && (
//                                 <span className="ml-2 text-xs text-gray-500">
//                                   ({o.assetCount} eszköz)
//                                 </span>
//                               )}
//                             </>
//                           }
//                           right={
//                             <>
//                               <button
//                                 className="rounded bg-blue-600 px-2 py-1 text-white hover:bg-blue-700 text-xs"
//                                 onClick={() => onEdit("location", o)}>
//                                 <Pencil className="w-3 h-3" />
//                               </button>
//                               <button
//                                 className="rounded bg-red-600 px-2 py-1 text-white hover:bg-red-700 text-xs"
//                                 onClick={() => onDelete("location", o)}>
//                                 <Trash2 className="w-3 h-3" />
//                               </button>
//                             </>
//                           }
//                         />
//                       </DraggableRow>
//                     ))}
//                   </SortableContext>
//                 </div>
//               )}
//             </Row>
//           );
//         })}
//       </div>
//     </DndContext>
//   );
// }
