// File: app/admin/own-company/edit/[id]/page.tsx
"use client";

import { useEffect, useState, useCallback } from "react";
import { useParams, useRouter } from "next/navigation";
import Input from "@/components/shared/input/Input";
import { useForm } from "@/hooks/custom-form-hook";
import Select from "react-select";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import {
  VALIDATOR_REQUIRE,
  VALIDATOR_MAXLENGTH,
  VALIDATOR_MINLENGTH,
  VALIDATOR_EMAIL,
  VALIDATOR_PHONE_HU,
  VALIDATOR_TAXNUMBER_HU,
} from "@/utils/validators";

const ExpandedRow = ({ id, onRowUpdated }) => {
  console.log("ID on expand", id);
  const router = useRouter();
  const [companyOptions, setCompanyOptions] = useState([]);
  const [formState, handleInput, handleSetInputData] = useForm({}, true);
  const [selectedGroup, setSelectedGroup] = useState(null);
  const [companyData, setCompanyData] = useState(null);
  const [shortCodes, setShortCodes] = useState([]);
  const [selectedShortCode, setSelectedShortCode] = useState(null);
  console.log("formState", formState.inputs, formState);
  useEffect(() => {
    const fetchData = async () => {
      try {
        const [companyRes, optionsRes, shortCodesRes] = await Promise.all([
          ApiCall.ownCompanies.getEdit(id),
          ApiCall.ownCompanies.getAllGroupNames(),
          ApiCall.ownCompanies.getAllShortNames(),
        ]);
        console.log(
          "companyRes",
          companyRes?.data,
          "\noptionsRes",
          optionsRes?.data,
          "\nshortCodesRes",
          shortCodesRes?.data
        );
        if (companyRes.status === 200) {
          const data = companyRes.data.company;
          setCompanyData(data);

          if (data.company_group_id) {
            setSelectedGroup({
              value: data.company_group_id,
              label: data.company_group_name || "Kijelölt cég",
            });
          }
          if (data.short_code_id) {
            setSelectedShortCode({
              value: data.short_code_id,
              label: data.short_code || "",
            });
          }
        }

        if (optionsRes.status === 200) {
          setCompanyOptions(optionsRes.data.groups);
        }
        if (shortCodesRes.status === 200) {
          setShortCodes(shortCodesRes.data.shortCodes);
        }
      } catch (err) {
        toast.error("Hiba történt az adatok betöltése közben");
      }
    };

    fetchData();
  }, [id]);

  const handleSubmit = useCallback(async () => {
    const payload = Object.fromEntries(
      Object.entries(formState.inputs).map(([k, v]) => [k, v.value])
    );
    payload.company_group_id = selectedGroup?.value || null;
    payload.short_code_id = selectedShortCode?.value || null;

    try {
      const res = await ApiCall.ownCompanies.postEdit(id, payload);
      if (res.status === 200) {
        toast.success(res.data.message || "Sikeres mentés");
        router.push("/admin/own-company/list");
      } else {
        toast.error(res.data.error || "Sikertelen mentés");
      }
    } catch (err) {
      toast.error("Mentés nem sikerült");
    }
  }, [formState, id, selectedGroup, selectedShortCode]);
  console.log(
    "selectedshortcode",
    selectedShortCode,
    "selectedgroup",
    selectedGroup
  );
  return (
    <div className="p-6 max-w-4xl mx-auto">
      <h2 className="text-xl font-bold mb-4">Saját cég szerkesztése</h2>
      {companyData && companyOptions && shortCodes && (
        <>
          <Input
            id="name"
            name="name"
            element="input"
            placeholder="Cégnév"
            validators={[VALIDATOR_REQUIRE(), VALIDATOR_MINLENGTH(2)]}
            errorText="Kötelező mező"
            onInput={handleInput}
            initialValid={true}
            initialValue={companyData?.name}
          />
          <div className="mb-4">
            <label className="text-sm text-gray-600 mb-1 block">
              Cég névkód (opcionális):
            </label>
            <Select
              id="short_code_id"
              name="short_code_id"
              options={shortCodes}
              isClearable
              placeholder="Válassz egy névkódot"
              value={selectedShortCode}
              onChange={(selected) => {
                setSelectedShortCode(selected);
                handleInput("short_code_id", selected?.value || "", true);
              }}
            />
          </div>
          <Input
            id="address"
            name="address"
            element="input"
            placeholder="Cím"
            onInput={handleInput}
            initialValid={true}
            initialValue={companyData?.address}
          />
          <Input
            id="tax_number"
            name="tax_number"
            element="input"
            placeholder="Adószám"
            validators={[VALIDATOR_TAXNUMBER_HU]}
            errorText="Nem megfelelő adószám"
            onInput={handleInput}
            initialValid={true}
            initialValue={companyData?.tax_number}
          />
          <Input
            id="company_registration_number"
            name="company_registration_number"
            element="input"
            placeholder="Cégjegyzékszám"
            onInput={handleInput}
            initialValid={true}
            initialValue={companyData?.company_registration_number}
          />
          <Input
            id="email"
            name="email"
            element="input"
            placeholder="Email"
            validators={[VALIDATOR_EMAIL]}
            errorText="Nem megfelelő email"
            onInput={handleInput}
            initialValid={true}
            initialValue={companyData?.email}
          />
          <Input
            id="phone"
            name="phone"
            element="input"
            placeholder="Telefon"
            validators={[VALIDATOR_PHONE_HU]}
            errorText="Nem megfelelő formátum"
            initialValid={true}
            onInput={handleInput}
            initialValue={companyData?.phone}
          />
          <Input
            id="notes"
            name="notes"
            element="textarea"
            placeholder="Megjegyzés"
            onInput={handleInput}
            initialValid={true}
            initialValue={companyData?.notes}
          />

          <div className="mb-4">
            <label className="text-sm text-gray-600 mb-1 block">
              Cégcsoport (opcionális):
            </label>
            <Select
              id="company_group_id"
              name="company_group_id"
              options={companyOptions}
              isClearable
              placeholder="Válassz egy cégcsoportot"
              value={selectedGroup}
              onChange={(selected) => {
                setSelectedGroup(selected);
                handleInput("company_group_id", selected?.value || "", true);
              }}
            />
          </div>

          <button
            className={`mt-4 px-4 py-2 ${
              !formState.isValid
                ? "bg-gray-500"
                : "bg-green-600 hover:bg-green-700"
            } text-white rounded`}
            onClick={handleSubmit}
            disabled={!formState.isValid}>
            Mentés
          </button>
        </>
      )}
    </div>
  );
};

export default ExpandedRow;
