// src/components/people/create/page.tsx
"use client";

import { useState, useCallback } from "react";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";
import Select from "react-select";

type HrOption = { value: string; label: string; code?: string };

type Props = {
  isOpen: boolean;
  onClose: () => void;
  onCreated?: () => void;
  statusOptions: HrOption[];
  departmentOptions: HrOption[];
  companyOptions: HrOption[];
};

export default function CreatePersonModal({
  isOpen,
  onClose,
  onCreated,
  statusOptions,
  departmentOptions,
  companyOptions,
}: Props) {
  const [displayName, setDisplayName] = useState("");
  const [statusId, setStatusId] = useState<string | null>(null);
  const [departmentId, setDepartmentId] = useState<string | null>(null);
  const [hiredAt, setHiredAt] = useState<string>("");
  const [saving, setSaving] = useState(false);
  const [companyId, setCompanyId] = useState<string | null>(null);

  const handleSave = useCallback(async () => {
    if (!displayName.trim()) {
      toast.error("Név megadása kötelező");
      return;
    }
    setSaving(true);
    try {
      await ApiCall.peoples.upsert({
        display_name: displayName.trim(),
        is_active: true,
        status_id: statusId ?? null,
        company_department_id: departmentId ?? null,
        hired_at: hiredAt ? new Date(hiredAt).toISOString() : null,
        exited_at: null,
        employment_company_id: companyId ?? null,
      });
      toast.success("Személy létrehozva");
      onCreated?.();
      onClose();
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.message || "Személy létrehozása sikertelen"
      );
    } finally {
      setSaving(false);
    }
  }, [displayName, statusId, departmentId, hiredAt, onCreated, onClose]);

  if (!isOpen) return null;
  return (
    <div className="fixed inset-0 bg-black/20 flex items-center justify-center z-50">
      <div className="bg-white rounded-lg shadow-lg p-4 w-full max-w-lg space-y-3">
        <div className="flex justify-between items-center">
          <h2 className="font-semibold text-sm">Új személy</h2>
          <button onClick={onClose} className="text-xs text-gray-500">
            Bezár
          </button>
        </div>

        <div className="space-y-2 text-xs">
          <div>
            <div className="font-semibold">Név *</div>
            <input
              className="border rounded px-2 py-1 w-full"
              value={displayName}
              onChange={(e) => setDisplayName(e.target.value)}
            />
          </div>

          <div>
            <div className="font-semibold">Státusz</div>
            <Select
              classNamePrefix="hr"
              options={statusOptions}
              value={
                statusId
                  ? statusOptions.find((s) => s.value === statusId) ?? null
                  : null
              }
              onChange={(v) => setStatusId((v as any)?.value ?? null)}
              isClearable
            />
          </div>

          <div>
            <div className="font-semibold">Részleg</div>
            <Select
              classNamePrefix="hr"
              options={departmentOptions}
              value={
                departmentId
                  ? departmentOptions.find((d) => d.value === departmentId) ??
                    null
                  : null
              }
              onChange={(v) => setDepartmentId((v as any)?.value ?? null)}
              isClearable
            />
          </div>
          <div>
            <div className="font-semibold">Cég (own_companies)</div>
            <Select
              classNamePrefix="hr"
              options={companyOptions}
              value={
                companyId
                  ? companyOptions.find((c) => c.value === companyId) ?? null
                  : null
              }
              onChange={(v) => setCompanyId((v as any)?.value ?? null)}
              isClearable
              placeholder="Válassz céget"
            />
          </div>

          <div>
            <div className="font-semibold">Belépés dátuma</div>
            <input
              type="date"
              className="border rounded px-2 py-1"
              value={hiredAt}
              onChange={(e) => setHiredAt(e.target.value)}
            />
          </div>
        </div>

        <div className="flex justify-end gap-2">
          <button
            type="button"
            onClick={onClose}
            className="px-3 py-1.5 rounded border text-xs">
            Mégse
          </button>
          <button
            type="button"
            disabled={saving}
            onClick={handleSave}
            className="px-3 py-1.5 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white text-xs disabled:opacity-50">
            {saving ? "Mentés…" : "Létrehozás"}
          </button>
        </div>
      </div>
    </div>
  );
}
