// src/components/people/ExpandedPerson.tsx
"use client";

import { useEffect, useState, useCallback } from "react";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";
import Select from "react-select";
import AsyncSelect from "react-select/async";

type ExpandedPersonProps = {
  id: string;
  onUpdated?: () => void;
};
type PersonDetail = {
  person: any;
  emails: any[];
  phone_lines: any[];
  phone_ownerships: any[];
  assets: any[];
  aliases: any[];
};
type HrOption = { value: string; label: string; code?: string };

type HrTask = {
  id: string;
  title: string;
  description?: string | null;
  category?: string | null; // 'onboarding','offboarding', etc.
  status: "pending" | "in_progress" | "done" | "skipped";
  due_date?: string | null;
  assignee_id?: string | null;
  assignee_name?: string | null;
  completed_at?: string | null;

  // 👇 add these two
  is_overdue?: boolean;
  document_code?: string | null;
};

type TimelineItem = {
  id: string;
  type: string;
  occurred_at: string;
  note?: string | null;
  payload?: any;
  created_by_name?: string | null;
};
type PersonDocument = {
  id: string;
  template_code: string;
  title: string;
  created_at: string;
  created_by_name?: string | null;
};
type PersonEmployment = {
  id: string;
  company_id: string;
  company_name: string;
  company_short_code?: string | null;
  company_address?: string | null;
  hired_at?: string | null;
  exited_at?: string | null;
  employment_type?: string | null;
  contract_number?: string | null;
  is_primary?: boolean | null;
};
type AuditEvent = {
  id: string;
  event_type: "generated" | "viewed" | "export_docx" | "printed" | string;
  occurred_at: string;
  actor_name: string;
  meta?: any;
};
const TAB_LABELS = [
  { id: 1, label: "Áttekintés" },
  { id: 2, label: "E-mail címek" },
  { id: 3, label: "Telefonok" },
  { id: 4, label: "Eszközök" },
  { id: 5, label: "Tulajdon történet" },
  { id: 6, label: "HR feladatok" },
];

export default function ExpandedPerson({ id, onUpdated }: ExpandedPersonProps) {
  const [data, setData] = useState<PersonDetail | null>(null);
  const [activeTab, setActiveTab] = useState<number>(1);
  const [loading, setLoading] = useState(false);

  // 🔹 HR form state
  const [saving, setSaving] = useState(false);
  const [statusOptions, setStatusOptions] = useState<HrOption[]>([]);
  const [departmentOptions, setDepartmentOptions] = useState<HrOption[]>([]);

  const [displayName, setDisplayName] = useState("");
  const [isActive, setIsActive] = useState(true);
  const [statusId, setStatusId] = useState<string | null>(null);
  const [departmentId, setDepartmentId] = useState<string | null>(null);
  const [positionTitle, setPositionTitle] = useState<string>("");
  const [managerPersonId, setManagerPersonId] = useState<string | null>(null);
  const [managerLabel, setManagerLabel] = useState<string | null>(null);
  const [hiredAt, setHiredAt] = useState<string | null>(null); // yyyy-mm-dd
  const [exitedAt, setExitedAt] = useState<string | null>(null);
  // HR timeline
  const [timeline, setTimeline] = useState<TimelineItem[]>([]);
  const [timelineLoading, setTimelineLoading] = useState(false);

  // HR tasks
  const [tasks, setTasks] = useState<HrTask[]>([]);
  const [tasksLoading, setTasksLoading] = useState(false);

  // simple “create task” mini-form
  const [newTaskTitle, setNewTaskTitle] = useState("");
  const [newTaskCategory, setNewTaskCategory] = useState<string>("onboarding");
  const [newTaskDueDate, setNewTaskDueDate] = useState<string>("");

  const [documents, setDocuments] = useState<PersonDocument[]>([]);
  const [documentsLoading, setDocumentsLoading] = useState(false);

  const [docTemplates, setDocTemplates] = useState<
    { id: string; title: string }[]
  >([]);
  const [docLoading, setDocLoading] = useState(false);

  // company / employment
  const [employments, setEmployments] = useState<PersonEmployment[]>([]);
  const [companyOptions, setCompanyOptions] = useState<
    { value: string; label: string }[]
  >([]);

  const [companyMasterOptions, setCompanyMasterOptions] = useState<
    { value: string; label: string }[]
  >([]);

  const [audit, setAudit] = useState<AuditEvent[]>([]);
  const [auditLoading, setAuditLoading] = useState(false);

  const [employmentForm, setEmploymentForm] = useState<{
    id: string | null;
    company_id: string | null;
    hired_at: string;
    exited_at: string;
    employment_type: string;
    contract_number: string;
    is_primary: boolean;
  }>({
    id: null,
    company_id: null,
    hired_at: "",
    exited_at: "",
    employment_type: "",
    contract_number: "",
    is_primary: false,
  });
  const [selectedCompanyId, setSelectedCompanyId] = useState<string | null>(
    null
  );
  const [companyLoading, setCompanyLoading] = useState(false);

  const reloadTimeline = useCallback(async (personId: string) => {
    setTimelineLoading(true);
    try {
      const r = await ApiCall.peoples.getTimeline(personId);
      setTimeline(r.data?.items ?? []);
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message || "HR események betöltése sikertelen"
      );
    } finally {
      setTimelineLoading(false);
    }
  }, []);

  const reloadTasks = useCallback(
    async (personId: string, companyId?: string | null) => {
      setTasksLoading(true);
      try {
        const r = await ApiCall.peoples.getTasks(personId, {
          companyId: selectedCompanyId ?? undefined,
        });
        setTasks(r.data?.items ?? []);
      } catch (err: any) {
        console.error(err);
        toast.error(
          err?.response?.data?.message || "HR feladatok betöltése sikertelen"
        );
      } finally {
        setTasksLoading(false);
      }
    },
    []
  );
  const reloadDocuments = useCallback(
    async (personId: string, companyId?: string | null) => {
      setDocumentsLoading(true);
      try {
        const r = await ApiCall.hr.listPersonDocuments(personId, {
          companyId: companyId ?? undefined,
        });
        setDocuments(r.data?.items ?? []);
      } catch (err: any) {
        console.error(err);
        toast.error(
          err?.response?.data?.message || "Dokumentumok betöltése sikertelen"
        );
      } finally {
        setDocumentsLoading(false);
      }
    },
    []
  );
  const reloadCompanies = useCallback(async (personId: string) => {
    setCompanyLoading(true);
    try {
      const r = await ApiCall.peoples.getEmployments(personId);
      const items: PersonEmployment[] = r.data?.items ?? [];
      setEmployments(items);

      const opts = items.map((e) => ({
        value: e.company_id,
        label: e.company_short_code
          ? `${e.company_short_code} – ${e.company_name}`
          : e.company_name,
      }));
      setCompanyOptions(opts);

      const primary =
        items.find((e) => e.is_primary) ?? (items.length > 0 ? items[0] : null);
      setSelectedCompanyId(primary ? primary.company_id : null);
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message || "Cég kapcsolatok betöltése sikertelen"
      );
    } finally {
      setCompanyLoading(false);
    }
  }, []);

  const handleGenerateAndOpenDoc = useCallback(
    async (templateDocumentId: string) => {
      if (!templateDocumentId) return;
      if (!selectedCompanyId) {
        toast.error("Válassz céget a dokumentum generálásához");
        return;
      }

      if (!data?.person) {
        toast.error("Személy adatai még nem töltődtek be");
        return;
      }

      try {
        const values = {
          employee_name: data.person.display_name,
          today: new Date().toISOString().slice(0, 10),
          // ide jöhetnek még további tokenek
        };

        const res = await ApiCall.templateDocuments.fillAndSend(
          templateDocumentId,
          {
            values,
            person_id: id,
          }
        );

        const hrDocId = res.data?.hr_document_id as string | undefined;

        toast.success("Dokumentum generálva és mentve");

        await reloadTimeline(id);
        await reloadTasks(id, selectedCompanyId);
        await reloadDocuments(id, selectedCompanyId);

        if (hrDocId) {
          window.open(`/admin/hr/person-documents/${hrDocId}`, "_blank");
        }
      } catch (err: any) {
        console.error(err);
        toast.error(
          err?.response?.data?.error || "Dokumentum generálása sikertelen"
        );
      }
    },
    [id, data, selectedCompanyId, reloadTimeline, reloadTasks, reloadDocuments]
  );
  useEffect(() => {
    const loadCompanyOptions = async () => {
      try {
        const r = await ApiCall.peoples.getCompanyOptions();
        setCompanyMasterOptions(r.data?.items ?? []);
      } catch (err: any) {
        console.error(err);
        toast.error(
          err?.response?.data?.message || "Cég törzsadatok betöltése sikertelen"
        );
      }
    };

    loadCompanyOptions();
  }, []);
  useEffect(() => {
    const loadDocs = async () => {
      setDocLoading(true);
      try {
        const r = await ApiCall.templateDocuments.getList({
          kind_code: "hr_person_doc",
          is_active: "yes",
        });
        // backend: res.json(r.rows) => plain array
        setDocTemplates(r.data ?? []);
      } catch (e: any) {
        console.error(e);
        toast.error(
          e?.response?.data?.error ||
            "HR dokumentum sablonok betöltése sikertelen"
        );
      } finally {
        setDocLoading(false);
      }
    };
    const loadAudit = async () => {
      setAuditLoading(true);
      try {
        const res = await ApiCall.hr.getPersonDocumentAuditLog(id);
        setAudit(res.data?.items ?? res.data ?? []);
      } catch (err: any) {
        console.error(err);
        toast.error(
          err?.response?.data?.message ||
            "Napló bejegyzések betöltése sikertelen"
        );
      } finally {
        setAuditLoading(false);
      }
    };

    loadDocs();
    loadAudit();
  }, []);

  const eventLabel = (e: AuditEvent) => {
    switch (e.event_type) {
      case "generated":
        return "Létrehozva";
      case "viewed":
        return "Megnyitva";
      case "export_docx":
        return "DOCX export";
      case "printed":
        return "Nyomtatás";
      default:
        return e.event_type;
    }
  };

  useEffect(() => {
    if (!id) return;

    const fetchAll = async () => {
      setLoading(true);
      try {
        const [personRes, statusRes, deptRes] = await Promise.all([
          ApiCall.peoples.getOne(id),
          ApiCall.peoples.getStatuses(),
          ApiCall.peoples.getDepartments(),
        ]);

        const detail = personRes.data as PersonDetail;
        setData(detail);

        const sItems: HrOption[] = (statusRes.data?.items ?? []).map(
          (s: any) => ({
            value: s.value,
            label: s.label,
            code: s.code,
          })
        );
        const dItems: HrOption[] = (deptRes.data?.items ?? []).map(
          (d: any) => ({
            value: d.value,
            label: d.label,
          })
        );

        setStatusOptions(sItems);
        setDepartmentOptions(dItems);

        const p = detail.person;
        setDisplayName(p.display_name ?? "");
        setIsActive(!!p.is_active);
        setStatusId(p.status_id ?? null);
        setDepartmentId(p.company_department_id ?? null);
        setPositionTitle(p.position_title ?? "");
        setManagerPersonId(p.manager_person_id ?? null);
        setManagerLabel(p.manager_name ?? null);

        const toDateInput = (val?: string | null) =>
          val ? new Date(val).toISOString().slice(0, 10) : null;

        setHiredAt(toDateInput(p.hired_at));
        setExitedAt(toDateInput(p.exited_at));

        // 🔹 load timeline + tasks
        reloadTimeline(id);
        reloadTasks(id, selectedCompanyId);
        reloadDocuments(id, selectedCompanyId);
        reloadCompanies(id);
      } catch (err: any) {
        console.error(err);
        toast.error(
          err?.response?.data?.message ||
            "Személy adatainak betöltése sikertelen"
        );
      } finally {
        setLoading(false);
      }
    };

    fetchAll();
  }, [id, reloadTimeline, reloadTasks, reloadDocuments]);

  const loadManagerOptions = useCallback(async (inputValue: string) => {
    const r = await ApiCall.peoples.searchPeopleOptions(inputValue ?? "");
    return (r.data?.items ?? []).map((i: any) => ({
      value: i.value,
      label: i.label,
    }));
  }, []);
  const startNewEmployment = () => {
    setEmploymentForm({
      id: null,
      company_id: null,
      hired_at: "",
      exited_at: "",
      employment_type: "",
      contract_number: "",
      is_primary: employments.length === 0, // first one by default primary
    });
  };

  const startEditEmployment = (e: PersonEmployment) => {
    setEmploymentForm({
      id: e.id,
      company_id: e.company_id,
      hired_at: e.hired_at
        ? new Date(e.hired_at).toISOString().slice(0, 10)
        : "",
      exited_at: e.exited_at
        ? new Date(e.exited_at).toISOString().slice(0, 10)
        : "",
      employment_type: e.employment_type ?? "",
      contract_number: e.contract_number ?? "",
      is_primary: !!e.is_primary,
    });
  };

  const saveEmployment = async () => {
    if (!employmentForm.company_id) {
      toast.error("Válassz céget");
      return;
    }
    try {
      await ApiCall.peoples.upsertEmployment(id, {
        id: employmentForm.id ?? undefined,
        company_id: employmentForm.company_id,
        hired_at: employmentForm.hired_at
          ? new Date(employmentForm.hired_at).toISOString()
          : null,
        exited_at: employmentForm.exited_at
          ? new Date(employmentForm.exited_at).toISOString()
          : null,
        employment_type: employmentForm.employment_type || null,
        contract_number: employmentForm.contract_number || null,
        is_primary: employmentForm.is_primary,
      });
      toast.success("Munkaviszony mentve");
      await reloadCompanies(id);
      // ensure selected company reflects primary
      const primary =
        employments.find((e) => e.is_primary) ??
        (employments.length > 0 ? employments[0] : null);
      setSelectedCompanyId(
        primary ? primary.company_id : employmentForm.company_id
      );
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message || "Munkaviszony mentése sikertelen"
      );
    }
  };

  const deleteEmployment = async (employmentId: string) => {
    if (!confirm("Biztosan törlöd ezt a munkaviszonyt?")) return;
    try {
      await ApiCall.peoples.deleteEmployment(employmentId);
      toast.success("Munkaviszony törölve");
      await reloadCompanies(id);
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message || "Munkaviszony törlése sikertelen"
      );
    }
  };

  const handleSaveHr = async () => {
    if (!data) return;
    setSaving(true);
    try {
      await ApiCall.peoples.upsert({
        id: data.person.id,
        display_name: displayName.trim() || data.person.display_name,
        is_active: isActive,
        status_id: statusId ?? null,
        company_department_id: departmentId ?? null,
        position_title: positionTitle || null,
        manager_person_id: managerPersonId ?? null,
        hired_at: hiredAt ? new Date(hiredAt).toISOString() : null,
        exited_at: exitedAt ? new Date(exitedAt).toISOString() : null,
      });

      toast.success("HR adatok mentve");
      await reloadTimeline(data.person.id); // backend inserts hr_person_events
      onUpdated?.();
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.message || "HR adatok mentése sikertelen"
      );
    } finally {
      setSaving(false);
    }
  };

  if (loading && !data) {
    return <div className="p-4 text-sm text-gray-600">Betöltés…</div>;
  }

  if (!data) return null;

  const { person, emails, phone_lines, phone_ownerships, assets, aliases } =
    data;

  return (
    <div
      className="w-full px-2 py-2 space-y-3"
      style={{ overflow: "auto hidden", maxWidth: "85vw" }}
      onClick={(e) => e.stopPropagation()}>
      {/* Tabs */}
      <div className="flex gap-2 border-b pb-2">
        {TAB_LABELS.map((t) => (
          <button
            key={t.id}
            className={`px-3 py-1 text-sm rounded-t ${
              activeTab === t.id
                ? "bg-blue-600 text-white"
                : "bg-gray-100 text-gray-700 hover:bg-gray-200"
            }`}
            onClick={() => setActiveTab(t.id)}>
            {t.label}
          </button>
        ))}
      </div>

      {/* TAB 1: Overview */}
      {/* {activeTab === 1 && (
        <div className="space-y-2 text-sm">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-2">
            <div>
              <div className="font-semibold">Név</div>
              <div>{person.display_name}</div>
            </div>
            <div>
              <div className="font-semibold">Aktív?</div>
              <div>{person.is_active ? "Aktív" : "Inaktív"}</div>
            </div>
            <div>
              <div className="font-semibold">Részleg</div>
              <div>{person.department_name ?? "—"}</div>
            </div>
            <div>
              <div className="font-semibold">Beosztás</div>
              <div>{person.position_title ?? "—"}</div>
            </div>
            <div>
              <div className="font-semibold">Közvetlen felettes</div>
              <div>{person.manager_name ?? "—"}</div>
            </div>
            <div>
              <div className="font-semibold">Belépés</div>
              <div>
                {person.hired_at
                  ? new Date(person.hired_at).toLocaleDateString()
                  : "—"}
              </div>
            </div>
            <div>
              <div className="font-semibold">Kilépés</div>
              <div>
                {person.exited_at
                  ? new Date(person.exited_at).toLocaleDateString()
                  : "—"}
              </div>
            </div>
          </div>

          {aliases?.length ? (
            <div>
              <div className="font-semibold mt-3">Aliasok</div>
              <ul className="list-disc list-inside text-xs">
                {aliases.map((a: any) => (
                  <li key={a.id}>{a.alias}</li>
                ))}
              </ul>
            </div>
          ) : null}
        </div>
      )} */}
      {activeTab === 1 && (
        <div className="space-y-4 text-sm">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
            <div>
              <div className="font-semibold">Név</div>
              <input
                className="border rounded px-2 py-1 w-full"
                value={displayName}
                onChange={(e) => setDisplayName(e.target.value)}
              />
            </div>

            <div className="flex flex-col">
              <div className="font-semibold">Aktív?</div>
              <label className="inline-flex items-center gap-2 mt-1">
                <input
                  type="checkbox"
                  className="h-4 w-4"
                  checked={isActive}
                  onChange={(e) => setIsActive(e.target.checked)}
                />
                <span>{isActive ? "Aktív" : "Inaktív"}</span>
              </label>
            </div>

            <div className="flex flex-col">
              <div className="font-semibold">Státusz</div>
              <Select
                classNamePrefix="hr"
                options={statusOptions}
                value={
                  statusId
                    ? statusOptions.find((s) => s.value === statusId) ?? null
                    : null
                }
                onChange={(v) => setStatusId((v as any)?.value ?? null)}
                isClearable
                placeholder="Válassz státuszt"
              />
            </div>

            <div className="flex flex-col">
              <div className="font-semibold">Részleg</div>
              <Select
                classNamePrefix="hr"
                options={departmentOptions}
                value={
                  departmentId
                    ? departmentOptions.find((d) => d.value === departmentId) ??
                      null
                    : null
                }
                onChange={(v) => setDepartmentId((v as any)?.value ?? null)}
                isClearable
                placeholder="Válassz részleget"
              />
            </div>

            <div>
              <div className="font-semibold">Beosztás</div>
              <input
                className="border rounded px-2 py-1 w-full"
                value={positionTitle}
                onChange={(e) => setPositionTitle(e.target.value)}
              />
            </div>

            <div className="flex flex-col">
              <div className="font-semibold">Közvetlen felettes</div>
              <AsyncSelect
                classNamePrefix="hr"
                cacheOptions
                defaultOptions
                loadOptions={loadManagerOptions}
                value={
                  managerPersonId
                    ? { value: managerPersonId, label: managerLabel ?? "—" }
                    : null
                }
                onChange={(v: any) => {
                  setManagerPersonId(v?.value ?? null);
                  setManagerLabel(v?.label ?? null);
                }}
                isClearable
                placeholder="Válassz felettest"
              />
            </div>

            <div>
              <div className="font-semibold">Belépés</div>
              <input
                type="date"
                className="border rounded px-2 py-1"
                value={hiredAt ?? ""}
                onChange={(e) => setHiredAt(e.target.value || null)}
              />
            </div>

            <div>
              <div className="font-semibold">Kilépés</div>
              <input
                type="date"
                className="border rounded px-2 py-1"
                value={exitedAt ?? ""}
                onChange={(e) => setExitedAt(e.target.value || null)}
              />
            </div>
            {/* Audit log */}
            <section className="border rounded p-3 bg-gray-50 space-y-2">
              <div className="flex justify-between items-center">
                <h2 className="text-sm font-semibold">
                  Napló (megnyitások, exportok)
                </h2>
                {auditLoading && (
                  <span className="text-[11px] text-gray-500">Betöltés…</span>
                )}
              </div>

              {audit.length === 0 && !auditLoading ? (
                <p className="text-xs text-gray-500">
                  Még nincs naplózva esemény ehhez a dokumentumhoz.
                </p>
              ) : (
                <table className="min-w-full text-xs border bg-white">
                  <thead className="bg-gray-100">
                    <tr>
                      <th className="px-2 py-1 border text-left">Esemény</th>
                      <th className="px-2 py-1 border text-left">
                        Felhasználó
                      </th>
                      <th className="px-2 py-1 border text-left">Időpont</th>
                      <th className="px-2 py-1 border text-left">Részletek</th>
                    </tr>
                  </thead>
                  <tbody>
                    {audit.map((e) => (
                      <tr key={e.id}>
                        <td className="px-2 py-1 border">{eventLabel(e)}</td>
                        <td className="px-2 py-1 border">{e.actor_name}</td>
                        <td className="px-2 py-1 border">
                          {new Date(e.occurred_at).toLocaleString("hu-HU")}
                        </td>
                        <td className="px-2 py-1 border">
                          {e.meta?.ip ? (
                            <span className="text-[11px] text-gray-500">
                              IP: {e.meta.ip}
                            </span>
                          ) : (
                            "—"
                          )}
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              )}
            </section>
          </div>

          {aliases?.length ? (
            <div>
              <div className="font-semibold mt-3">Aliasok</div>
              <ul className="list-disc list-inside text-xs">
                {aliases.map((a: any) => (
                  <li key={a.id}>{a.alias}</li>
                ))}
              </ul>
            </div>
          ) : null}

          {timelineLoading && (
            <div className="text-xs text-gray-500 mt-3">
              HR események betöltése…
            </div>
          )}

          {timeline.length > 0 && !timelineLoading && (
            <div className="mt-4">
              <div className="font-semibold mb-1 text-sm">HR események</div>
              <ul className="text-xs space-y-1 max-h-52 overflow-auto pr-1">
                {timeline.map((ev) => (
                  <li
                    key={ev.id}
                    className="border border-gray-200 rounded px-2 py-1 bg-gray-50">
                    <div className="flex justify-between gap-2">
                      <span className="font-semibold">
                        {ev.type.replace(/_/g, " ")}
                      </span>
                      <span className="text-[11px] text-gray-500">
                        {new Date(ev.occurred_at).toLocaleString("hu-HU")}
                      </span>
                    </div>
                    {ev.note && (
                      <div className="text-[11px] text-gray-700">{ev.note}</div>
                    )}
                    {ev.created_by_name && (
                      <div className="text-[10px] text-gray-500 mt-0.5">
                        Rögzítette: {ev.created_by_name}
                      </div>
                    )}
                  </li>
                ))}
              </ul>
            </div>
          )}

          <div className="flex justify-end">
            <button
              type="button"
              onClick={handleSaveHr}
              disabled={saving}
              className="px-3 py-2 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white text-xs disabled:opacity-50">
              {saving ? "Mentés…" : "HR adatok mentése"}
            </button>
          </div>
        </div>
      )}
      {/* TAB 2: Emails */}
      {activeTab === 2 && (
        <div className="text-xs">
          {emails.length === 0 ? (
            <div className="text-gray-500">Nincs e-mail cím.</div>
          ) : (
            <table className="min-w-full text-xs border">
              <thead className="bg-gray-100">
                <tr>
                  <th className="px-2 py-1 border">E-mail</th>
                  <th className="px-2 py-1 border">Típus</th>
                  <th className="px-2 py-1 border">Archiválni</th>
                  <th className="px-2 py-1 border">Átirányítás</th>
                  <th className="px-2 py-1 border">Megjegyzés</th>
                </tr>
              </thead>
              <tbody>
                {emails.map((e: any) => (
                  <tr key={e.id}>
                    <td className="px-2 py-1 border">{e.email}</td>
                    <td className="px-2 py-1 border">{e.type ?? "—"}</td>
                    <td className="px-2 py-1 border">
                      {e.should_archive ? "Igen" : "Nem"}
                    </td>
                    <td className="px-2 py-1 border">
                      {e.is_forwarded ? "Igen" : "Nem"}
                    </td>
                    <td className="px-2 py-1 border">{e.notes ?? ""}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          )}
        </div>
      )}

      {/* TAB 3: Phone lines */}
      {activeTab === 3 && (
        <div className="text-xs">
          {phone_lines.length === 0 ? (
            <div className="text-gray-500">Nincs hozzárendelt telefonszám.</div>
          ) : (
            <table className="min-w-full text-xs border">
              <thead className="bg-gray-100">
                <tr>
                  <th className="px-2 py-1 border">Telefonszám</th>
                  <th className="px-2 py-1 border">Megnevezés</th>
                  <th className="px-2 py-1 border">Státusz</th>
                  <th className="px-2 py-1 border">Internet csomag</th>
                  <th className="px-2 py-1 border">Költséghely</th>
                  <th className="px-2 py-1 border">Költségviselő</th>
                </tr>
              </thead>
              <tbody>
                {phone_lines.map((pl: any) => (
                  <tr key={pl.id}>
                    <td className="px-2 py-1 border">{pl.phone_number}</td>
                    <td className="px-2 py-1 border">
                      {pl.display_name ?? "—"}
                    </td>
                    <td className="px-2 py-1 border">
                      {pl.status_name ?? "—"}
                    </td>
                    <td className="px-2 py-1 border">{pl.data_plan ?? "—"}</td>
                    <td className="px-2 py-1 border">
                      {pl.current_cost_center ?? "—"}
                    </td>
                    <td className="px-2 py-1 border">
                      {pl.cost_bearer ?? "—"}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          )}
        </div>
      )}

      {/* TAB 4: Assets */}
      {activeTab === 4 && (
        <div className="text-xs">
          {assets.length === 0 ? (
            <div className="text-gray-500">Nincs hozzárendelt eszköz.</div>
          ) : (
            <table className="min-w-full text-xs border">
              <thead className="bg-gray-100">
                <tr>
                  <th className="px-2 py-1 border">Leltári szám</th>
                  <th className="px-2 py-1 border">Eszköz név</th>
                  <th className="px-2 py-1 border">Csoport</th>
                  <th className="px-2 py-1 border">Státusz</th>
                  <th className="px-2 py-1 border">Gyári szám</th>
                </tr>
              </thead>
              <tbody>
                {assets.map((a: any) => (
                  <tr key={a.id}>
                    <td className="px-2 py-1 border">
                      {a.asset_tag ?? a.inventory_number ?? "—"}
                    </td>
                    <td className="px-2 py-1 border">{a.name ?? "—"}</td>
                    <td className="px-2 py-1 border">{a.category ?? "—"}</td>
                    <td className="px-2 py-1 border">{a.status ?? "—"}</td>
                    <td className="px-2 py-1 border">
                      {a.serial_number ?? "—"}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          )}
        </div>
      )}

      {/* TAB 5: Ownership history (phones) */}
      {activeTab === 5 && (
        <div className="text-xs">
          {phone_ownerships.length === 0 ? (
            <div className="text-gray-500">
              Nincs telefon tulajdon történet.
            </div>
          ) : (
            <table className="min-w-full text-xs border">
              <thead className="bg-gray-100">
                <tr>
                  <th className="px-2 py-1 border">Telefonszám</th>
                  <th className="px-2 py-1 border">Kezdete</th>
                  <th className="px-2 py-1 border">Vége</th>
                </tr>
              </thead>
              <tbody>
                {phone_ownerships.map((h: any) => (
                  <tr key={h.id}>
                    <td className="px-2 py-1 border">
                      {h.phone_number ?? "—"}
                    </td>
                    <td className="px-2 py-1 border">
                      {h.started_at
                        ? new Date(h.started_at).toLocaleDateString()
                        : "—"}
                    </td>
                    <td className="px-2 py-1 border">
                      {h.ended_at
                        ? new Date(h.ended_at).toLocaleDateString()
                        : "—"}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          )}
        </div>
      )}

      {/* TAB 6: HR tasks / checklists */}
      {/* TAB 6: HR feladatok */}
      {activeTab === 6 && (
        <div className="text-xs space-y-4">
          {/* Employment / companies overview + editor */}
          <div className="mt-4 border rounded p-2 bg-gray-50 space-y-3 text-xs">
            <div className="flex justify-between items-center">
              <div className="font-semibold text-[13px]">
                Munkaviszonyok (cégek)
              </div>
              <button
                type="button"
                onClick={startNewEmployment}
                className="px-2 py-1 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white text-[11px]">
                Új munkaviszony
              </button>
            </div>

            {employments.length === 0 ? (
              <div className="text-gray-500">
                Még nincs ehhez a személyhez rögzített munkaviszony.
              </div>
            ) : (
              <ul className="space-y-1">
                {employments.map((e) => (
                  <li
                    key={e.id}
                    className="flex justify-between items-center border border-gray-200 rounded px-2 py-1 bg-white">
                    <div>
                      <div className="font-semibold">
                        {e.company_short_code
                          ? `${e.company_short_code} – ${e.company_name}`
                          : e.company_name}
                        {e.is_primary ? " • (elsődleges)" : ""}
                      </div>
                      <div className="text-[11px] text-gray-500">
                        {e.hired_at
                          ? `Belépés: ${new Date(e.hired_at).toLocaleDateString(
                              "hu-HU"
                            )}`
                          : "Belépés: —"}
                        {e.exited_at
                          ? ` • Kilépés: ${new Date(
                              e.exited_at
                            ).toLocaleDateString("hu-HU")}`
                          : ""}
                      </div>
                    </div>
                    <div className="flex gap-1">
                      <button
                        type="button"
                        onClick={() => startEditEmployment(e)}
                        className="px-2 py-1 rounded bg-blue-600/80 hover:bg-blue-600 text-white text-[11px]">
                        Szerkesztés
                      </button>
                      <button
                        type="button"
                        onClick={() => deleteEmployment(e.id)}
                        className="px-2 py-1 rounded bg-red-600/80 hover:bg-red-600 text-white text-[11px]">
                        Törlés
                      </button>
                    </div>
                  </li>
                ))}
              </ul>
            )}

            {/* Inline form for new / edit */}
            {employmentForm.company_id !== null ||
            employmentForm.id === null ? (
              <div className="border-t pt-2 mt-2">
                <div className="font-semibold text-[12px] mb-1">
                  {employmentForm.id
                    ? "Munkaviszony szerkesztése"
                    : "Új munkaviszony"}
                </div>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-2 items-center">
                  <div className="md:col-span-2">
                    <div className="text-[11px] mb-0.5">Cég *</div>
                    <Select
                      classNamePrefix="hr-company-master"
                      options={companyMasterOptions}
                      value={
                        employmentForm.company_id
                          ? companyMasterOptions.find(
                              (o) => o.value === employmentForm.company_id
                            ) ?? null
                          : null
                      }
                      onChange={(v: any) =>
                        setEmploymentForm((prev) => ({
                          ...prev,
                          company_id: v?.value ?? null,
                        }))
                      }
                      placeholder="Válassz céget"
                    />
                  </div>
                  <div className="flex items-center gap-2">
                    <input
                      id="employment-primary"
                      type="checkbox"
                      className="h-4 w-4"
                      checked={employmentForm.is_primary}
                      onChange={(e) =>
                        setEmploymentForm((prev) => ({
                          ...prev,
                          is_primary: e.target.checked,
                        }))
                      }
                    />
                    <label htmlFor="employment-primary" className="text-[11px]">
                      Elsődleges munkaviszony
                    </label>
                  </div>

                  <div>
                    <div className="text-[11px] mb-0.5">Belépés dátuma</div>
                    <input
                      type="date"
                      className="border rounded px-2 py-1 w-full"
                      value={employmentForm.hired_at}
                      onChange={(e) =>
                        setEmploymentForm((prev) => ({
                          ...prev,
                          hired_at: e.target.value,
                        }))
                      }
                    />
                  </div>
                  <div>
                    <div className="text-[11px] mb-0.5">Kilépés dátuma</div>
                    <input
                      type="date"
                      className="border rounded px-2 py-1 w-full"
                      value={employmentForm.exited_at}
                      onChange={(e) =>
                        setEmploymentForm((prev) => ({
                          ...prev,
                          exited_at: e.target.value,
                        }))
                      }
                    />
                  </div>
                  <div>
                    <div className="text-[11px] mb-0.5">
                      Munkaviszony típusa
                    </div>
                    <input
                      className="border rounded px-2 py-1 w-full"
                      value={employmentForm.employment_type}
                      onChange={(e) =>
                        setEmploymentForm((prev) => ({
                          ...prev,
                          employment_type: e.target.value,
                        }))
                      }
                      placeholder="pl. Határozatlan idejű"
                    />
                  </div>
                  <div>
                    <div className="text-[11px] mb-0.5">Szerződésszám</div>
                    <input
                      className="border rounded px-2 py-1 w-full"
                      value={employmentForm.contract_number}
                      onChange={(e) =>
                        setEmploymentForm((prev) => ({
                          ...prev,
                          contract_number: e.target.value,
                        }))
                      }
                      placeholder="pl. 2025/HR-001"
                    />
                  </div>

                  <div className="md:col-span-3 flex justify-end gap-2 mt-1">
                    <button
                      type="button"
                      onClick={startNewEmployment}
                      className="px-2 py-1 rounded border text-[11px]">
                      Mégse
                    </button>
                    <button
                      type="button"
                      onClick={saveEmployment}
                      className="px-3 py-1.5 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white text-[11px]">
                      Mentés
                    </button>
                  </div>
                </div>
              </div>
            ) : null}
          </div>
          {/* Actions row */}
          <div className="flex flex-wrap gap-2 items-center">
            <button
              type="button"
              onClick={async () => {
                try {
                  if (!selectedCompanyId) {
                    toast.error("Válassz céget az onboarding indításához");
                    return;
                  }
                  await ApiCall.peoples.startOnboarding(id, selectedCompanyId);
                  toast.success("Onboarding feladatok generálva");
                  await reloadTasks(id, selectedCompanyId);
                  await reloadTimeline(id);
                } catch (error) {
                  console.error(error);
                  toast.error(
                    error?.response?.data?.message ||
                      "Onboarding indítása sikertelen"
                  );
                }
              }}
              className="px-3 py-1.5 rounded bg-green-600/80 hover:bg-green-600 text-white disabled:opacity-50">
              Onboarding indítása
            </button>

            <button
              type="button"
              disabled={tasksLoading}
              onClick={async () => {
                try {
                  if (!selectedCompanyId) {
                    toast.error("Válassz céget az onboarding indításához");
                    return;
                  }
                  await ApiCall.peoples.startOffboarding(id, selectedCompanyId);
                  toast.success("Offboarding elindítva, feladatok létrehozva");
                  await reloadTasks(id, selectedCompanyId);
                  await reloadTimeline(id);
                } catch (err: any) {
                  console.error(err);
                  toast.error(
                    err?.response?.data?.message ||
                      "Offboarding indítása sikertelen"
                  );
                }
              }}
              className="px-3 py-1.5 rounded bg-amber-600/80 hover:bg-amber-600 text-white disabled:opacity-50">
              Offboarding indítása
            </button>

            {tasksLoading && (
              <span className="text-gray-500">Feladatok betöltése…</span>
            )}
          </div>
          {/* Company selector for documents/tasks context */}
          <div className="border rounded p-2 bg-gray-50 space-y-2">
            <div className="font-semibold text-[13px]">Cég (munkaviszony)</div>
            {companyLoading ? (
              <div className="text-xs text-gray-500">Cégek betöltése…</div>
            ) : companyOptions.length === 0 ? (
              <div className="text-xs text-gray-500">
                Nincs ehhez a személyhez rögzített munkaviszony.
              </div>
            ) : (
              <Select
                classNamePrefix="hr-company"
                options={companyOptions}
                value={
                  selectedCompanyId
                    ? companyOptions.find(
                        (c) => c.value === selectedCompanyId
                      ) ?? null
                    : null
                }
                onChange={(v) =>
                  setSelectedCompanyId((v as any)?.value ?? null)
                }
                isClearable={false}
                placeholder="Válassz céget"
              />
            )}
          </div>

          {/* Quick add task */}
          <div className="border rounded p-2 bg-gray-50 space-y-2">
            <div className="font-semibold text-[13px]">
              Új HR feladat hozzáadása
            </div>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-2 items-center">
              <input
                className="border rounded px-2 py-1 text-xs md:col-span-2"
                placeholder="Feladat megnevezése (pl. „Munkaszerződés aláíratása”)"
                value={newTaskTitle}
                onChange={(e) => setNewTaskTitle(e.target.value)}
              />
              <select
                className="border rounded px-2 py-1 text-xs"
                value={newTaskCategory}
                onChange={(e) => setNewTaskCategory(e.target.value)}>
                <option value="onboarding">Belépés</option>
                <option value="offboarding">Kilépés</option>
                <option value="other">Egyéb</option>
              </select>
              <div className="flex items-center gap-2">
                <span>Határidő:</span>
                <input
                  type="date"
                  className="border rounded px-2 py-1 text-xs"
                  value={newTaskDueDate}
                  onChange={(e) => setNewTaskDueDate(e.target.value)}
                />
              </div>
              <div className="md:col-span-3 flex justify-end">
                <button
                  type="button"
                  disabled={!newTaskTitle.trim() || tasksLoading}
                  onClick={async () => {
                    try {
                      await ApiCall.peoples.createOrUpdateTask(id, {
                        title: newTaskTitle.trim(),
                        category: newTaskCategory,
                        due_date: newTaskDueDate || null,
                        status: "pending",
                        company_id: selectedCompanyId ?? null,
                      });
                      setNewTaskTitle("");
                      setNewTaskDueDate("");
                      toast.success("Feladat hozzáadva");
                      await reloadTasks(id, selectedCompanyId);
                      await reloadTimeline(id);
                    } catch (err: any) {
                      console.error(err);
                      toast.error(
                        err?.response?.data?.message ||
                          "Feladat hozzáadása sikertelen"
                      );
                    }
                  }}
                  className="px-3 py-1.5 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white text-xs disabled:opacity-50">
                  Hozzáadás
                </button>
              </div>
            </div>
          </div>

          {/* Task list */}
          {tasks.length === 0 && !tasksLoading ? (
            <div className="text-gray-500">
              Nincs HR feladat ehhez a személyhez.
            </div>
          ) : (
            <table className="min-w-full text-xs border">
              <thead className="bg-gray-100">
                <tr>
                  <th className="px-2 py-1 border">Feladat</th>
                  <th className="px-2 py-1 border">Kategória</th>
                  <th className="px-2 py-1 border">Határidő</th>
                  <th className="px-2 py-1 border">Státusz</th>
                  <th className="px-2 py-1 border">Felelős</th>
                  <th className="px-2 py-1 border">Művelet</th>
                </tr>
              </thead>
              <tbody>
                {tasks.map((t) => (
                  <tr key={t.id}>
                    <td className="px-2 py-1 border">{t.title}</td>
                    <td className="px-2 py-1 border">
                      {t.category === "onboarding"
                        ? "Belépés"
                        : t.category === "offboarding"
                        ? "Kilépés"
                        : t.category || "—"}
                    </td>
                    <td className="px-2 py-1 border">
                      <input
                        type="date"
                        className="border rounded px-1 py-0.5 text-[11px]"
                        value={
                          t.due_date
                            ? new Date(t.due_date).toISOString().slice(0, 10)
                            : ""
                        }
                        onChange={async (e) => {
                          const value = e.target.value || null;
                          try {
                            await ApiCall.peoples.createOrUpdateTask(id, {
                              id: t.id,
                              person_id: id,
                              title: t.title,
                              description: t.description,
                              category: t.category,
                              status: t.status,
                              due_date: value,
                              assignee_id: t.assignee_id ?? null,
                              company_id: selectedCompanyId ?? null,
                            });
                            await reloadTasks(id, selectedCompanyId);
                            await reloadTimeline(id);
                          } catch (err: any) {
                            toast.error("Határidő mentése sikertelen");
                          }
                        }}
                      />
                    </td>
                    <td className="px-2 py-1 border">
                      {t.status === "done" ? (
                        <span className="inline-flex items-center px-2 py-0.5 rounded text-[11px] bg-emerald-100 text-emerald-700">
                          Kész
                        </span>
                      ) : t.is_overdue ? (
                        <span className="inline-flex items-center px-2 py-0.5 rounded text-[11px] bg-red-100 text-red-700">
                          Lejárt
                        </span>
                      ) : (
                        <span className="inline-flex items-center px-2 py-0.5 rounded text-[11px] bg-amber-100 text-amber-700">
                          Folyamatban
                        </span>
                      )}
                    </td>
                    <td className="px-2 py-1 border">
                      <AsyncSelect
                        classNamePrefix="hr-task"
                        cacheOptions
                        defaultOptions
                        loadOptions={loadManagerOptions} // same helper as for manager field
                        value={
                          t.assignee_id
                            ? {
                                value: t.assignee_id,
                                label: t.assignee_name ?? "—",
                              }
                            : null
                        }
                        onChange={async (v: any) => {
                          try {
                            await ApiCall.peoples.createOrUpdateTask(id, {
                              id: t.id,
                              person_id: id,
                              title: t.title,
                              description: t.description,
                              category: t.category,
                              status: t.status,
                              due_date: t.due_date ?? null,
                              assignee_id: v?.value ?? null,
                              company_id: selectedCompanyId ?? null,
                            });
                            await reloadTasks(id, selectedCompanyId);
                            await reloadTimeline(id);
                          } catch (err: any) {
                            toast.error("Felelős mentése sikertelen");
                          }
                        }}
                        isClearable
                        placeholder="Felelős"
                      />
                    </td>
                    <td className="px-2 py-1 border text-center">
                      {t.status === "done" ? (
                        t.completed_at ? (
                          <span className="text-[11px] text-gray-500">
                            {new Date(t.completed_at).toLocaleDateString(
                              "hu-HU"
                            )}
                          </span>
                        ) : (
                          "—"
                        )
                      ) : (
                        <>
                          <button
                            type="button"
                            disabled={tasksLoading}
                            onClick={async () => {
                              try {
                                await ApiCall.peoples.completeTask(t.id);
                                toast.success("Feladat lezárva");
                                await reloadTasks(id, selectedCompanyId);
                                await reloadTimeline(id);
                              } catch (err: any) {
                                console.error(err);
                                toast.error(
                                  err?.response?.data?.message ||
                                    "Feladat lezárása sikertelen"
                                );
                              }
                            }}
                            className="px-2 py-1 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white disabled:opacity-50">
                            Kész
                          </button>
                        </>
                      )}
                      {t.document_code && (
                        <button
                          type="button"
                          onClick={() =>
                            handleGenerateAndOpenDoc(t.document_code!)
                          }
                          className="ml-1 px-2 py-1 rounded bg-sky-600/80 hover:bg-sky-600 text-white text-[11px]">
                          Dokumentum
                        </button>
                      )}
                      <button
                        type="button"
                        onClick={async () => {
                          await ApiCall.peoples.deleteTask(t.id);
                          toast.success("Feladat törölve");
                          await reloadTasks(id, selectedCompanyId);
                          await reloadTimeline(id);
                        }}
                        className="ml-1 px-2 py-1 rounded bg-red-600/80 hover:bg-red-600 text-white">
                        Törlés
                      </button>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          )}
          <div className="border rounded p-2 bg-gray-50 space-y-2">
            <div className="font-semibold text-[13px]">
              Dokumentum generálása
            </div>
            {docLoading ? (
              <div className="text-xs text-gray-500">Betöltés…</div>
            ) : (
              <div className="flex gap-2 items-center">
                <select
                  className="border rounded px-2 py-1 text-xs"
                  onChange={async (e) => {
                    const templateId = e.target.value;
                    if (!templateId) return;
                    await handleGenerateAndOpenDoc(templateId);
                    e.target.value = "";
                  }}>
                  <option value="">Válassz dokumentumot…</option>
                  {docTemplates.map((tpl) => (
                    <option key={tpl.id} value={tpl.id}>
                      {tpl.title}
                    </option>
                  ))}
                </select>
              </div>
            )}
          </div>
          {/* Saved documents list */}
          <div className="border rounded p-2 bg-gray-50 space-y-2">
            <div className="font-semibold text-[13px]">
              Korábban generált dokumentumok
            </div>
            {documentsLoading ? (
              <div className="text-xs text-gray-500">Betöltés…</div>
            ) : documents.length === 0 ? (
              <div className="text-xs text-gray-500">
                Még nincs generált dokumentum ehhez a személyhez.
              </div>
            ) : (
              <ul className="space-y-1 max-h-52 overflow-auto pr-1">
                {documents.map((d) => (
                  <li
                    key={d.id}
                    className="flex justify-between items-center border border-gray-200 rounded px-2 py-1 bg-white">
                    <div>
                      <div className="font-semibold text-[12px]">{d.title}</div>
                      <div className="text-[11px] text-gray-500">
                        Kód: {d.template_code} •{" "}
                        {new Date(d.created_at).toLocaleString("hu-HU")}
                        {d.created_by_name
                          ? ` • Készítette: ${d.created_by_name}`
                          : null}
                      </div>
                    </div>
                    <button
                      type="button"
                      onClick={() =>
                        window.open(
                          `/admin/hr/person-documents/${d.id}`,
                          "_blank"
                        )
                      }
                      className="ml-2 px-2 py-1 rounded bg-sky-600/80 hover:bg-sky-600 text-white text-[11px]">
                      Megnyitás
                    </button>
                  </li>
                ))}
              </ul>
            )}
          </div>
        </div>
      )}
    </div>
  );
}
