// src/components/phone-registry/ExpandedPhoneRegistry.tsx
"use client";

import { useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import PhoneRegistryForm, { PhoneUpsertPayload } from "./PhoneRegistryForm";

export default function ExpandedPhoneRegistry({
  row,
  onSaved,
  onDeleted,
}: {
  row: any; // PhoneRow from the table
  onSaved?: () => void;
  onDeleted?: () => void;
}) {
  const [saving, setSaving] = useState(false);
  const [removing, setRemoving] = useState(false);

  const handleSave = async (payload: PhoneUpsertPayload) => {
    setSaving(true);
    try {
      await ApiCall.phoneRegistry.upsert(payload);
      toast.success("Frissítve");
      onSaved?.();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.message || "Mentés sikertelen");
    } finally {
      setSaving(false);
    }
  };

  const handleDelete = async () => {
    if (!confirm("Biztosan törli?")) return;
    setRemoving(true);
    try {
      await ApiCall.phoneRegistry.remove(row.id);
      toast.success("Törölve");
      onDeleted?.();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.message || "Törlés sikertelen");
    } finally {
      setRemoving(false);
    }
  };

  return (
    <div className="rounded border p-3 bg-white/50 space-y-3 max-w-[90vw]">
      <PhoneRegistryForm
        initial={{
          sort_order: row.sort_order ?? null,
          phone_number: row.phone_number,
          handler: row.handler_name ?? null,
          group: row.segment_name ?? null,
          status: row.status_name ?? null,
          pause_start: row.pause_start ?? null,
          pause_end: row.pause_end ?? null,
          display_name: row.display_name ?? null,
          issued_at: row.issued_at ?? null,
          location_text:
            row.location_short_code ??
            row.location_department_name ??
            row.location_text ??
            null,
          device_brand: row.device_brand_name ?? null,
          device_model: row.device_model_name ?? null,
          device_serial: row.device_serial ?? null,
          imei: Array.isArray(row.imeis) ? row.imeis : [],
          sim_pins: Array.isArray(row.sim_pins) ? row.sim_pins : [],
          sim_puks: Array.isArray(row.sim_puks) ? row.sim_puks : [],
          sim_card_serial: row.sim_card_serial ?? null,
          data_plan: row.internet_package_name ?? row.data_plan ?? null,
          current_cost_center: row.current_cost_center ?? null,
          cost_bearer: row.cost_bearer ?? null,
          future_cost_center: row.future_cost_center ?? null,
          notes: row.notes ?? null,
        }}
        onSubmit={handleSave}
        saving={saving}
      />

      <div className="flex items-center justify-between">
        <span className="text-xs text-gray-500">
          Utoljára módosítva: {row.updated_at ?? "-"}
        </span>
        <button
          onClick={handleDelete}
          disabled={removing}
          className="px-3 py-2 bg-red-600/80 hover:bg-red-600 text-white rounded disabled:opacity-50">
          Törlés
        </button>
      </div>
    </div>
  );
}
