"use client";

import React, { useMemo, useRef, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

import {
  Card,
  CardHeader,
  CardBody,
  CardTitle,
  CardMeta,
  ToolbarRow,
  TextInput,
  TextArea,
  Button,
  SecondaryButton,
  Select,
  Field,
  Chip,
  EmptyState,
} from "@/components/ui/CardBits";
import { Drawer } from "@/components/ui/Drawer";

import {
  DndContext,
  DragEndEvent,
  PointerSensor,
  KeyboardSensor,
  useSensor,
  useSensors,
  closestCenter,
} from "@dnd-kit/core";
import {
  SortableContext,
  useSortable,
  arrayMove,
  sortableKeyboardCoordinates,
  verticalListSortingStrategy,
} from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";

type FlowType = "projekt" | "folyamatos" | "";

type StepDraft = {
  title: string;
  description: string;
  saving?: boolean;
};

function normalizeDesc(v?: string | null) {
  return (v ?? "").trim();
}
function normalizeTitle(v?: string | null) {
  return (v ?? "").trim();
}
function computeSortOrder(index: number) {
  return index * 10;
}

function StepRow({
  step,
  draft,
  onChange,
  onSave,
  onReset,
  onDelete,
  disabled,
}: {
  step: any; // PlaybookStep
  draft: StepDraft;
  onChange: (patch: Partial<StepDraft>) => void;
  onSave: () => void;
  onReset: () => void;
  onDelete: () => void;
  disabled?: boolean;
}) {
  const {
    attributes,
    listeners,
    setNodeRef,
    transform,
    transition,
    isDragging,
  } = useSortable({ id: step.id });

  const style: React.CSSProperties = {
    transform: CSS.Transform.toString(transform),
    transition,
    opacity: isDragging ? 0.8 : 1,
  };

  const dirty =
    normalizeTitle(draft.title) !== normalizeTitle(step.title) ||
    normalizeDesc(draft.description) !== normalizeDesc(step.description);

  return (
    <Card
      ref={setNodeRef as any}
      style={style}
      className="border border-black/10">
      <CardHeader>
        <div style={{ minWidth: 0 }}>
          <CardTitle className="flex items-center gap-2">
            <span className="text-xs opacity-60">#{step.sort_order}</span>

            <span
              className="cursor-grab select-none px-2 py-1 rounded-md border border-black/10 opacity-80"
              title="Húzd a rendezéshez"
              {...attributes}
              {...listeners}>
              ⠿
            </span>

            <span className="truncate">
              {normalizeTitle(draft.title) || "(cím nélküli)"}
            </span>
          </CardTitle>
          <CardMeta>ID: {step.id}</CardMeta>
        </div>

        <div className="flex gap-2 flex-wrap">
          <SecondaryButton
            onClick={onReset}
            disabled={!dirty || disabled || draft.saving}>
            Visszaállít
          </SecondaryButton>
          <Button
            onClick={onSave}
            disabled={
              !dirty || disabled || draft.saving || !normalizeTitle(draft.title)
            }>
            {draft.saving ? "Mentés…" : "Mentés"}
          </Button>
          <SecondaryButton
            onClick={onDelete}
            disabled={disabled || draft.saving}>
            Törlés
          </SecondaryButton>
        </div>
      </CardHeader>

      <CardBody>
        <div className="grid gap-3">
          <Field label="Cím">
            <TextInput
              value={draft.title}
              onChange={(e) => onChange({ title: e.target.value })}
              placeholder="pl. Ügyfél azonosítása"
              disabled={disabled || draft.saving}
            />
          </Field>

          <Field label="Leírás (opcionális)">
            <TextArea
              value={draft.description}
              onChange={(e) => onChange({ description: e.target.value })}
              placeholder="pl. Mit kell ellenőrizni, mire figyeljünk"
              disabled={disabled || draft.saving}
            />
          </Field>
        </div>
      </CardBody>
    </Card>
  );
}

export type PlaybookEditorDrawerProps = {
  open: boolean;
  onClose: () => void;

  /** UUID / key (amit a playbooksApi.detail elfogad) */
  playbookRef: string;

  /** Drawer title override (optional) */
  title?: string;

  /** After any successful save (meta/step/reorder/create/delete), call this */
  onAfterSave?: () => Promise<void> | void;

  /** Optional: show a "Open full editor" CTA */
  openFullEditorHref?: string;
};

export default function PlaybookEditorDrawer({
  open,
  onClose,
  playbookRef,
  title = "Playbook szerkesztése",
  onAfterSave,
  openFullEditorHref,
}: PlaybookEditorDrawerProps) {
  const [loading, setLoading] = useState(false);

  // playbook meta drafts
  const [pbName, setPbName] = useState("");
  const [pbDesc, setPbDesc] = useState("");
  const [pbDefaultCaseType, setPbDefaultCaseType] = useState("");
  const [pbDefaultFlowType, setPbDefaultFlowType] = useState<FlowType>("");
  const [pbIsActive, setPbIsActive] = useState(true);
  const [savingMeta, setSavingMeta] = useState(false);

  // steps
  const [steps, setSteps] = useState<any[]>([]);
  const [drafts, setDrafts] = useState<Record<string, StepDraft>>({});
  const bulkBusyRef = useRef(false);

  // create step
  const [newTitle, setNewTitle] = useState("");
  const [newDesc, setNewDesc] = useState("");
  const [creatingStep, setCreatingStep] = useState(false);

  const sensors = useSensors(
    useSensor(PointerSensor, { activationConstraint: { distance: 6 } }),
    useSensor(KeyboardSensor, { coordinateGetter: sortableKeyboardCoordinates })
  );

  const sortedSteps = useMemo(() => {
    return [...steps].sort((a, b) => {
      const d = (a.sort_order ?? 0) - (b.sort_order ?? 0);
      if (d !== 0) return d;
      return String(a.title ?? "").localeCompare(String(b.title ?? ""));
    });
  }, [steps]);

  function patchDraft(stepId: string, patch: Partial<StepDraft>) {
    setDrafts((prev) => ({
      ...prev,
      [stepId]: {
        ...(prev[stepId] ?? { title: "", description: "" }),
        ...patch,
      },
    }));
  }

  function resetDraft(step: any) {
    setDrafts((prev) => ({
      ...prev,
      [step.id]: {
        title: step.title ?? "",
        description: step.description ?? "",
      },
    }));
  }

  async function load() {
    if (!playbookRef) return;

    setLoading(true);
    try {
      const r = await ApiCall.playbooks.detail(playbookRef);
      const pb = r.data.playbook;
      const st = r.data.steps ?? [];

      setPbName(pb.name ?? "");
      setPbDesc(pb.description ?? "");
      setPbDefaultCaseType(pb.default_case_type ?? "");
      setPbDefaultFlowType((pb.default_flow_type ?? "") as FlowType);
      setPbIsActive(!!pb.is_active);

      setSteps(st);

      const d: Record<string, StepDraft> = {};
      for (const s of st)
        d[s.id] = { title: s.title ?? "", description: s.description ?? "" };
      setDrafts(d);
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? "Playbook betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  }

  // load when opening (and when playbookRef changes)
  React.useEffect(() => {
    if (!open) return;
    void load();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [open, playbookRef]);

  async function afterSave() {
    if (onAfterSave) await onAfterSave();
  }

  async function saveMeta() {
    if (!playbookRef) return;
    if (!pbName.trim()) {
      toast.error("A playbook neve kötelező");
      return;
    }

    setSavingMeta(true);
    try {
      await ApiCall.playbooks.update(playbookRef, {
        name: pbName.trim(),
        description: pbDesc.trim() ? pbDesc.trim() : null,
        defaultCaseType: pbDefaultCaseType.trim()
          ? pbDefaultCaseType.trim()
          : null,
        defaultFlowType: pbDefaultFlowType ? pbDefaultFlowType : null,
        isActive: pbIsActive,
      } as any);

      toast.success("Playbook mentve");
      await load();
      await afterSave();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Mentés sikertelen");
    } finally {
      setSavingMeta(false);
    }
  }

  async function createStep() {
    if (!playbookRef) return;
    if (!newTitle.trim()) return;

    setCreatingStep(true);
    try {
      const sortOrder = computeSortOrder(sortedSteps.length);
      const r = await ApiCall.playbooks.createStep(playbookRef, {
        sortOrder,
        title: newTitle.trim(),
        description: newDesc.trim() ? newDesc.trim() : null,
      } as any);

      const created = r.data.step;
      setSteps((prev) => [...prev, created]);
      setDrafts((prev) => ({
        ...prev,
        [created.id]: {
          title: created.title ?? "",
          description: created.description ?? "",
        },
      }));

      setNewTitle("");
      setNewDesc("");
      toast.success("Lépés hozzáadva");
      await afterSave();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Lépés létrehozása sikertelen");
    } finally {
      setCreatingStep(false);
    }
  }

  async function saveStep(step: any) {
    if (!playbookRef) return;
    const d = drafts[step.id];
    if (!d) return;

    const title = normalizeTitle(d.title);
    const description = normalizeDesc(d.description);

    const dirty =
      title !== normalizeTitle(step.title) ||
      description !== normalizeDesc(step.description);

    if (!dirty) return;
    if (!title) {
      toast.error("A lépés címe kötelező");
      return;
    }

    patchDraft(step.id, { saving: true });
    try {
      await ApiCall.playbooks.updateStep(playbookRef, step.id, {
        title,
        description: description ? description : null,
      } as any);

      setSteps((prev) =>
        prev.map((s) =>
          s.id === step.id
            ? { ...s, title, description: description ? description : null }
            : s
        )
      );

      toast.success("Lépés mentve");
      await afterSave();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Lépés mentése sikertelen");
    } finally {
      patchDraft(step.id, { saving: false });
    }
  }

  async function deleteStep(stepId: string) {
    if (!playbookRef) return;
    if (!confirm("Biztos törlöd ezt a lépést?")) return;

    try {
      await ApiCall.playbooks.deleteStep(playbookRef, stepId);
      setSteps((prev) => prev.filter((s) => s.id !== stepId));
      setDrafts((prev) => {
        const copy = { ...prev };
        delete copy[stepId];
        return copy;
      });

      toast.success("Lépés törölve");
      await afterSave();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Törlés sikertelen");
    }
  }

  async function applyReorderAndPersist(next: any[]) {
    if (!playbookRef) return;

    const updates: Array<{ id: string; sortOrder: number }> = [];
    const nextWithOrders = next.map((s, idx) => {
      const sortOrder = computeSortOrder(idx);
      if (s.sort_order !== sortOrder) updates.push({ id: s.id, sortOrder });
      return { ...s, sort_order: sortOrder };
    });

    setSteps(nextWithOrders);
    if (!updates.length) return;

    bulkBusyRef.current = true;
    try {
      await Promise.all(
        updates.map((u) =>
          ApiCall.playbooks.updateStep(playbookRef, u.id, {
            sortOrder: u.sortOrder,
          } as any)
        )
      );
      toast.success("Sorrend mentve");
      await afterSave();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Sorrend mentése sikertelen");
      await load();
    } finally {
      bulkBusyRef.current = false;
    }
  }

  async function onDragEnd(e: DragEndEvent) {
    const { active, over } = e;
    if (!over) return;
    if (active.id === over.id) return;

    const from = sortedSteps.findIndex((s) => s.id === active.id);
    const to = sortedSteps.findIndex((s) => s.id === over.id);
    if (from < 0 || to < 0) return;

    const next = arrayMove(sortedSteps, from, to);
    await applyReorderAndPersist(next);
  }

  return (
    <Drawer open={open} onClose={onClose} title={title} widthClass="max-w-4xl">
      {loading ? (
        <div className="text-sm opacity-70">Betöltés…</div>
      ) : (
        <div className="grid gap-3">
          <Card>
            <CardHeader>
              <div>
                <CardTitle>Playbook beállítások</CardTitle>
                <CardMeta>Alapértékek + státusz</CardMeta>
              </div>
              <div className="flex gap-2 flex-wrap">
                <Chip tone={pbIsActive ? "good" : "warn"}>
                  {pbIsActive ? "active" : "inactive"}
                </Chip>
              </div>
            </CardHeader>

            <CardBody>
              <div className="grid gap-3">
                <div className="grid gap-3 md:grid-cols-2">
                  <Field label="Név">
                    <TextInput
                      value={pbName}
                      onChange={(e) => setPbName(e.target.value)}
                    />
                  </Field>

                  <Field label="Aktív?">
                    <Select
                      value={pbIsActive ? "1" : "0"}
                      onChange={(e) => setPbIsActive(e.target.value === "1")}>
                      <option value="1">Igen</option>
                      <option value="0">Nem</option>
                    </Select>
                  </Field>
                </div>

                <Field label="Leírás">
                  <TextArea
                    value={pbDesc}
                    onChange={(e) => setPbDesc(e.target.value)}
                  />
                </Field>

                <div className="grid gap-3 md:grid-cols-2">
                  <Field label="Default case_type (opcionális)">
                    <TextInput
                      value={pbDefaultCaseType}
                      onChange={(e) => setPbDefaultCaseType(e.target.value)}
                      placeholder="pl. panasz"
                    />
                  </Field>

                  <Field label="Default flow_type (opcionális)">
                    <Select
                      value={pbDefaultFlowType}
                      onChange={(e) =>
                        setPbDefaultFlowType(e.target.value as any)
                      }>
                      <option value="">(nincs)</option>
                      <option value="projekt">Projekt</option>
                      <option value="folyamatos">Folyamatos</option>
                    </Select>
                  </Field>
                </div>

                <ToolbarRow>
                  <Button
                    onClick={() => void saveMeta()}
                    disabled={savingMeta || !pbName.trim()}>
                    {savingMeta ? "Mentés…" : "Playbook mentése"}
                  </Button>

                  {openFullEditorHref ? (
                    <SecondaryButton
                      onClick={() =>
                        (window.location.href = openFullEditorHref)
                      }
                      disabled={savingMeta}>
                      Megnyitás teljes szerkesztőben
                    </SecondaryButton>
                  ) : null}
                </ToolbarRow>
              </div>
            </CardBody>
          </Card>

          <Card>
            <CardHeader>
              <div>
                <CardTitle>Javasolt lépések</CardTitle>
                <CardMeta>
                  Húzd a ⠿ ikonnal a sorrendezéshez • lépésenként menthető
                </CardMeta>
              </div>
            </CardHeader>

            <CardBody>
              <div className="grid gap-3">
                <div className="grid gap-3 md:grid-cols-2">
                  <Field label="Új lépés címe">
                    <TextInput
                      value={newTitle}
                      onChange={(e) => setNewTitle(e.target.value)}
                    />
                  </Field>
                  <Field label="Új lépés leírása (opcionális)">
                    <TextInput
                      value={newDesc}
                      onChange={(e) => setNewDesc(e.target.value)}
                    />
                  </Field>
                </div>

                <ToolbarRow>
                  <Button
                    onClick={() => void createStep()}
                    disabled={creatingStep || !newTitle.trim()}>
                    {creatingStep ? "..." : "Lépés hozzáadása"}
                  </Button>
                </ToolbarRow>

                {sortedSteps.length ? (
                  <DndContext
                    sensors={sensors}
                    collisionDetection={closestCenter}
                    onDragEnd={(e) => void onDragEnd(e)}>
                    <SortableContext
                      items={sortedSteps.map((s) => s.id)}
                      strategy={verticalListSortingStrategy}>
                      <div className="grid gap-2">
                        {sortedSteps.map((s) => (
                          <StepRow
                            key={s.id}
                            step={s}
                            draft={
                              drafts[s.id] ?? {
                                title: s.title ?? "",
                                description: s.description ?? "",
                              }
                            }
                            onChange={(p) => patchDraft(s.id, p)}
                            onReset={() => resetDraft(s)}
                            onSave={() => void saveStep(s)}
                            onDelete={() => void deleteStep(s.id)}
                            disabled={bulkBusyRef.current}
                          />
                        ))}
                      </div>
                    </SortableContext>
                  </DndContext>
                ) : (
                  <EmptyState
                    title="Nincs lépés"
                    hint="Adj hozzá lépéseket, és húzással állítsd be a javasolt sorrendet."
                  />
                )}
              </div>
            </CardBody>
          </Card>

          <ToolbarRow>
            <SecondaryButton onClick={onClose}>Bezár</SecondaryButton>
          </ToolbarRow>
        </div>
      )}
    </Drawer>
  );
}
