"use client";
import React, {
  useState,
  useRef,
  useEffect,
  useContext,
  useCallback,
} from "react";
import Image from "next/image";
// import MediaPreview from "../MediaObject/MediaPreview";
import Loader from "@/components/shared/loader/global-loading";
import icons from "@/assets/icons/icons";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";
import useDynamicManager from "@/store/managers/useDynamicManager";

const UploadCategories = () => {
  const [selectedFiles, setSelectedFiles] = useState([]);
  const [isDragging, setIsDragging] = useState(false);
  const [dragAreaStyle, setDragAreaStyle] = useState({});
  const [filePreviews, setFilePreviews] = useState([]);
  const [status, setStatus] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const { state, clearLogs } = useDynamicManager("uploadLog");
  const [helperOpen, setHelperOpen] = useState(false);
  useEffect(() => {
    // Reset all state on initial mount
    setSelectedFiles([]);
    setFilePreviews([]);
    clearLogs();
    setStatus("");
  }, []);

  useEffect(() => {
    return () => {
      filePreviews.forEach((fp) => URL.revokeObjectURL(fp.preview));
    };
  }, [filePreviews]);

  useEffect(() => {
    setDragAreaStyle({
      border: isDragging ? "2px dashed rgb(156, 163, 175)" : "none",
      backgroundColor: isDragging ? "rgb(147, 197, 253)" : "lightgray",
    });
  }, [isDragging]);

  useEffect(() => {
    const handleDrop = (e) => {
      e.preventDefault();
      clearLogs();
      const files = Array.from(e.dataTransfer.files);
      setSelectedFiles(files);
      setFilePreviews(
        files.map((file) => {
          const item = { preview: URL.createObjectURL(file), data: file };
          return item;
        })
      );
      setIsDragging(false);
    };

    const handleDragOver = (e) => {
      e.preventDefault();
      setIsDragging(true);
    };
    const handleDragLeave = (e) => {
      e.preventDefault();
      setIsDragging(false);
    };
    document.addEventListener("drop", handleDrop);
    document.addEventListener("dragover", handleDragOver);
    document.addEventListener("dragleave", handleDragLeave);

    return () => {
      document.removeEventListener("drop", handleDrop);
      document.removeEventListener("dragover", handleDragOver);
      document.removeEventListener("dragleave", handleDragLeave);
    };
  }, []);

  const handleSelect = (e) => {
    clearLogs();
    const files = Array.from(e.target.files);
    setSelectedFiles(files);
    setFilePreviews(
      files.map((file) => {
        const item = { preview: URL.createObjectURL(file), data: file };
        return item;
      })
    );
  };

  const handleRemove = (index) => {
    const newFiles = [...selectedFiles];
    newFiles.splice(index, 1);
    setSelectedFiles(newFiles);
    setFilePreviews(
      newFiles.map((file) => {
        const item = { preview: URL.createObjectURL(file), data: file };
        return item;
      })
    );
  };
  const handleSubmit = useCallback(
    async (e) => {
      e.preventDefault();
      // setIsLoading(true);

      try {
        const formData = new FormData();

        for (let i = 0; i < selectedFiles.length; i++) {
          formData.append("file", selectedFiles[i]);
        }

        const response = await ApiCall.productCategories.upload(formData); ////!!!!!!!!!!

        if (response.data.error) {
          toast.error(response.data.error, { removeDelay: 15000 });
          return;
        }

        toast.success("✅ Sikeres feltöltés");
        setSelectedFiles([]);
        setFilePreviews([]);
        setStatus("Sikeres feltöltés");
      } catch (error) {
        if (error.response) {
          const serverMsg =
            error.response.data?.message || "Ismeretlen szerverhiba";
          const missingSheets = error.response.data?.missingSheets;
          const missingHeaders = error.response.data?.missingHeaders;

          const sheetInfo = missingSheets?.length
            ? `Hiányzó munkalap(ok): ${missingSheets.join(", ")}`
            : "";

          const headerInfo = missingHeaders?.length
            ? `Hiányzó fejléc(ek): ${missingHeaders
                .map((h) => `${h.sheet}: ${h.headers.join(", ")}`)
                .join(" | ")}`
            : "";

          const fullMessage = [serverMsg, sheetInfo, headerInfo]
            .filter(Boolean)
            .join(" • ");

          toast.error(`❌ ${fullMessage}`, {
            id: "socket-status",
            duration: 15000,
          });
        } else if (error.request) {
          toast.error("❌ Nem sikerült kapcsolatot létesíteni a szerverrel.", {
            id: "socket-status",
            duration: 15000,
          });
        } else {
          toast.error(`Hiba történt: ${error.message}`, {
            id: "socket-status",
            duration: 15000,
          });
        }
      }
    },
    [selectedFiles]
  );

  return (
    <div className="w-full">
      {isLoading && <Loader />}
      <h1 className="pb-4">Kategória feltöltés</h1>
      <div
        onClick={() => setHelperOpen(!helperOpen)}
        className="bg-gray-200 mb-4 p-2">
        <h2 className="flex flex-row w-max cursor-pointer justify-start items-center font-semibold">
          Feltöltési segédlet{" "}
          <span className="w-[1rem] h-[1rem] flex mx-4">
            {helperOpen ? icons.chevronUp : icons.chevronDown}
          </span>
        </h2>
        <ul className={`${helperOpen ? "block" : "hidden"} px-4 py-2`}>
          <li>
            - Cím oszlopban szereplő kategóriák neve lesz a létező kategóriák új
            neve
          </li>
          <li>
            - SEO Cím oszlopban szereplő szöveg kerül a kategória SEO címéhez
          </li>
          <li>- SEO Cím Slug automatikusan generálódik - NEM módosítandó</li>
          <li>
            - Meta leírás - az itt megadott szöveg kerül be az adott
            kategóriához a fejléc elembe SEO miatt - max. 150 karakter.
          </li>
          <li>
            - Ha az itt szereplő érték nem: Aktív, a feltöltés után a kategória
            inaktív lesz - nem szerepel az oldalon (az alá rendelt termékek sem)
          </li>
        </ul>
      </div>

      <div
        className={`w-auto bg-gray-100 text-blue-600 h-40 flex flex-row justify-center items-center`}
        style={dragAreaStyle}
        onDrop={(e) => {
          e.preventDefault();
          const files = Array.from(e.dataTransfer.files);
          setSelectedFiles(files);
          setFilePreviews(
            files.map((file) => {
              const item = { preview: URL.createObjectURL(file), data: file };
              return item;
            })
          );
          setIsDragging(false);
        }}
        onDragOver={(e) => {
          e.preventDefault();
          setIsDragging(true);
        }}
        onDragLeave={(e) => {
          e.preventDefault();
          setIsDragging(false);
        }}>
        <div className="flex flex-col justify-center items-center">
          <p className="font-bold text-xl">
            Húzza ide a kiválasztott fájlokat a feltöltéshez vagy
          </p>
          <input id="file-input" type="file" onChange={handleSelect} />
        </div>
      </div>

      {selectedFiles.length > 0 && (
        <div className="flex flex-row flex-wrap justify-even items-start my-4 w-full">
          {filePreviews?.map((file, index) => {
            switch (file?.data?.type) {
              case "application/pdf":
                return (
                  <div
                    className={`flex flex-col mr-2 mb-2 items-center border-1 w-1/6 mx-2`}
                    key={index}>
                    pdf
                    {/* <MediaPreview
                      data={file?.data}
                      className={"w-full"}
                      id={file?.data?.name}
                      preview={file?.preview}
                    /> */}
                    <button
                      onClick={() => handleRemove(index)}
                      className="text-red-600 my-4">
                      Eltávolítás
                    </button>
                  </div>
                );
              case "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet":
                return (
                  <div
                    className={`flex flex-col mr-2 mb-2 items-center border-1 w-1/6 mx-2`}
                    key={index}>
                    <span className="flex size-60">{icons.excel}</span>
                    <p>{file?.data?.name}</p>
                    <button
                      onClick={() => handleRemove(index)}
                      className="text-red-600 my-4">
                      Eltávolítás
                    </button>
                  </div>
                );
              default:
                return (
                  <div key={index} className="w-1/5 mx-2 ">
                    <Image
                      src={file?.preview}
                      width={250}
                      height={250}
                      className="w-full h-full object-contain flex flex-col justify-center items-start bg-gray-100"
                      style={{ maxHeight: "250px", height: "250px" }}
                      alt={file.data.name ?? ""}
                    />
                    <p>{file?.data?.name}</p>
                    <button
                      onClick={() => handleRemove(index)}
                      className="text-red-600">
                      Eltávolítás
                    </button>
                  </div>
                );
            }
          })}
        </div>
      )}
      {status && <h4 className="text-green-500/75 font-bold my-4">{status}</h4>}
      <button
        className="rounded bg-blue-500/75 hover:bg-blue-500 text-white px-4 py-2 my-4 sticky bottom-0 w-auto"
        disabled={selectedFiles.length === 0}
        onClick={(e) => handleSubmit(e)}>
        Feltöltés
      </button>
    </div>
  );
};

export default UploadCategories;
