import React, { useEffect, useState, useCallback } from "react";
import Input from "@/components/shared/input/Input";
import {
  VALIDATOR_REQUIRE,
  VALIDATOR_MINLENGTH,
  VALIDATOR_MAXLENGTH,
  VALIDATOR_NUMBER,
} from "@/utils/validators";
import Select from "react-select";
import { useForm } from "@/hooks/custom-form-hook";
import SuneditorComponent from "@/components/shared/texteditor/SuneditorComponent";
import useProductGroupManager from "@/store/managers/useProductGroupManager";
import ApiCall from "@/api/api-calls";
import icons from "@/assets/icons/icons";
import ImageWithFallback from "@/components/shared/image/ImageWithFallback";
import MediaGalleryModal from "@/components/media-store/GalleryModal/MediaGalleryModal";
import { toast } from "react-hot-toast";

const ProductGroup = ({ product, onProductGroupUpdated }) => {
  const [formState, handleInput] = useForm([], true);
  const [productData, setProductData] = useState(null);
  const [productProperties, setProductProperties] = useState(null);
  const {
    state,
    resetFeaturedImageState,
    setFeaturedImage,
    loadProductGroupInitialData,
    setProductGroupPriceType,
    setProductGroupCategory,
    setProductGroupFooterCategories,
    setProductGroupTags,
    setProductGroupColors,
    setProductGroupSizes,
    setProductGroupMark,
    setProductGroupConnectedProducts,
  } = useProductGroupManager();
  const [textContent, setTextContent] = useState(null);
  const [mediaStoreOpen, setMediaStoreOpen] = useState(false);
  const [statePopulated, setStatePopulated] = useState(false);

  const [checkboxes] = useState([
    { title: "Publikus", name: "isActive", checked: true },
    { title: "Új termék", name: "isLatest", checked: false },
    { title: "Kiemelt termék", name: "isEmphasized", checked: false },
  ]);

  useEffect(() => {
    setProductData(product?.data?.data);
    setProductProperties(product?.data);
    setTextContent(product?.data?.data?.productDescription);
  }, [product]);

  useEffect(() => {
    const ret = {
      data: productData,
      productProperties: productProperties,
    };
    loadProductGroupInitialData(ret);
    setStatePopulated(true);
  }, [productData, productProperties]);

  useEffect(() => {
    const featured = Array.isArray(productData?.productFeaturedImage)
      ? productData?.productFeaturedImage[0]
      : productData?.productFeaturedImage;
    if (featured) {
      setFeaturedImage({ image: featured });
    }
  }, [productData]);

  const handleSetProductGroupFeaturedImage = useCallback((image) => {
    setFeaturedImage({ image: image });
  }, []);

  const handleRemoveFeaturedImage = useCallback(() => {
    resetFeaturedImageState();
  }, []);

  const handleUpdateProductGroup = useCallback(() => {
    const ret = {
      ...formState.inputs,
      productDescription: textContent,
      ...state,
    };

    const sendData = async () => {
      const response = await ApiCall.productGroups.postEdit(
        productData._id,
        ret
      );

      if (response.status === 200) {
        toast.success(response.data.message);
        onProductGroupUpdated?.();
      } else {
        toast.error(response.data.message);
      }
    };

    sendData();
  }, [formState, textContent, state, productData, onProductGroupUpdated]);

  return (
    <div className="w-full grid grid-cols-1 md:grid-cols-4 gap-4">
      {productData && productProperties && statePopulated && (
        <>
          {/* left */}
          <div className="col-span-3">
            {/* Termék név */}
            <div className="form-group flex flex-col w-full">
              <Input
                element="input"
                type="text"
                name="productGroupTitle"
                id="productGroupTitle"
                validators={[
                  VALIDATOR_REQUIRE(),
                  VALIDATOR_MINLENGTH(2),
                  VALIDATOR_MAXLENGTH(254),
                ]}
                onInput={handleInput}
                errorText={"Nem lehet üres"}
                initialValid={true}
                placeholder="Termék csoport név"
                inputcls="outline-blue-500"
                value={productData?.productGroupTitle?.value}
              />
            </div>
            {/* Termék link cím */}
            <div className="form-group flex flex-col my-2 w-full">
              <Input
                element="input"
                type="text"
                name="productGroupTitleLink"
                id="productGroupTitleLink"
                validators={[
                  VALIDATOR_REQUIRE(),
                  VALIDATOR_MINLENGTH(2),
                  VALIDATOR_MAXLENGTH(254),
                ]}
                onInput={handleInput}
                errorText={"Nem lehet üres"}
                initialValid={true}
                placeholder="Termék csoport link cím"
                inputcls="outline-blue-500"
                value={productData?.productGroupTitleLink?.value}
              />
            </div>
            {/* Termék url */}
            <div className="form-group flex flex-col my-2 w-full">
              <Input
                element="input"
                type="text"
                name="productGroupSlug"
                id="productGroupSlug"
                onInput={handleInput}
                initialValid={true}
                placeholder="Termék url"
                inputcls="outline-blue-500"
                value={productData?.productGroupSlug?.value}
                readOnly={true}
              />
            </div>
            <div className="w-full grid grid-cols-1 md:grid-cols-2 gap-2">
              {/* Termék ár típusa*/}
              <div className="form-group flex flex-col my-2 w-full justify-center">
                <label
                  htmlFor="productPriceType"
                  className="font-bold text-xs text-gray-400 text-left">
                  Válasszon termék ár típust
                </label>
                <Select
                  options={productProperties?.priceTypeList}
                  className="w-full"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon termék ár típust"}
                  backspaceRemovesValue={false}
                  name={"productPriceType"}
                  id={"productPriceType"}
                  defaultValue={
                    productProperties?.currentSelectedPriceTypeElement
                  } // default selected
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    setProductGroupPriceType({ priceType: val });
                  }}
                  required
                />
              </div>
              {/* Termék webshop ár */}
              <div className="form-group flex flex-col my-2 w-full justify-center items-center">
                <Input
                  element="input"
                  type="number"
                  name="productPieceWebshopPrice"
                  id="productPieceWebshopPrice"
                  validators={[
                    VALIDATOR_REQUIRE(),
                    VALIDATOR_NUMBER(),
                    VALIDATOR_MINLENGTH(1),
                    VALIDATOR_MAXLENGTH(254),
                  ]}
                  onInput={handleInput}
                  errorText={"Nem lehet üres. Csak számot adhat meg."}
                  initialValid={true}
                  placeholder={`Termék webshop ár / db (${state.productPriceType})`}
                  inputcls="outline-blue-500"
                  style={{ width: "100%" }}
                  value={productData?.productPieceWebshopPrice?.value}
                />
              </div>
            </div>
            <div className="w-full grid grid-cols-1 md:grid-cols-2 gap-2">
              {/* Termék kategória */}
              <div className="form-group flex flex-col my-2 max-lg:w-full w-full">
                <label
                  htmlFor="productGroupCategory"
                  className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                  Válasszon termék kategóriát
                </label>
                <Select
                  options={productProperties?.productCategoriesList}
                  className="w-full"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon termék kategóriát"}
                  backspaceRemovesValue={false}
                  name={"productGroupCategory"}
                  id={"productGroupCategory"}
                  isMulti
                  defaultValue={
                    productProperties?.selectedProductCategoriesList
                  } // default selected
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    setProductGroupCategory({ category: val });
                  }}
                  required
                />
              </div>
              {/* Termék lábléc kategória */}
              <div className="form-group flex flex-col my-2 max-lg:w-full w-full">
                <label
                  htmlFor="productGroupFooterCategory"
                  className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                  Válasszon termék lábléc kategóriát
                </label>
                <Select
                  options={productProperties?.footerCategoriesList}
                  className="w-full"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon termék lábléc kategóriát"}
                  backspaceRemovesValue={false}
                  name={"productGroupFooterCategory"}
                  id={"productGroupFooterCategory"}
                  isMulti
                  defaultValue={productProperties?.selectedFooterCategoriesList} // default selected
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    setProductGroupFooterCategories({ footerCategories: val });
                  }}
                  required
                />
              </div>
            </div>
            <div className="w-full grid grid-cols-1 md:grid-cols-2 gap-2">
              {/* Termék mark */}
              <div className="form-group flex flex-col my-2 max-lg:w-full w-full">
                <label
                  htmlFor="productGroupMark"
                  className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                  Válasszon termék márkát
                </label>
                <Select
                  options={productProperties?.productMarksList}
                  className="w-full"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon termék márkát"}
                  backspaceRemovesValue={false}
                  name={"productGroupMark"}
                  id={"productGroupMark"}
                  defaultValue={productProperties?.selectedMarkList} // default selected
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    setProductGroupMark({ mark: val });
                  }}
                  required
                />
              </div>
              {/* Termék címke */}
              <div className="form-group flex flex-col my-2 max-lg:w-full w-full ">
                <label
                  htmlFor="productTagList"
                  className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                  Válasszon termék címkét
                </label>
                <Select
                  options={productProperties?.productTagList}
                  className="w-full"
                  menuPosition={"fixed"}
                  placeholder={"Válasszon termék címkét"}
                  backspaceRemovesValue={false}
                  name={"productTagList"}
                  isMulti
                  defaultValue={productProperties?.selectedProductTagList} // default selected
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    setProductGroupTags({ tags: val });
                  }}
                  required
                />
              </div>
            </div>
            <div className="w-full grid grid-cols-1 md:grid-cols-2 gap-2">
              {/* termék méretek */}
              <div className="form-group flex flex-col my-2 max-lg:w-full w-full ">
                <label
                  htmlFor="productSizesList"
                  className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                  Válasszon termék méreteket
                </label>
                <Select
                  options={productProperties?.productSizesList}
                  className="w-full"
                  menuPosition={"fixed"}
                  placeholder={" Válasszon termék méreteket"}
                  backspaceRemovesValue={false}
                  name={"productSizesList"}
                  id={"productSizesList"}
                  isMulti
                  defaultValue={productProperties?.selectedProductSizesList} // default selected
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    setProductGroupSizes({ sizes: val || [] });
                  }}
                  required
                />
              </div>
              {/* termék színek */}
              <div className="form-group flex flex-col my-2 max-lg:w-full w-full">
                <label
                  htmlFor="productColorsList"
                  className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                  Válasszon termék színeket
                </label>
                <Select
                  options={productProperties?.productColorsList}
                  className="w-full"
                  menuPosition={"fixed"}
                  placeholder={" Válasszon termék színeket"}
                  backspaceRemovesValue={false}
                  name={"productColorsList"}
                  id={"productColorsList"}
                  isMulti
                  defaultValue={productProperties?.selectedProductColorsList} // default selected
                  onChange={(val, e) => {
                    // second parameter shows action triggered
                    setProductGroupColors({ colors: val || [] });
                  }}
                  required
                />
              </div>
            </div>
            {/* Kapcsolódó termékek */}
            <div className="form-group flex flex-col my-2 max-lg:w-full w-1/2 pr-2">
              <label
                htmlFor="connectedProductsList"
                className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                Válasszon kapcsolódó terméket
              </label>
              <Select
                options={productProperties?.connectedProductsList}
                className="w-full"
                menuPosition={"fixed"}
                placeholder={"Válasszon kapcsolódó terméket"}
                backspaceRemovesValue={false}
                name={"connectedProductsList"}
                isMulti
                defaultValue={productProperties?.selectedConnectedProductsList} // default selected
                onChange={(val, e) => {
                  // second parameter shows action triggered

                  setProductGroupConnectedProducts({ connectedProducts: val });
                }}
                required
              />
            </div>
            {/* Termék leírás */}
            <div className="flex flex-col my-2 w-full z-0">
              <label
                htmlFor={"productDescription"}
                className="w-1/4  text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                <p className="w-full font-bold text-xs">Termék leírás: </p>
              </label>
              <SuneditorComponent
                lang={"en"}
                name="productDescription"
                id="productDescription"
                className="w-full h-auto !z-0"
                defaultValue={textContent}
                placeholder={"Termék leírás"}
                onChange={(value) => setTextContent(value)}
                options={{
                  buttonList: [
                    ["font", "fontSize", "formatBlock"],
                    [
                      "bold",
                      "underline",
                      "italic",
                      "strike",
                      "subscript",
                      "superscript",
                    ],
                    ["align", "list", "table"],
                    ["fontColor", "hiliteColor"],
                    ["outdent", "indent"],
                    ["undo", "redo"],
                    ["removeFormat"],
                    ["link", "image"],
                    ["preview", "print"],
                    ["fullScreen", "showBlocks", "codeView"],
                  ],
                  tableWidth: "100%",
                  tableStyles: {
                    border: "1px solid black",
                  },
                }}
              />
            </div>
          </div>
          {/* right */}
          <div className="col-span-1">
            <div className="w-full flex flex-col">
              {/* Featured Image */}
              <div className="w-full h-80 border-2 border-dashed relative mb-4">
                {state.featuredImage?.length > 0 && (
                  <ImageWithFallback
                    src={ApiCall.mediaStore.getResized(
                      "/4/4/1",
                      state.featuredImage[0].originalname
                    )}
                    alt="Featured"
                    fill
                    className="object-cover"
                  />
                )}
                <span
                  onClick={() => setMediaStoreOpen(true)}
                  className="absolute top-2 left-2 w-[3rem] h-[3rem] hover:border-1 bg-white rounded-full flex justify-center items-center cursor-pointer">
                  {icons.change}
                </span>
                <span
                  onClick={handleRemoveFeaturedImage}
                  className="absolute top-2 right-2 w-[2rem] h-[2rem] !fill-red !stroke-red rounded-full flex justify-center items-center cursor-pointer">
                  {icons.clear}
                </span>
              </div>
              {/* checkboxes */}
              <div className="w-full flex flex-row flex-wrap justify-center items-center">
                {checkboxes?.map((checkbox, index) => (
                  <div
                    key={checkbox.name}
                    className="form-group w-1/2 flex flex-col my-4 justify-center items-center">
                    <Input
                      element="checkbox"
                      type="checkbox"
                      name={checkbox.name}
                      id={checkbox.name}
                      placeholder={checkbox.title}
                      onInput={handleInput}
                      initialValid={true}
                      className="justify-center items-center"
                      inputcls="outline-blue-500 w-8 h-8"
                      isChecked={productData[checkbox.name]?.value}
                    />
                  </div>
                ))}
              </div>
              <div className="flex flex-row flex-wrap w-full justify-center items-center text-center">
                {/* Áfa érték */}
                <div className="form-group flex flex-col my-2 mx-4 w-full">
                  <Input
                    element="input"
                    type="text"
                    name="productVat"
                    id="productVat"
                    placeholder="Áfa érték %"
                    onInput={handleInput}
                    initialValid={true}
                    inputcls=" outline-blue-500"
                    value={product?.data?.productVat?.value ?? 27}
                    readOnly={
                      formState.inputs &&
                      formState?.inputs?.productPieceNettoBuyPrice?.value > 0
                        ? false
                        : true
                    }
                  />
                </div>
                {/* Termék azonosító */}
                <div className="form-group flex flex-col my-2 mx-4 w-full">
                  <Input
                    element="input"
                    type="text"
                    name="productItemNumberIdentifier"
                    id="productItemNumberIdentifier"
                    placeholder="Termék csoport azonosító"
                    onInput={handleInput}
                    initialValid={true}
                    inputcls="w-full outline-blue-500"
                    value={productData?.productItemNumberIdentifier?.value}
                  />
                </div>
                {/* Termék kedvezmény % */}
                <div className="form-group flex flex-col my-2 mx-4 w-full">
                  <Input
                    element="input"
                    type="number"
                    name="productDiscount"
                    id="productDiscount"
                    validators={[
                      VALIDATOR_NUMBER(),
                      VALIDATOR_MINLENGTH(1),
                      VALIDATOR_MAXLENGTH(254),
                    ]}
                    onInput={handleInput}
                    errorText={"Csak számot adhat meg."}
                    initialValid={true}
                    placeholder="Termék kedvezmény %"
                    inputcls="w-full outline-blue-500"
                    value={productData?.productDiscount?.value}
                  />
                </div>
                {/* Rendelési minimum beállítása - db */}
                <div className="form-group flex flex-col my-2 mx-4 w-full">
                  <Input
                    element="input"
                    type="number"
                    name="minQty"
                    id="minQty"
                    validators={[
                      VALIDATOR_NUMBER(),
                      VALIDATOR_MINLENGTH(1),
                      VALIDATOR_MAXLENGTH(254),
                    ]}
                    onInput={handleInput}
                    errorText={"Csak számot adhat meg."}
                    initialValid={true}
                    placeholder="Rendelési minimum - db"
                    inputcls="w-full outline-blue-500"
                    value={productData?.minQty?.value ?? 0}
                  />
                </div>
              </div>

              <MediaGalleryModal
                isOpen={mediaStoreOpen}
                onClose={() => setMediaStoreOpen(false)}
                action="productGroupFeatured"
                onConfirmSelection={(selectedImage) =>
                  handleSetProductGroupFeaturedImage(selectedImage)
                }
              />
            </div>
          </div>
          <div className="w-full col-span-4">
            {formState.isValid && (
              <button
                onClick={() => handleUpdateProductGroup()}
                className="w-full px-4 py-2 flex flex-row justify-center items-center bg-green-500/75 hover:bg-green-500 text-white">
                Termékcsoport mentése
              </button>
            )}
          </div>
        </>
      )}
    </div>
  );
};

export default ProductGroup;
