import React, { useEffect, useState } from "react";
import Input from "@/components/shared/input/Input";
import {
  VALIDATOR_REQUIRE,
  VALIDATOR_MINLENGTH,
  VALIDATOR_MAXLENGTH,
  VALIDATOR_NUMBER,
} from "@/utils/validators";
import Select from "react-select";
import { useForm } from "@/hooks/custom-form-hook";
import SuneditorComponent from "@/components/shared/texteditor/SuneditorComponent";
import useProductGroupManager from "@/store/managers/useProductGroupManager";

const ProductGroupForm = ({ product, setIsValid }) => {
  const [formState, handleInput] = useForm([], true);
  const [textContent, setTextContent] = useState(
    product?.data?.productDescription
  );
  const {
    state,
    loadProductGroupInitialData,
    setProductGroupPriceType,
    setProductGroupCategory,
    setProductGroupFooterCategories,
    setProductGroupTags,
    setProductGroupColors,
    setProductGroupSizes,
    setProductGroupMark,
    setProductGroupConnectedProducts,
  } = useProductGroupManager();

  useEffect(() => {
    setIsValid(formState?.isValid);
  }, [formState]);

  useEffect(() => {
    loadProductGroupInitialData(product?.data);
  }, [product]);

  return (
    <>
      {product && (
        <>
          {/* Termék név */}
          <div className="form-group flex flex-col w-full">
            <Input
              element="input"
              type="text"
              name="productGroupTitle"
              id="productGroupTitle"
              validators={[
                VALIDATOR_REQUIRE(),
                VALIDATOR_MINLENGTH(2),
                VALIDATOR_MAXLENGTH(254),
              ]}
              onInput={handleInput}
              errorText={"Nem lehet üres"}
              initialValid={true}
              placeholder="Termék csoport név"
              inputcls="outline-blue-500"
              value={product?.data?.productGroupTitle?.value}
            />
          </div>
          {/* Termék link cím */}
          <div className="form-group flex flex-col my-2 w-full">
            <Input
              element="input"
              type="text"
              name="productGroupTitleLink"
              id="productGroupTitleLink"
              validators={[
                VALIDATOR_REQUIRE(),
                VALIDATOR_MINLENGTH(2),
                VALIDATOR_MAXLENGTH(254),
              ]}
              onInput={handleInput}
              errorText={"Nem lehet üres"}
              initialValid={true}
              placeholder="Termék csoport link cím"
              inputcls="outline-blue-500"
              value={product?.data?.productGroupTitleLink?.value}
            />
          </div>
          {/* Termék url */}
          <div className="form-group flex flex-col my-2 w-full">
            <Input
              element="input"
              type="text"
              name="productGroupSlug"
              id="productGroupSlug"
              onInput={handleInput}
              initialValid={true}
              placeholder="Termék url"
              inputcls="outline-blue-500"
              value={product?.data?.productGroupSlug?.value}
              readOnly={true}
            />
          </div>
          <div className="w-full grid grid-cols-1 md:grid-cols-2 gap-2">
            {/* Termék ár típusa*/}
            <div className="form-group flex flex-col my-2 w-full justify-center">
              <label
                htmlFor="productPriceType"
                className="font-bold text-xs text-gray-400 text-left">
                Válasszon termék ár típust
              </label>
              <Select
                options={product?.priceTypeList}
                className="w-full"
                menuPosition={"fixed"}
                placeholder={"Válasszon termék ár típust"}
                backspaceRemovesValue={false}
                name={"productPriceType"}
                id={"productPriceType"}
                defaultValue={product?.priceTypeList?.find(
                  (pt) => pt.value === state?.currentSelectedPriceTypeElement
                )} // default selected
                onChange={(selected, e) => {
                  // second parameter shows action triggered
                  setProductGroupPriceType({ priceType: selected.val });
                }}
                required
              />
            </div>
            {/* Termék webshop ár */}
            <div className="form-group flex flex-col my-2 w-full justify-center items-center">
              <Input
                element="input"
                type="number"
                name="productPieceWebshopPrice"
                id="productPieceWebshopPrice"
                validators={[
                  VALIDATOR_REQUIRE(),
                  VALIDATOR_NUMBER(),
                  VALIDATOR_MINLENGTH(1),
                  VALIDATOR_MAXLENGTH(254),
                ]}
                onInput={handleInput}
                errorText={"Nem lehet üres. Csak számot adhat meg."}
                initialValid={true}
                placeholder="Termék webshop ár / db (EUR)"
                inputcls="outline-blue-500"
                style={{ width: "100%" }}
                value={product?.data?.productPieceWebshopPrice?.value}
              />
            </div>
          </div>
          <div className="w-full grid grid-cols-1 md:grid-cols-2 gap-2">
            {/* Termék kategória */}
            <div className="form-group flex flex-col my-2 max-lg:w-full w-full">
              <label
                htmlFor="productGroupCategory"
                className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                Válasszon termék kategóriát
              </label>
              <Select
                options={product?.productCategoriesList}
                className="w-full"
                menuPosition={"fixed"}
                placeholder={"Válasszon termék kategóriát"}
                backspaceRemovesValue={false}
                name={"productGroupCategory"}
                id={"productGroupCategory"}
                isMulti
                defaultValue={product?.selectedProductCategoriesList} // default selected
                onChange={(val, e) => {
                  // second parameter shows action triggered
                  setProductGroupCategory({ category: val });
                }}
                required
              />
            </div>
            {/* Termék lábléc kategória */}
            <div className="form-group flex flex-col my-2 max-lg:w-full w-full">
              <label
                htmlFor="productGroupFooterCategory"
                className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                Válasszon termék lábléc kategóriát
              </label>
              <Select
                options={product?.footerCategoriesList}
                className="w-full"
                menuPosition={"fixed"}
                placeholder={"Válasszon termék lábléc kategóriát"}
                backspaceRemovesValue={false}
                name={"productGroupFooterCategory"}
                id={"productGroupFooterCategory"}
                isMulti
                defaultValue={product?.selectedFooterCategoriesList} // default selected
                onChange={(val, e) => {
                  // second parameter shows action triggered
                  setProductGroupFooterCategories({ footerCategories: val });
                }}
                required
              />
            </div>
          </div>
          <div className="w-full grid grid-cols-1 md:grid-cols-2 gap-2">
            {/* Termék mark */}
            <div className="form-group flex flex-col my-2 max-lg:w-full w-full">
              <label
                htmlFor="productGroupMark"
                className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                Válasszon termék márkát
              </label>
              <Select
                options={product?.productMarksList}
                className="w-full"
                menuPosition={"fixed"}
                placeholder={"Válasszon termék márkát"}
                backspaceRemovesValue={false}
                name={"productGroupMark"}
                id={"productGroupMark"}
                isMulti
                defaultValue={product?.selectedMarkList} // default selected
                onChange={(val, e) => {
                  // second parameter shows action triggered
                  setProductGroupMark({ mark: val });
                }}
                required
              />
            </div>
            {/* Termék címke */}
            <div className="form-group flex flex-col my-2 max-lg:w-full w-full ">
              <label
                htmlFor="productTagList"
                className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                Válasszon termék címkét
              </label>
              <Select
                options={product?.productTagList}
                className="w-full"
                menuPosition={"fixed"}
                placeholder={"Válasszon termék címkét"}
                backspaceRemovesValue={false}
                name={"productTagList"}
                isMulti
                defaultValue={product?.selectedProductTagList} // default selected
                onChange={(val, e) => {
                  // second parameter shows action triggered
                  setProductGroupTags({ tags: val });
                }}
                required
              />
            </div>
          </div>
          <div className="w-full grid grid-cols-1 md:grid-cols-2 gap-2">
            {/* termék méretek */}
            <div className="form-group flex flex-col my-2 max-lg:w-full w-full ">
              <label
                htmlFor="productSizesList"
                className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                Válasszon termék méreteket
              </label>
              <Select
                options={product?.productSizesList}
                className="w-full"
                menuPosition={"fixed"}
                placeholder={" Válasszon termék méreteket"}
                backspaceRemovesValue={false}
                name={"productSizesList"}
                id={"productSizesList"}
                isMulti
                defaultValue={product?.selectedProductSizesList} // default selected
                onChange={(val, e) => {
                  // second parameter shows action triggered
                  setProductGroupSizes({ sizes: val });
                }}
                required
              />
            </div>
            {/* termék színek */}
            <div className="form-group flex flex-col my-2 max-lg:w-full w-full">
              <label
                htmlFor="productColorsList"
                className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
                Válasszon termék színeket
              </label>
              <Select
                options={product?.productColorsList}
                className="w-full"
                menuPosition={"fixed"}
                placeholder={" Válasszon termék színeket"}
                backspaceRemovesValue={false}
                name={"productColorsList"}
                id={"productColorsList"}
                isMulti
                defaultValue={product?.selectedProductColorsList} // default selected
                onChange={(val, e) => {
                  // second parameter shows action triggered
                  setProductGroupColors({ colors: val });
                }}
                required
              />
            </div>
          </div>
          {/* Kapcsolódó termékek */}
          <div className="form-group flex flex-col my-2 max-lg:w-full w-1/2 pr-2">
            <label
              htmlFor="connectedProductsList"
              className="font-bold text-xs text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
              Válasszon kapcsolódó terméket
            </label>
            <Select
              options={product?.connectedProductsList}
              className="w-full"
              menuPosition={"fixed"}
              placeholder={"Válasszon kapcsolódó terméket"}
              backspaceRemovesValue={false}
              name={"connectedProductsList"}
              isMulti
              defaultValue={product?.selectedConnectedProductsList} // default selected
              onChange={(val, e) => {
                // second parameter shows action triggered

                setProductGroupConnectedProducts({ connectedProducts: val });
              }}
              required
            />
          </div>
          {/* Termék leírás */}
          <div className="form-group flex flex-col my-2 w-full z-0">
            <label
              htmlFor={"productDescription"}
              className="w-1/4  text-gray-400 whitespace-pre flex-flex-col flex-wrap pr-2">
              <p className="w-full font-bold text-xs">Termék leírás: </p>
            </label>
            <SuneditorComponent
              lang={"en"}
              name="productDescription"
              id="productDescription"
              className="w-full h-auto"
              defaultValue={textContent}
              placeholder={"Termék leírás"}
              onChange={(value) => setTextContent(value)}
              options={{
                buttonList: [
                  ["font", "fontSize", "formatBlock"],
                  [
                    "bold",
                    "underline",
                    "italic",
                    "strike",
                    "subscript",
                    "superscript",
                  ],
                  ["align", "list", "table"],
                  ["fontColor", "hiliteColor"],
                  ["outdent", "indent"],
                  ["undo", "redo"],
                  ["removeFormat"],
                  ["link", "image"],
                  ["preview", "print"],
                  ["fullScreen", "showBlocks", "codeView"],
                ],
                tableWidth: "100%",
                tableStyles: {
                  border: "1px solid black",
                },
              }}
            />
          </div>
        </>
      )}
    </>
  );
};

export default ProductGroupForm;
