import React, { useCallback, useEffect, useState } from "react";
import Input from "@/components/shared/input/Input";
import ApiCall from "@/api/api-calls";
import icons from "@/assets/icons/icons";
import ImageWithFallback from "@/components/shared/image/ImageWithFallback";
import useProductGroupManager from "@/store/managers/useProductGroupManager";
import MediaGalleryModal from "@/components/media-store/GalleryModal/MediaGalleryModal";
import { useForm } from "@/hooks/custom-form-hook";
import {
  VALIDATOR_REQUIRE,
  VALIDATOR_MINLENGTH,
  VALIDATOR_MAXLENGTH,
  VALIDATOR_NUMBER,
} from "@/utils/validators";

const ProductGroupSidePanel = ({
  product,
  checkboxes,
  setIsSidepanelValid,
}) => {
  const [formState, handleInput] = useForm([], true);
  const { state, resetFeaturedImageState, setFeaturedImage } =
    useProductGroupManager();
  const [mediaStoreOpen, setMediaStoreOpen] = useState(false);

  useEffect(() => {
    const featured = Array.isArray(product?.data?.productFeaturedImage)
      ? product?.data?.productFeaturedImage[0]
      : product?.data?.productFeaturedImage;
    if (featured) {
      setFeaturedImage({ image: featured });
    }
  }, [product]);

  useEffect(() => {
    setIsSidepanelValid(formState?.isValid);
  }, [formState]);

  const handleSetProductGroupFeaturedImage = useCallback((image) => {
    setFeaturedImage({ image: image });
  }, []);

  const handleRemoveFeaturedImage = useCallback(() => {
    resetFeaturedImageState();
  }, []);

  return (
    <div className="w-full flex flex-col">
      {/* Featured Image */}
      <div className="w-full h-80 border-2 border-dashed relative mb-4">
        {state.featuredImage?.length > 0 && (
          <ImageWithFallback
            src={ApiCall.mediaStore.getResized(
              "/4/4/1",
              state.featuredImage[0].originalname
            )}
            alt="Featured"
            fill
            className="object-cover"
          />
        )}
        <span
          onClick={() => setMediaStoreOpen(true)}
          className="absolute top-2 left-2 w-[2rem] h-[2rem] bg-white rounded-full flex justify-center items-center cursor-pointer">
          {icons.change}
        </span>
        <span
          onClick={handleRemoveFeaturedImage}
          className="absolute top-2 right-2 w-[2rem] h-[2rem] !fill-red !stroke-red rounded-full flex justify-center items-center cursor-pointer">
          {icons.clear}
        </span>
      </div>
      {/* checkboxes */}
      <div className="w-full flex flex-row flex-wrap justify-center items-center">
        {checkboxes?.map((checkbox, index) => (
          <div
            key={checkbox.name}
            className="form-group w-1/2 flex flex-col my-4 justify-center items-center">
            <Input
              element="checkbox"
              type="checkbox"
              name={checkbox.name}
              id={checkbox.name}
              placeholder={checkbox.title}
              onInput={handleInput}
              initialValid={true}
              className="justify-center items-center"
              inputcls="outline-blue-500 w-8 h-8"
              isChecked={product?.data[checkbox.name]?.value}
            />
          </div>
        ))}
      </div>
      <div className="flex flex-row flex-wrap w-full justify-center items-center text-center">
        {/* Áfa érték */}
        <div className="form-group flex flex-col my-2 mx-4 w-full">
          <Input
            element="input"
            type="text"
            name="productVat"
            id="productVat"
            placeholder="Áfa érték %"
            onInput={handleInput}
            initialValid={true}
            inputcls=" outline-blue-500"
            value={product?.data?.productVat?.value ?? 27}
            readOnly={
              formState.inputs &&
              formState?.inputs?.productPieceNettoBuyPrice?.value > 0
                ? false
                : true
            }
          />
        </div>
        {/* Termék azonosító */}
        <div className="form-group flex flex-col my-2 mx-4 w-full">
          <Input
            element="input"
            type="text"
            name="productItemNumberIdentifier"
            id="productItemNumberIdentifier"
            placeholder="Termék csoport azonosító"
            onInput={handleInput}
            initialValid={true}
            inputcls="w-full outline-blue-500"
            value={product?.data?.productItemNumberIdentifier?.value}
          />
        </div>
        {/* Termék kedvezmény % */}
        <div className="form-group flex flex-col my-2 mx-4 w-full">
          <Input
            element="input"
            type="number"
            name="productDiscount"
            id="productDiscount"
            validators={[
              VALIDATOR_NUMBER(),
              VALIDATOR_MINLENGTH(1),
              VALIDATOR_MAXLENGTH(254),
            ]}
            onInput={handleInput}
            errorText={"Csak számot adhat meg."}
            initialValid={true}
            placeholder="Termék kedvezmény %"
            inputcls="w-full outline-blue-500"
            value={product?.data?.productDiscount?.value}
          />
        </div>
        {/* Rendelési minimum beállítása - db */}
        <div className="form-group flex flex-col my-2 mx-4 w-full">
          <Input
            element="input"
            type="number"
            name="minQty"
            id="minQty"
            validators={[
              VALIDATOR_NUMBER(),
              VALIDATOR_MINLENGTH(1),
              VALIDATOR_MAXLENGTH(254),
            ]}
            onInput={handleInput}
            errorText={"Csak számot adhat meg."}
            initialValid={true}
            placeholder="Rendelési minimum - db"
            inputcls="w-full outline-blue-500"
            value={product?.data?.minQty?.value ?? 0}
          />
        </div>
      </div>

      <MediaGalleryModal
        isOpen={mediaStoreOpen}
        onClose={() => setMediaStoreOpen(false)}
        action="productGroupFeatured"
        onConfirmSelection={(selectedImage) =>
          handleSetProductGroupFeaturedImage(selectedImage)
        }
      />
    </div>
  );
};

export default ProductGroupSidePanel;
