// app/admin/product-group/[id]/versions/client.jsx
"use client";

import { useEffect, useState } from "react";
import { Button } from "@mui/material";
import { Card, CardContent } from "@/components/shared/ui/card";
import { formatDistanceToNow } from "date-fns";
import { useModalManager } from "@/hooks/useModalManager";
import Modal from "@/components/shared/modal/Modal";
import DiffViewer from "@/components/shared/ui/diff-viewer";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";

export default function VersionListPage({ id, onProductGroupUpdated }) {
  const { toggleModal, isModalOpen } = useModalManager();
  const [versions, setVersions] = useState([]);
  const [currentData, setCurrentData] = useState(null);
  const [selectedVersion, setSelectedVersion] = useState(null);
  const [loading, setLoading] = useState(false);

  const fetchVersions = async () => {
    try {
      const response = await ApiCall.productGroups.getVersions(id);
      if (response.status === 200) {
        setVersions(response?.data.versions || []);
        toast.success("Verziók betöltve.");
      } else {
        toast.error("Hiba a betöltésnél");
      }
    } catch (err) {
      console.error("Failed to load versions:", err);
    }
  };

  const fetchCurrentData = async () => {
    try {
      const response = await ApiCall.productGroups.getEdit(id);

      if (response.status === 200) {
        setCurrentData(response.data.data || {});
        toast.success("Eredeti termék betöltve.");
      } else {
        toast.error("Hiba a betöltésnél");
      }
    } catch (err) {
      console.error("Failed to load current product group:", err);
    }
  };

  const handleRestore = async (versionId) => {
    setLoading(true);
    const response = await ApiCall.productGroups.restoreVersion(id, versionId);
    if (response.status === 200) {
      await fetchVersions();
      await fetchCurrentData();
      toast.success("Verzió sikeresen visszaállítva!");
      onProductGroupUpdated?.();
    } else {
      toast.success("Verzió visszaállítás sikertelen!");
    }
    setLoading(false);
  };

  const openRestoreModal = (version) => {
    setSelectedVersion(version);
    toggleModal("showVersionsModal");
  };

  const confirmRestore = async () => {
    if (selectedVersion) {
      await handleRestore(selectedVersion._id);
      toggleModal("showVersionsModal");
      setSelectedVersion(null);
    }
  };

  useEffect(() => {
    fetchVersions();
    fetchCurrentData();
  }, []);

  return (
    <div className="p-4 max-w-4xl mx-auto">
      <h1 className="text-2xl font-bold mb-4">Termékcsoport verzió történet</h1>
      <h3 className="pb-2">
        Termék csoportonként összesen 10 verzió készülhet. Ennél több verzió
        esetén a legrégebbi verzió kerül felülírásra.
      </h3>
      {versions && versions?.length === 0 && (
        <>Ebből a csoportból nem áll rendelkezésre másik verzió.</>
      )}
      {versions.map((version) => (
        <Card key={version._id} className="mb-3">
          <CardContent className="flex justify-between items-center p-4">
            <div>
              <div className="font-semibold">
                {version.versionedBy?.username?.value || "Unknown"}{" "}
                <span className="text-sm text-muted-foreground">
                  ({version.versionedBy?._id || "Unknown user"})
                </span>
              </div>
              <div className="text-sm text-muted-foreground">
                <p>{version?.snapshot?.productGroupTitle.value}</p>
              </div>
              <div className="text-sm text-muted-foreground">
                {formatDistanceToNow(new Date(version.versionedAt), {
                  addSuffix: true,
                })}
              </div>
            </div>
            <Button
              disabled={loading}
              onClick={() => openRestoreModal(version)}>
              Visszaállítás
            </Button>
          </CardContent>
        </Card>
      ))}

      <Modal
        title="Verzió visszaállítás ellenőrzése"
        show={isModalOpen("showVersionsModal")}
        onClose={() => toggleModal("showVersionsModal")}
        height="auto">
        <div className="p-4 space-y-4">
          <p>
            Biztosan visszaállítja a kiválasztott verziót? Ez felülírja a
            jelenleg éles változatot.
          </p>

          {selectedVersion?.snapshot && currentData && (
            <div className="border rounded-md p-2 max-h-[400px] overflow-auto">
              <DiffViewer
                oldData={currentData}
                newData={selectedVersion.snapshot}
              />
            </div>
          )}

          <div className="flex justify-end gap-2 mt-4">
            <Button
              variant="outlined"
              onClick={() => toggleModal("showVersionsModal")}>
              Mégse
            </Button>
            <Button
              variant="contained"
              color="primary"
              onClick={confirmRestore}
              disabled={loading}>
              Visszaállítom
            </Button>
          </div>
        </div>
      </Modal>
    </div>
  );
}
