import React, { useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import icons from "@/assets/icons/icons";
import toast from "react-hot-toast";
import Product from "./Product";
import { useExchangeRate } from "@/context/ExchangeRateContext";

const createEmptyProduct = (parentId, productGroup) => ({
  _id: `new-${Date.now()}`,
  parentId,
  isActive: true,
  inStock: false,
  productSize: [],
  productColor: [],
  productTags: [],
  productContent: [],
  productGalleryImages: [],
  productItemNumber: { value: "" },
  productEAN: { value: "" },
  productWeight: { value: 0 },
  productDescription: "",
  productPriceType: { value: productGroup?.productPriceType?.value },
});

const ProductList = ({
  parentId,
  productGroup,
  productIds,
  onProductGroupUpdated,
}) => {
  const [priceTypeList, setPriceTypeList] = useState([]);
  const [productList, setProductList] = useState([]);
  const [productColorsList, setProductColorsList] = useState([]);
  const [productSizesList, setProductSizesList] = useState([]);
  const [productMarksList, setProductMarksList] = useState([]);
  const [productTagList, setProductTagList] = useState([]);
  const [activeTab, setActiveTab] = useState(0);
  const [productGroupOptions, setProductGroupOptions] = useState([]);
  // const { exchangeRate } = useExchangeRate();

  const [checkboxes, setCheckboxes] = useState([
    { title: "Publikus", name: "isActive", checked: true },
    { title: "Raktáron", name: "inStock", checked: true },
    { title: "Új termék", name: "isLatest", checked: false },
    { title: "Kiemelt termék", name: "isEmphasized", checked: false },
  ]);

  useEffect(() => {
    const fetchProductList = async () => {
      const ret = {
        parentId,
        productIds,
      };
      const response = await ApiCall.productGroups.getProductList(ret);
      if (response.status === 200) {
        setPriceTypeList(response?.data.priceTypeList);
        setProductList(response?.data.productList);
        setProductColorsList(response?.data.productColorsList);
        setProductSizesList(response?.data.productSizesList);
        setProductMarksList(response?.data.productMarksList);
        setProductTagList(response?.data.productTagList);
        setProductGroupOptions(response?.data.productGroupList);
        toast.success(
          productGroup?.productGroupTitle?.value + " termékei betöltve"
        );
      }
    };
    fetchProductList();
  }, [parentId, productIds]);

  const toggleTab = (index) => {
    activeTab === index ? setActiveTab(null) : setActiveTab(index);
  };

  const handleRemoveLocalProduct = (productId) => {
    setProductList((prev) => prev.filter((p) => p._id !== productId));
    toast.success("Új termék eltávolítva a listából.");
  };

  return (
    <div className="w-full flex flex-col px-2">
      <button
        onClick={() => {
          const newProduct = createEmptyProduct(parentId, productGroup);
          toast.success(
            <p>
              Új - üres termék - került hozzáadása a{" "}
              <span>
                <b>{productGroup?.productGroupTitle?.value}</b>{" "}
                termékcsoporthoz.
              </span>
            </p>
          );
          setProductList((prev) => [...prev, newProduct]);
        }}
        className="my-4 bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 cursor-pointer">
        ➕ Új termék hozzáadása
      </button>

      {productList &&
        productList.map((product, idx) => {
          return (
            <React.Fragment key={product?._id}>
              <div
                className={`w-full grid grid-cols-2 md:grid-cols-6 justify-center items-center cursor-pointer hover:bg-gray-500 hover:text-white border-b border-gray-700 px-4 py-2 my-1 ${
                  idx % 2 === 0 ? "bg-gray-200" : "bg-gray-100"
                } ${!product.isActive.value ? "bg-red-500 text-white" : ""}`}
                onClick={() => toggleTab(idx)}>
                <div>Cikkszám: {product?.productItemNumber?.value}</div>
                <div>EAN: {product?.productEAN?.value}</div>
                <div>
                  Ár: {product?.productPieceWebshopPrice?.value}{" "}
                  {product?.productPriceType?.value ?? "Nincs ártípus!"}
                </div>
                <div>Aktív: {product?.isActive?.value ? "Igen" : "Nem"}</div>
                <div>
                  Raktáron:{" "}
                  {product?.inStock?.value
                    ? product?.stock?.value > 0
                      ? product?.stock?.value
                      : "Külső raktár"
                    : "Külső raktár"}
                </div>
                <span
                  className={`size-6  ${
                    activeTab === idx ? "fill-blue-500" : "fill-green-800"
                  } flex flex-row justify-center items-center mx-2`}>
                  {activeTab === idx ? icons.chevronDown : icons.chevronUp}
                </span>
              </div>
              {activeTab === idx && (
                <Product
                  parentId={parentId}
                  productGroupOptions={productGroupOptions}
                  product={product}
                  priceTypeList={priceTypeList}
                  productColorsList={productColorsList}
                  productSizesList={productSizesList}
                  productMarksList={productMarksList}
                  productTagList={productTagList}
                  onProductGroupUpdated={onProductGroupUpdated}
                  onRemoveLocalProduct={handleRemoveLocalProduct}
                />
              )}
            </React.Fragment>
          );
        })}
    </div>
  );
};

export default ProductList;
