import React from "react";
import Image from "next/image";
import ApiCall from "@/api/api-calls";

const getImageSrc = (img) => {
  if (!img) return "/images/no-image.png";
  const name = typeof img === "string" ? img : img?.originalname || img?._id;
  return ApiCall.mediaStore.getResized("/2/2/0", name);
};

const getReadableValue = (val) => {
  if (!val || val?.length === 0)
    return <span className="italic text-gray-400">(nincs)</span>;
  if (typeof val === "string") return val;
  if (Array.isArray(val)) {
    return val
      .map((entry) => entry?.value || entry?.originalname || entry?._id || "?")
      .join(", ");
  }
  if (typeof val === "object" && val?.value !== undefined) {
    return val.value || <span className="italic text-gray-400">(nincs)</span>;
  }
  return JSON.stringify(val);
};

const highlightDiff = (live = "", staging = "") => {
  if (!live || !staging || live === staging) return staging;
  const liveWords = live.split(/\s+/);
  const stagingWords = staging.split(/\s+/);
  return stagingWords.map((word, idx) => {
    const liveWord = liveWords[idx];
    const isDifferent = word !== liveWord;
    return (
      <span
        key={idx}
        className={isDifferent ? "bg-yellow-200 px-1 rounded-sm" : ""}>
        {word + " "}
      </span>
    );
  });
};

const getDiffClass = (liveDisplay, stagingDisplay) => {
  const cleanLive = (liveDisplay || "").toString().trim().toLowerCase();
  const cleanStaging = (stagingDisplay || "").toString().trim().toLowerCase();
  if (!cleanStaging && cleanLive)
    return "bg-red-100 text-red-800 px-2 py-1 rounded";
  if (cleanLive !== cleanStaging)
    return "bg-yellow-100 text-yellow-800 font-medium px-2 py-1 rounded";
  return "bg-green-100 text-green-800 font-medium px-2 py-1 rounded";
};

export default function ProductConflictDetails({
  item,
  onApprove,
  onDiscard,
  index,
  uniqueId,
}) {
  const conflicts = item.conflictFields || [];

  const renderMetaImage = (key) => {
    const stagingImg = item[key.replace("Meta", "")]?.[0] || null;
    const meta = item[key] || {};
    const liveOriginalFileName =
      item.conflictMeta?.differences?.[key]?.live?.originalFileName;
    const liveImg = liveOriginalFileName
      ? { originalname: liveOriginalFileName }
      : null;

    return (
      <details open className="mb-4">
        <summary className="cursor-pointer font-semibold text-gray-700 mb-1">
          {key}
        </summary>
        <div className="flex gap-4 mt-2">
          <div className="text-center text-xs">
            <div className="text-gray-500">Live</div>
            <Image
              src={getImageSrc(liveImg)}
              width={80}
              height={80}
              alt="live"
              className="w-16 h-16 object-contain border rounded"
            />
            <div className="text-gray-500">{liveImg?.originalname || "—"}</div>
          </div>
          <div className="text-center text-xs">
            <div className="text-blue-500">Staging</div>
            <Image
              src={getImageSrc(stagingImg)}
              width={80}
              height={80}
              alt="staging"
              className="w-16 h-16 object-contain border rounded"
            />
            <div className="text-gray-500">
              {stagingImg?.originalname || "—"}
            </div>
            <div className="text-gray-500">
              Match: <strong>{meta.matchType || "—"}</strong>
            </div>
            <div className="text-gray-500">
              Matched: <strong>{meta.matched ? "✅" : "❌"}</strong>
            </div>
          </div>
        </div>
      </details>
    );
  };

  const renderGalleryMetaImages = () => {
    const galleryMeta = item.productGalleryImageMeta || [];
    return (
      <details open className="mb-4">
        <summary className="cursor-pointer font-semibold text-gray-700 mb-1">
          productGalleryImageMeta
        </summary>
        <div className="flex flex-wrap gap-4 mt-2">
          {galleryMeta.map((meta) => (
            <div key={meta._id} className="text-xs text-center">
              <Image
                src={getImageSrc(meta.originalFileName)}
                width={60}
                height={60}
                alt={meta.originalFileName}
                className="border rounded shadow-sm"
              />
              <div className="text-gray-700">{meta.originalFileName}</div>
              <div className="text-gray-500">
                Method: {meta.selectionMethod}
              </div>
              <div className="text-gray-500">Used: {meta.usedCode}</div>
              <div className="text-gray-500">Match: {meta.matchType}</div>
              <div className="text-gray-500">{meta.matched ? "✅" : "❌"}</div>
            </div>
          ))}
        </div>
      </details>
    );
  };

  return (
    <div className="space-y-4" key={`${item._id}-diff-index-${index}`}>
      {conflicts.map((key) => {
        const uniqueKey = uniqueId++;
        if (key === "productGalleryImageMeta") return renderGalleryMetaImages();
        if (key.endsWith("Meta")) return renderMetaImage(key);

        const liveRaw = item.conflictMeta?.differences?.[key]?.live;
        const stagingRaw = item[key];
        const liveDisplay = getReadableValue(liveRaw);
        const stagingDisplay = getReadableValue(stagingRaw);
        const shouldHighlight =
          typeof liveDisplay === "string" && typeof stagingDisplay === "string";

        return (
          <details open key={uniqueKey} className="border-b pb-2">
            <summary className="cursor-pointer font-semibold text-gray-700 mb-1">
              {key}
            </summary>
            <div className="text-sm mt-2">
              <div>
                <span className="text-gray-500">Live:</span>{" "}
                <span className="bg-gray-100 px-2 py-1 rounded inline-block max-w-[300px] break-words">
                  {liveDisplay}
                </span>
              </div>
              <div className="mt-1">
                <span className="text-blue-500">Staging:</span>{" "}
                <span
                  className={`${getDiffClass(
                    liveDisplay,
                    stagingDisplay
                  )} inline-block max-w-[300px] break-words`}>
                  {shouldHighlight
                    ? highlightDiff(liveDisplay, stagingDisplay)
                    : stagingDisplay}
                </span>
              </div>
            </div>
          </details>
        );
      })}

      {(onApprove || onDiscard) && (
        <div className="pt-2 flex gap-2">
          <button
            onClick={() => onApprove(item._id)}
            className="bg-green-500 text-white px-3 py-1 rounded">
            ✅ Elfogad
          </button>
          <button
            onClick={() => onDiscard(item._id)}
            className="bg-red-500 text-white px-3 py-1 rounded">
            ❌ Eldob
          </button>
        </div>
      )}
    </div>
  );
}
