"use client";
import React, { useEffect, useState } from "react";
import ApiCall from "@/api/api-calls";
import Image from "next/image";

export default function ReviewProducts({ batchId: initialBatchId }) {
  const [items, setItems] = useState([]);
  const [batchId, setBatchId] = useState(initialBatchId);
  const [statusFilter, setStatusFilter] = useState("conflict");
  const [loadingAll, setLoadingAll] = useState(false);

  const fetchData = async () => {
    const { data } = await ApiCall.staging.getProductStaging({
      batchId,
      conflictStatus: statusFilter,
    });
    setItems(data.items || []);
  };

  useEffect(() => {
    if (batchId) fetchData();
  }, [batchId, statusFilter]);

  const handleApprove = async (id) => {
    await ApiCall.staging.approveProduct(id);
    await fetchData();
  };

  const handleDiscard = async (id) => {
    await ApiCall.staging.discardProduct(id);
    await fetchData();
  };

  const handleApproveAll = async () => {
    if (!batchId) return;
    await ApiCall.staging.approveAllProducts({
      batchId,
      conflictStatus: statusFilter,
    });
    await fetchData();
  };

  const handleDiscardAll = async () => {
    if (!batchId) return;
    await ApiCall.staging.discardAllProducts({
      batchId,
      conflictStatus: statusFilter,
    });
    await fetchData();
  };

  const getImageSrc = (img) => {
    if (!img) return "/images/no-image.png";
    const name = typeof img === "string" ? img : img?.originalname || img?._id;
    return ApiCall.mediaStore.getResized("/2/2/0", name);
  };

  const renderImageRow = (title, liveImg, stagingImg) => (
    <details open className="mb-2">
      <summary className="font-semibold text-gray-700 mb-1">{title}</summary>
      <div className="flex gap-6 mt-2">
        {[
          { label: "Live", img: liveImg },
          { label: "Staging", img: stagingImg },
        ].map(({ label, img }) => (
          <div key={label} className="text-center text-xs">
            <div
              className={label === "Live" ? "text-gray-500" : "text-blue-500"}>
              {label}
            </div>
            <Image
              src={getImageSrc(img)}
              width={80}
              height={80}
              alt={label}
              className="border rounded object-contain w-20 h-20"
            />
            <div className="text-gray-400 mt-1">
              {img?.originalname || "(nincs)"}
            </div>
          </div>
        ))}
      </div>
    </details>
  );

  const extractGalleryImages = (content = []) => {
    const images = [];
    try {
      content.forEach((block) =>
        block?.content?.forEach((c1) =>
          c1?.content?.forEach((c2) =>
            c2?.content?.forEach((leaf) => {
              if (leaf?.type === "gallery" && Array.isArray(leaf.images)) {
                images.push(...leaf.images.map((img) => img.image));
              }
            })
          )
        )
      );
    } catch (e) {
      console.warn("Could not extract gallery images", e);
    }
    return images;
  };

  const renderMeta = (meta) => {
    if (!meta || (Array.isArray(meta) && meta.length === 0)) return "(nincs)";
    return Array.isArray(meta) ? (
      <ul className="list-disc pl-4 text-xs text-gray-700">
        {meta.map((m, i) => (
          <li key={i}>
            {m.originalFileName || m._id || "?"} ({m.usedCode},{" "}
            {m.selectionMethod})
          </li>
        ))}
      </ul>
    ) : (
      <span>
        {meta.originalFileName} ({meta.usedCode}, {meta.selectionMethod})
      </span>
    );
  };

  const getReadable = (val, key) => {
    if (!val || (Array.isArray(val) && val.length === 0)) return "(nincs)";
    if (typeof val === "string" || typeof val === "number") return val;

    if (key === "productContent") {
      const images = extractGalleryImages(val);
      return images.length ? (
        <div className="flex gap-1 flex-wrap">
          {images.map((img, i) => (
            <Image
              key={i}
              src={getImageSrc(img)}
              width={60}
              height={60}
              alt="gallery-img"
              className="border rounded object-contain w-14 h-14"
            />
          ))}
        </div>
      ) : (
        "(nincs)"
      );
    }

    if (Array.isArray(val)) {
      return val
        .map((entry) => {
          if (!entry) return "?";
          if (typeof entry === "string") return entry;

          switch (key) {
            case "productColor":
              return entry?.colorTitle?.value ?? entry._id;
            case "productSize":
              return entry?.sizeTitle?.value ?? entry._id;
            case "productMark":
              return entry?.markTitle?.value ?? entry._id;
            case "productCategory":
              return entry?.productCategoryTitle?.value ?? entry._id;
            case "productGalleryImages":
              return entry?.originalname || entry?._id;
            default:
              return entry._id || "?";
          }
        })
        .join(", ");
    }

    if (typeof val === "object" && val?.value !== undefined)
      return val.value || "(nincs)";
    if (key === "productParentGroup")
      return val?.productGroupTitle?.value || val._id || "(nincs)";
    if (key === "productColor" && val?.colorTitle?.value)
      return val.colorTitle.value;
    if (key === "productSize" && val?.sizeTitle?.value)
      return val.sizeTitle.value;
    if (key === "productMark" && val?.markTitle?.value)
      return val.markTitle.value;
    return JSON.stringify(val);
  };

  const grouped = items.reduce((acc, product) => {
    const groupId = product.productParentGroup?._id || "no-group";
    const groupTitle =
      product.productParentGroup?.productGroupTitle?.value || "No Group";
    if (!acc[groupId]) acc[groupId] = { groupId, groupTitle, products: [] };
    acc[groupId].products.push(product);
    return acc;
  }, {});

  return (
    <div className="p-4">
      <h2 className="text-xl font-bold mb-4">🧾 Review Products</h2>

      <div className="flex flex-wrap gap-4 mb-6 items-center">
        <input
          type="text"
          placeholder="Import batch ID"
          value={batchId}
          onChange={(e) => setBatchId(e.target.value)}
          className="border px-3 py-1 rounded w-64"
        />
        <select
          value={statusFilter}
          onChange={(e) => setStatusFilter(e.target.value)}
          className="border px-2 py-1 rounded">
          <option value="">All</option>
          <option value="conflict">Conflict</option>
          <option value="new">New</option>
          <option value="update">Update</option>
        </select>

        <button
          onClick={handleApproveAll}
          className="bg-green-600 text-white px-4 py-1 rounded disabled:opacity-50"
          disabled={loadingAll}>
          ✅ Összes elfogadása
        </button>

        <button
          onClick={handleDiscardAll}
          className="bg-red-600 text-white px-4 py-1 rounded disabled:opacity-50"
          disabled={loadingAll}>
          🗑️ Összes eldobása
        </button>
      </div>

      {Object.values(grouped).map((group) => (
        <div key={group.groupId} className="mb-10">
          <h3 className="text-lg font-semibold mb-2 border-b pb-1">
            📦 {group.groupTitle}
          </h3>
          <details>
            <table className="w-full table-auto border text-sm">
              <thead>
                <tr className="bg-gray-100 text-left">
                  <th className="p-2 border">Title</th>
                  <th className="p-2 border">Conflicts</th>
                  <th className="p-2 border">Details</th>
                  <th className="p-2 border">Actions</th>
                </tr>
              </thead>
              <tbody>
                {group.products.map((product) => (
                  <tr key={product._id} className="align-top border-t">
                    <td className="p-2 border font-medium text-gray-800">
                      {product.parentName || "(nincs)"}
                    </td>
                    <td className="p-2 border space-y-1">
                      {(product.conflictFields || []).map((key) => (
                        <div
                          key={key}
                          className="flex items-center gap-2 text-gray-700">
                          <span>🔁</span>
                          <span className="text-sm">{key}</span>
                        </div>
                      ))}
                    </td>
                    <td className="p-2 border space-y-3">
                      {(product.conflictFields || []).map((key) => {
                        // const liveVal =
                        //   product.conflictMeta?.differences?.[key]?.live ??
                        //   product.conflictMeta?.liveMatchId?.[key];
                        // const stagingVal = product[key];
                        const liveVal =
                          product.conflictMeta?.differences?.[key]?.existing ??
                          product.conflictMeta?.liveMatchId?.[key];

                        const stagingVal =
                          product.conflictMeta?.differences?.[key]?.incoming ??
                          product[key];

                        if (key === "productGalleryImageMeta") {
                          return (
                            <details open key={key}>
                              <summary className="font-semibold text-gray-600">
                                {key}
                              </summary>
                              <div className="mb-1">
                                <span className="text-gray-500">Live:</span>{" "}
                                <div>{renderMeta(liveVal)}</div>
                              </div>
                              <div>
                                <span className="text-blue-500">Staging:</span>{" "}
                                <div>{renderMeta(stagingVal)}</div>
                              </div>
                            </details>
                          );
                        }

                        if (
                          [
                            "productGalleryImages",
                            "productFeaturedImage",
                          ].includes(key)
                        ) {
                          const stagingImg = Array.isArray(stagingVal)
                            ? stagingVal[0]
                            : stagingVal;
                          const liveImg = Array.isArray(liveVal)
                            ? liveVal[0]
                            : liveVal;
                          return renderImageRow(key, liveImg, stagingImg);
                        }

                        const liveDisplay = getReadable(liveVal, key);
                        const stagingDisplay = getReadable(stagingVal, key);

                        return (
                          <details open key={key}>
                            <summary className="font-semibold text-gray-600">
                              {key}
                            </summary>
                            <div className="mb-1">
                              <span className="text-gray-500">Live:</span>{" "}
                              <div>{liveDisplay}</div>
                            </div>
                            <div>
                              <span className="text-blue-500">Staging:</span>{" "}
                              <div>{stagingDisplay}</div>
                            </div>
                          </details>
                        );
                      })}
                    </td>
                    <td className="p-2 border">
                      <div className="flex flex-col gap-2">
                        <button
                          onClick={() => handleApprove(product._id)}
                          className="bg-green-500 text-white px-3 py-1 rounded">
                          Elfogad
                        </button>
                        <button
                          onClick={() => handleDiscard(product._id)}
                          className="bg-red-500 text-white px-3 py-1 rounded">
                          Eldob
                        </button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </details>
        </div>
      ))}
    </div>
  );
}
