"use client";

import { useEffect, useMemo, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

type TargetType = "person" | "user" | "department" | "position_title";

type RoleRow = { id: string; name: string };
type PeopleOpt = { id: string; name: string };
type DeptOpt = { id: string; name: string };

type UserOpt = { id: string; name?: string | null; email?: string | null };

function userLabel(u: UserOpt) {
  return u.name || u.email || u.id;
}

async function searchUsersBestEffort(search: string): Promise<UserOpt[]> {
  const q = search.trim();
  if (!q) return [];

  const tries = [
    `?q=${encodeURIComponent(q)}`,
    `?search=${encodeURIComponent(q)}`,
    `?query=${encodeURIComponent(q)}`,
    `?page=1&pageSize=10&q=${encodeURIComponent(q)}`,
  ];

  for (const t of tries) {
    try {
      const r = await ApiCall.users.getList(t);
      const data = r.data;
      const items = (data?.items ??
        data?.rows ??
        data?.data ??
        data?.list ??
        data?.results ??
        []) as any[];

      const mapped = items
        .map((x) => ({
          id: String(x.id ?? x.user_id ?? x.uuid ?? ""),
          name: x.name ?? x.full_name ?? null,
          email: x.email ?? null,
        }))
        .filter((x) => x.id);

      if (mapped.length) return mapped;
    } catch {}
  }

  return [];
}

export default function AssignmentModal(props: {
  packageId: string;
  onClose: () => void;
  onCreated: () => void;
}) {
  const [targetType, setTargetType] = useState<TargetType>("person");
  const [targetId, setTargetId] = useState<string>("");
  const [targetText, setTargetText] = useState<string>("");

  const [dueAt, setDueAt] = useState<string>("");
  const [note, setNote] = useState<string>("");

  // pickers
  const [search, setSearch] = useState("");
  const [people, setPeople] = useState<PeopleOpt[]>([]);
  const [users, setUsers] = useState<UserOpt[]>([]);
  const [depts, setDepts] = useState<DeptOpt[]>([]);
  const [busy, setBusy] = useState(false);
  const [busySearch, setBusySearch] = useState(false);

  // departments: we can reuse HR departments endpoint from peoples api
  const loadDepartments = async () => {
    try {
      const r = await ApiCall.peoples.getDepartments();
      const items = (r.data?.items ?? r.data ?? []) as any[];
      setDepts(
        items.map((x) => ({
          id: String(x.id),
          name: String(x.name ?? x.title ?? "Department"),
        }))
      );
    } catch {
      setDepts([]);
    }
  };

  useEffect(() => {
    loadDepartments();
  }, []);

  useEffect(() => {
    setTargetId("");
    setTargetText("");
    setSearch("");
  }, [targetType]);

  useEffect(() => {
    const q = search.trim();
    const t = setTimeout(async () => {
      setBusySearch(true);
      try {
        if (targetType === "person") {
          const r = await ApiCall.peoples.searchPeopleOptions(q);
          const items = (r.data?.items ?? r.data ?? []) as any[];
          setPeople(
            items
              .map((x) => ({
                id: String(x.id),
                name: String(x.name ?? x.label ?? "Person"),
              }))
              .slice(0, 20)
          );
          return;
        }
        if (targetType === "user") {
          const us = await searchUsersBestEffort(q);
          setUsers(us.slice(0, 20));
          return;
        }
      } finally {
        setBusySearch(false);
      }
    }, 350);

    return () => clearTimeout(t);
  }, [search, targetType]);

  const canSubmit = useMemo(() => {
    if (targetType === "position_title") return !!targetText.trim();
    return !!targetId;
  }, [targetId, targetText, targetType]);

  const submit = async () => {
    if (!canSubmit) return toast.error("Hiányzó target");

    setBusy(true);
    try {
      await ApiCall.qms.createAssignment({
        packageId: props.packageId,
        targetType,
        targetId: targetType === "position_title" ? null : targetId,
        targetText: targetType === "position_title" ? targetText.trim() : null,
        dueAt: dueAt ? dueAt : null,
        note: note.trim() || null,
      });

      props.onCreated();
      props.onClose();
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Assignment létrehozása sikertelen"
      );
    } finally {
      setBusy(false);
    }
  };

  return (
    <div className="p-4 space-y-3">
      <div className="grid grid-cols-1 md:grid-cols-3 gap-2">
        <div>
          <label className="text-xs text-gray-600">Target type</label>
          <select
            className="w-full border rounded px-3 py-2"
            value={targetType}
            onChange={(e) => setTargetType(e.target.value as TargetType)}>
            <option value="person">person</option>
            <option value="user">user</option>
            <option value="department">department</option>
            <option value="position_title">position_title</option>
          </select>
        </div>

        <div>
          <label className="text-xs text-gray-600">Due date (optional)</label>
          <input
            type="date"
            className="w-full border rounded px-3 py-2"
            value={dueAt}
            onChange={(e) => setDueAt(e.target.value)}
          />
        </div>

        <div className="md:col-span-1">
          <label className="text-xs text-gray-600">Note (optional)</label>
          <input
            className="w-full border rounded px-3 py-2"
            value={note}
            onChange={(e) => setNote(e.target.value)}
          />
        </div>
      </div>

      {targetType === "position_title" ? (
        <div>
          <label className="text-xs text-gray-600">
            Position title (exact/ILIKE match)
          </label>
          <input
            className="w-full border rounded px-3 py-2"
            value={targetText}
            onChange={(e) => setTargetText(e.target.value)}
            placeholder='Pl. "Nurse", "Recepciós", stb.'
          />
        </div>
      ) : targetType === "department" ? (
        <div>
          <label className="text-xs text-gray-600">Department</label>
          <select
            className="w-full border rounded px-3 py-2"
            value={targetId}
            onChange={(e) => setTargetId(e.target.value)}>
            <option value="">Válassz…</option>
            {depts.map((d) => (
              <option key={d.id} value={d.id}>
                {d.name}
              </option>
            ))}
          </select>
        </div>
      ) : (
        <div className="border rounded p-3 space-y-2">
          <div className="flex items-center gap-2">
            <div className="font-semibold">Pick {targetType}</div>
            <input
              className="ml-auto border rounded px-3 py-1"
              value={search}
              onChange={(e) => setSearch(e.target.value)}
              placeholder="Keresés…"
            />
          </div>

          {busySearch ? (
            <div className="text-xs text-gray-500">Keresés…</div>
          ) : targetType === "person" ? (
            <div className="max-h-[220px] overflow-auto border rounded">
              {(people ?? []).map((p) => (
                <button
                  key={p.id}
                  className={[
                    "w-full text-left px-3 py-2 border-b last:border-b-0 hover:bg-gray-50",
                    targetId === p.id ? "bg-amber-50" : "",
                  ].join(" ")}
                  onClick={() => setTargetId(p.id)}>
                  {p.name}
                </button>
              ))}
              {!people.length ? (
                <div className="p-3 text-xs text-gray-500">Nincs találat.</div>
              ) : null}
            </div>
          ) : (
            <div className="max-h-[220px] overflow-auto border rounded">
              {(users ?? []).map((u) => (
                <button
                  key={u.id}
                  className={[
                    "w-full text-left px-3 py-2 border-b last:border-b-0 hover:bg-gray-50",
                    targetId === u.id ? "bg-amber-50" : "",
                  ].join(" ")}
                  onClick={() => setTargetId(u.id)}>
                  {userLabel(u)}
                </button>
              ))}
              {!users.length ? (
                <div className="p-3 text-xs text-gray-500">Nincs találat.</div>
              ) : null}
            </div>
          )}
        </div>
      )}

      <div className="flex gap-2 justify-end pt-2 border-t">
        <button
          className="px-3 py-2 rounded border"
          onClick={props.onClose}
          disabled={busy}>
          Mégse
        </button>
        <button
          className="px-3 py-2 rounded bg-emerald-600 text-white disabled:opacity-60"
          onClick={submit}
          disabled={!canSubmit || busy}>
          {busy ? "Létrehozás…" : "Létrehoz"}
        </button>
      </div>
    </div>
  );
}
