"use client";
import { useEffect, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

export default function CategoryModal({
  mode, // "create" | "edit"
  category,
  parentId,
  onClose,
  onSaved,
}: {
  mode: "create" | "edit";
  category?: any; // {id,name,sort_order,is_active,parent_id}
  parentId?: string | null;
  onClose: () => void;
  onSaved: () => void;
}) {
  const [name, setName] = useState("");
  const [sortOrder, setSortOrder] = useState<number>(0);
  const [isActive, setIsActive] = useState(true);
  const [busy, setBusy] = useState(false);

  useEffect(() => {
    setName(category?.name ?? "");
    setSortOrder(Number(category?.sort_order ?? 0));
    setIsActive(category?.is_active ?? true);
  }, [category]);

  const save = async () => {
    if (!name.trim()) return toast.error("Hiányzó név");
    setBusy(true);
    try {
      if (mode === "create") {
        await ApiCall.qms.createCategory({
          name: name.trim(),
          parentId: parentId ?? null,
          sortOrder,
          isActive,
        });
      } else {
        await ApiCall.qms.updateCategory(category.id, {
          name: name.trim(),
          sortOrder,
          isActive,
          // parentId: category.parent_id ?? null, // add if you want parent change in edit
        });
      }
      toast.success("Mentve");
      onSaved();
      onClose();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Mentés sikertelen");
    } finally {
      setBusy(false);
    }
  };

  return (
    <div className="p-4 space-y-3">
      <div>
        <label className="text-xs text-gray-600">Név</label>
        <input
          className="w-full border rounded px-3 py-2"
          value={name}
          onChange={(e) => setName(e.target.value)}
        />
      </div>

      <div className="grid grid-cols-2 gap-2">
        <div>
          <label className="text-xs text-gray-600">Sorrend</label>
          <input
            type="number"
            className="w-full border rounded px-3 py-2"
            value={sortOrder}
            onChange={(e) => setSortOrder(Number(e.target.value))}
          />
        </div>

        <div className="flex items-end gap-2">
          <input
            id="catActive"
            type="checkbox"
            checked={isActive}
            onChange={(e) => setIsActive(e.target.checked)}
          />
          <label htmlFor="catActive" className="text-sm">
            Aktív
          </label>
        </div>
      </div>

      <div className="flex gap-2 justify-end pt-2 border-t">
        <button
          className="px-3 py-2 rounded border"
          onClick={onClose}
          disabled={busy}>
          Mégse
        </button>
        <button
          className="px-3 py-2 rounded bg-emerald-600 text-white disabled:opacity-60"
          onClick={save}
          disabled={busy}>
          {busy ? "Mentés…" : "Mentés"}
        </button>
      </div>
    </div>
  );
}
