"use client";

import { useEffect, useMemo, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

type Effect = "allow" | "deny" | null;
type Action = "read" | "create" | "update" | "delete" | "manage";

const ACTIONS: { key: Action; label: string }[] = [
  { key: "read", label: "Read" },
  { key: "create", label: "Create" },
  { key: "update", label: "Update" },
  { key: "delete", label: "Delete" },
  { key: "manage", label: "Manage" },
];

type SubjectType = "role" | "user";

type RoleRow = { id: string; name: string };

type UserRow = {
  id: string;
  username?: string;
  email?: string;
  title?: string;
  name?: string;
};

type GrantRow = {
  subjectType: SubjectType;
  subjectId: string;
  actions: Record<Action, Effect>;
};

function subjectLabelUser(u: UserRow) {
  const title = u.title || u.name || u.username || u.email || u.id;
  const extra = u.email && u.email !== title ? ` • ${u.email}` : "";
  return `${title}${extra}`;
}

function normalizeUsersFromGetListPayload(payload: any): UserRow[] {
  // supports: payload.tableData OR payload.users/items/data OR payload.result.tableData etc.
  const candidates =
    payload?.tableData ??
    payload?.users ??
    payload?.items ??
    payload?.data ??
    payload?.result?.tableData ??
    payload?.result?.users ??
    payload?.result?.items ??
    payload?.result?.data ??
    [];

  const arr = Array.isArray(candidates) ? candidates : [];

  return arr
    .map((u: any) => ({
      id: String(u?.id ?? u?._id ?? ""),
      username: u?.username ?? u?.userName ?? u?.name ?? undefined,
      email: u?.email ?? undefined,
      title: u?.title ?? u?.fullName ?? undefined,
      name: u?.name ?? undefined,
    }))
    .filter((u: UserRow) => !!u.id);
}

export default function QmsAclEditorModal({
  objectType,
  objectId,
  onClose,
}: {
  objectType: "category" | "document";
  objectId: string;
  onClose: () => void;
}) {
  const [loading, setLoading] = useState(true);
  const [subjectType, setSubjectType] = useState<SubjectType>("role");

  const [roles, setRoles] = useState<RoleRow[]>([]);
  const [users, setUsers] = useState<UserRow[]>([]);

  const [selectedSubjectId, setSelectedSubjectId] = useState<string>("");
  const [existingGrants, setExistingGrants] = useState<GrantRow[]>([]);

  const [actions, setActions] = useState<Record<Action, Effect>>({
    read: null,
    create: null,
    update: null,
    delete: null,
    manage: null,
  });

  const loadRoles = async (): Promise<RoleRow[]> => {
    const r = await ApiCall.qms.listRoles();
    const payload = r?.data?.result ?? r?.data;
    return (payload?.items ?? []) as RoleRow[];
  };

  const loadUsers = async (): Promise<UserRow[]> => {
    // IMPORTANT: pass query string as a string (or change to params object if your ApiCall expects that)
    const r = await ApiCall.users.getList("?page=1&pageSize=5000");
    const payload = r?.data?.result ?? r?.data;
    return normalizeUsersFromGetListPayload(payload);
  };

  const fetchAcl = async () => {
    const r = await ApiCall.qms.listAcl({ objectType, objectId });
    setExistingGrants((r?.data?.grants ?? []) as GrantRow[]);
  };

  const loadAll = async () => {
    setLoading(true);
    try {
      const [rr, uu] = await Promise.all([loadRoles(), loadUsers()]);
      setRoles(rr ?? []);
      setUsers(uu ?? []);
      await fetchAcl();
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? e?.message ?? "ACL betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    loadAll();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [objectType, objectId]);

  const subjects = useMemo(() => {
    if (subjectType === "role") {
      return roles.map((r) => ({ id: r.id, label: r.name }));
    }
    return users.map((u) => ({ id: u.id, label: subjectLabelUser(u) }));
  }, [subjectType, roles, users]);

  useEffect(() => {
    if (!selectedSubjectId) {
      setActions({
        read: null,
        create: null,
        update: null,
        delete: null,
        manage: null,
      });
      return;
    }

    const g = existingGrants.find(
      (x) => x.subjectType === subjectType && x.subjectId === selectedSubjectId
    );

    setActions(
      g?.actions ?? {
        read: null,
        create: null,
        update: null,
        delete: null,
        manage: null,
      }
    );
  }, [selectedSubjectId, existingGrants, subjectType]);

  const save = async () => {
    if (!selectedSubjectId)
      return toast.error("Válassz szerepkört / felhasználót");

    try {
      await ApiCall.qms.saveAclBulk({
        objectType,
        objectId,
        subjectType,
        subjectId: selectedSubjectId,
        actions,
      });

      toast.success("ACL mentve");
      await fetchAcl();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Mentés sikertelen");
    }
  };

  const setAll = (value: Effect) => {
    setActions({
      read: value,
      create: value,
      update: value,
      delete: value,
      manage: value,
    });
  };

  // Optional: helps you see what's happening quickly
  // console.log("subjectType:", subjectType, "roles:", roles.length, "users:", users.length);

  if (loading) {
    return <div className="p-4 text-sm text-gray-500">Betöltés…</div>;
  }

  return (
    <div className="p-4 space-y-4">
      <div className="flex flex-wrap items-center gap-2">
        <div className="text-sm text-gray-600">
          Object: <b>{objectType}</b> • <code>{objectId}</code>
        </div>

        <div className="ml-auto flex gap-2">
          <button
            className="px-3 py-2 rounded border"
            onClick={() => setAll(null)}>
            Összes: inherit
          </button>
          <button
            className="px-3 py-2 rounded border"
            onClick={() => setAll("allow")}>
            Összes: allow
          </button>
          <button
            className="px-3 py-2 rounded border"
            onClick={() => setAll("deny")}>
            Összes: deny
          </button>
        </div>
      </div>

      <div className="flex flex-wrap gap-2 items-center">
        {/* 1) Use currentTarget + reset selection */}
        <select
          className="border rounded px-3 py-2"
          value={subjectType}
          onChange={(e) => {
            const next = e.currentTarget.value as SubjectType;
            setSubjectType(next);
            setSelectedSubjectId("");
          }}>
          <option value="role">Szerepkör</option>
          <option value="user">Felhasználó</option>
        </select>

        {/* 2) key={subjectType} forces remount when switching role/user (prevents “stuck options”) */}
        <select
          key={subjectType}
          className="border rounded px-3 py-2 min-w-[360px]"
          value={selectedSubjectId}
          onChange={(e) => setSelectedSubjectId(e.currentTarget.value)}>
          <option value="">— Válassz —</option>
          {subjects.map((s) => (
            <option key={s.id} value={s.id}>
              {s.label}
            </option>
          ))}
        </select>

        <button
          className="ml-auto px-3 py-2 rounded bg-gray-800 text-white"
          onClick={save}
          disabled={!selectedSubjectId}>
          Mentés
        </button>

        <button className="px-3 py-2 rounded border" onClick={onClose}>
          Bezár
        </button>
      </div>

      <div className="overflow-auto">
        <table className="min-w-full text-sm border rounded">
          <thead className="text-left border-b bg-gray-50">
            <tr>
              <th className="py-2 px-3">Action</th>
              <th className="py-2 px-3">Value</th>
              <th className="py-2 px-3 text-gray-500">Notes</th>
            </tr>
          </thead>

          <tbody>
            {ACTIONS.map((a) => {
              const v = actions[a.key];
              return (
                <tr key={a.key} className="border-b">
                  <td className="py-2 px-3 font-medium">{a.label}</td>
                  <td className="py-2 px-3">
                    <select
                      className="border rounded px-2 py-1"
                      value={v ?? ""}
                      onChange={(e) => {
                        const val = e.currentTarget.value;
                        setActions((prev) => ({
                          ...prev,
                          [a.key]: val === "" ? null : (val as Effect),
                        }));
                      }}>
                      <option value="">inherit (none)</option>
                      <option value="allow">allow</option>
                      <option value="deny">deny</option>
                    </select>
                  </td>
                  <td className="py-2 px-3 text-gray-500">
                    deny &gt; allow &gt; inherit
                  </td>
                </tr>
              );
            })}
          </tbody>
        </table>
      </div>

      <details className="text-sm text-gray-600">
        <summary className="cursor-pointer select-none">
          Jelenlegi szabályok ehhez az objecthez
        </summary>

        <div className="mt-2 border rounded p-2 bg-gray-50 max-h-64 overflow-auto">
          {!existingGrants.length ? (
            <div className="text-gray-500">Nincs szabály</div>
          ) : (
            <ul className="space-y-2">
              {existingGrants.map((g) => (
                <li key={`${g.subjectType}:${g.subjectId}`}>
                  <div className="font-medium">
                    {g.subjectType}:{g.subjectId}
                  </div>
                  <div className="text-xs">
                    {Object.entries(g.actions)
                      .map(([k, v]) => `${k}=${v ?? "inherit"}`)
                      .join(" • ")}
                  </div>
                </li>
              ))}
            </ul>
          )}
        </div>
      </details>
    </div>
  );
}
