"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import StatusBadge from "@/components/qms/StatusBadge";
import Chips from "@/components/qms/Chips";

type WorkItemDetail = {
  id: string;
  title: string;
  status: string;
  due_at: string | null;
  notes: string | null;

  object_type: string;
  object_id: string;

  created_at: string | null;
  updated_at: string | null;

  createdByName?: string | null;
  updatedByName?: string | null;

  assignees: { subjectType: string; subjectId: string; displayName: string }[];
  comments: {
    id: string;
    body: string;
    createdAt: string;
    author: { name: string; email?: string | null };
  }[];
};

export default function WorkItemModal({
  id,
  onClose,
  onChanged,
}: {
  id: string;
  onClose: () => void;
  onChanged: () => void;
}) {
  const [data, setData] = useState<WorkItemDetail | null>(null);
  const [loading, setLoading] = useState(true);
  const [busy, setBusy] = useState(false);

  const [editStatus, setEditStatus] = useState("open");
  const [editDue, setEditDue] = useState<string>("");
  const [editNotes, setEditNotes] = useState("");

  const [comment, setComment] = useState("");

  const load = useCallback(async () => {
    setLoading(true);
    try {
      const r = await ApiCall.qms.getWorkItemDetail(id);
      const d = r.data as WorkItemDetail;
      setData(d);

      setEditStatus(d.status ?? "open");
      setEditDue(d.due_at ? String(d.due_at).slice(0, 10) : "");
      setEditNotes(d.notes ?? "");
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Work item betöltése sikertelen");
    } finally {
      setLoading(false);
    }
  }, [id]);

  useEffect(() => {
    load();
  }, [load]);

  const changed = useMemo(() => {
    if (!data) return false;
    const dueIso = editDue ? editDue : "";
    const currentDue = data.due_at ? String(data.due_at).slice(0, 10) : "";
    return (
      editStatus !== data.status ||
      (dueIso || "") !== (currentDue || "") ||
      (editNotes ?? "") !== (data.notes ?? "")
    );
  }, [data, editStatus, editDue, editNotes]);

  const save = async () => {
    if (!data) return;
    setBusy(true);
    try {
      await ApiCall.qms.updateWorkItem(data.id, {
        status: editStatus,
        dueAt: editDue || null,
        notes: editNotes || null,
      });
      toast.success("Mentve");
      await load();
      onChanged();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Mentés sikertelen");
    } finally {
      setBusy(false);
    }
  };

  const addComment = async () => {
    if (!comment.trim()) return toast.error("Írj kommentet");
    setBusy(true);
    try {
      await ApiCall.qms.addWorkItemComment(id, { body: comment.trim() });
      setComment("");
      toast.success("Komment hozzáadva");
      await load();
      onChanged();
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Komment sikertelen");
    } finally {
      setBusy(false);
    }
  };

  if (loading) return <div className="p-4">Betöltés…</div>;
  if (!data) return <div className="p-4">Nincs ilyen work item.</div>;

  return (
    <div className="p-4 space-y-4">
      <div className="flex items-center gap-2 flex-wrap">
        <div className="text-sm text-gray-600">
          <div className="font-semibold">{data.title}</div>
          <div className="text-xs">
            {data.object_type}:{data.object_id}
          </div>
          <div className="text-xs text-gray-500">
            Created: {data.createdByName ?? "—"} • Updated:{" "}
            {data.updatedByName ?? "—"}
          </div>
        </div>

        <div className="ml-auto flex items-center gap-2">
          <StatusBadge value={data.status} />
          <button className="px-2 py-1 border rounded" onClick={onClose}>
            Bezár
          </button>
        </div>
      </div>

      <section className="border rounded p-3 space-y-2">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-2">
          <div>
            <label className="text-xs text-gray-600">Státusz</label>
            <select
              className="w-full border rounded px-3 py-2"
              value={editStatus}
              onChange={(e) => setEditStatus(e.target.value)}>
              {["open", "in_progress", "blocked", "done"].map((s) => (
                <option key={s} value={s}>
                  {s}
                </option>
              ))}
            </select>
          </div>

          <div>
            <label className="text-xs text-gray-600">Határidő</label>
            <input
              type="date"
              className="w-full border rounded px-3 py-2"
              value={editDue}
              onChange={(e) => setEditDue(e.target.value)}
            />
          </div>

          <div className="md:col-span-3">
            <label className="text-xs text-gray-600">Jegyzet</label>
            <textarea
              className="w-full border rounded px-3 py-2 min-h-[90px]"
              value={editNotes}
              onChange={(e) => setEditNotes(e.target.value)}
            />
          </div>
        </div>

        <div className="flex gap-2 justify-end">
          <button
            className="px-3 py-2 rounded bg-emerald-600 text-white disabled:opacity-60"
            onClick={save}
            disabled={!changed || busy}>
            {busy ? "Mentés…" : "Mentés"}
          </button>
        </div>
      </section>

      <section className="border rounded p-3 space-y-2">
        <h3 className="font-semibold">Assignees</h3>
        {data.assignees?.length ? (
          <Chips
            items={data.assignees.map((a) => ({ displayName: a.displayName }))}
          />
        ) : (
          <div className="text-sm text-gray-500">—</div>
        )}
      </section>

      <section className="border rounded p-3 space-y-3">
        <h3 className="font-semibold">Kommentek</h3>

        <div className="space-y-2">
          {data.comments?.map((c) => (
            <div key={c.id} className="border rounded p-2">
              <div className="text-xs text-gray-500">
                <b>{c.author?.name ?? "User"}</b> • {c.createdAt}
              </div>
              <div className="text-sm whitespace-pre-wrap">{c.body}</div>
            </div>
          ))}
          {!data.comments?.length ? (
            <div className="text-sm text-gray-500">Nincs komment.</div>
          ) : null}
        </div>

        <div className="pt-2 border-t space-y-2">
          <textarea
            className="w-full border rounded px-3 py-2 min-h-[80px]"
            value={comment}
            onChange={(e) => setComment(e.target.value)}
            placeholder="Új komment…"
          />
          <div className="flex justify-end">
            <button
              className="px-3 py-2 rounded bg-gray-800 text-white disabled:opacity-60"
              onClick={addComment}
              disabled={busy}>
              {busy ? "Küldés…" : "Komment"}
            </button>
          </div>
        </div>
      </section>
    </div>
  );
}
