"use client";

import { useEffect, useRef, useState } from "react";
import toast from "react-hot-toast";
import attachmentsApi, { type Attachment } from "@/api/routes/attachments";
import {
  Card,
  CardBody,
  ToolbarRow,
  Button,
  SecondaryButton,
  DangerButton,
} from "@/components/ui/CardBits";

function fmt(iso?: string | null) {
  if (!iso) return "-";
  try {
    return new Date(iso).toLocaleString();
  } catch {
    return iso ?? "-";
  }
}

function bytesToHuman(n: string | number | null | undefined) {
  const num = typeof n === "string" ? Number(n) : n ?? 0;
  if (!Number.isFinite(num) || num <= 0) return "-";
  const units = ["B", "KB", "MB", "GB"];
  let v = num,
    i = 0;
  while (v >= 1024 && i < units.length - 1) {
    v /= 1024;
    i++;
  }
  return `${v.toFixed(i === 0 ? 0 : 1)} ${units[i]}`;
}

export default function AttachmentsPanel(props: {
  title?: string;
  autoLoad?: boolean;

  load: () => Promise<Attachment[]>;
  upload: (file: File) => Promise<any>;

  /** ha adsz unlink-et, megjelenik a gomb */
  unlink?: (attachmentId: string) => Promise<any>;
  unlinkLabel?: string; // default: "Unlink"

  /** ha nem adod meg, default: downloadByFetch */
  download?: (attachmentId: string) => Promise<any>;

  /** ha nem adod meg, default: previewByFetch -> window.open */
  preview?: (attachmentId: string) => Promise<any>;
}) {
  const {
    title = "Csatolmányok",
    autoLoad = true,
    load,
    upload,
    unlink,
    unlinkLabel = "Unlink",
    download,
    preview,
  } = props;

  const inputRef = useRef<HTMLInputElement | null>(null);

  const [items, setItems] = useState<Attachment[]>([]);
  const [loading, setLoading] = useState(false);
  const [uploading, setUploading] = useState(false);

  async function reload() {
    setLoading(true);
    try {
      setItems(await load());
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? "Csatolmányok betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  }

  useEffect(() => {
    if (autoLoad) void reload();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  async function onPickFile() {
    inputRef.current?.click();
  }

  async function onFile(file: File | null) {
    if (!file) return;
    setUploading(true);
    try {
      await upload(file);
      toast.success("Feltöltve");
      await reload();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Feltöltés sikertelen");
    } finally {
      setUploading(false);
      if (inputRef.current) inputRef.current.value = "";
    }
  }

  async function onDownload(id: string) {
    try {
      if (download) return await download(id);
      await attachmentsApi.downloadByFetch(id);
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Letöltés sikertelen");
    }
  }

  async function onPreview(id: string) {
    try {
      if (preview) return await preview(id);

      const r = await attachmentsApi.previewByFetch(id);
      // új tabban nyitjuk; blob url → auth mögött is jó
      const w = window.open(r.url, "_blank", "noopener,noreferrer");
      if (!w) toast("Popup blokkolva – engedélyezd a böngészőben.");
      // nem revoke-oljuk azonnal; a tab használja. (ha zavar, lehet timeout)
      setTimeout(() => URL.revokeObjectURL(r.url), 60_000);
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Preview sikertelen");
    }
  }

  async function onUnlink(id: string) {
    if (!unlink) return;
    if (!confirm("Biztosan lecsatolod (unlink)?")) return;

    try {
      await unlink(id);
      toast.success("Unlink kész");
      await reload();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Unlink sikertelen");
    }
  }

  return (
    <Card>
      <CardBody className="grid gap-2">
        <div className="text-sm font-semibold text-slate-800">{title}</div>

        <ToolbarRow>
          <Button onClick={() => void onPickFile()} disabled={uploading}>
            {uploading ? "..." : "+ Fájl feltöltés"}
          </Button>

          <SecondaryButton onClick={() => void reload()} disabled={loading}>
            {loading ? "..." : "Frissít"}
          </SecondaryButton>

          <input
            ref={inputRef}
            type="file"
            className="hidden"
            onChange={(e) => void onFile(e.target.files?.[0] ?? null)}
          />
        </ToolbarRow>

        <div className="grid gap-2">
          {items.map((a) => (
            <div
              key={a.id}
              className="flex items-center justify-between gap-3 rounded-xl border border-slate-200 p-3">
              <div className="min-w-0">
                <div className="font-semibold truncate">{a.original_name}</div>
                <div className="text-xs text-slate-600">
                  {a.mime_type ?? "unknown"} · {bytesToHuman(a.size_bytes)} ·{" "}
                  {fmt(a.uploaded_at)}
                </div>
              </div>

              <div className="flex gap-2">
                <SecondaryButton onClick={() => void onPreview(a.id)}>
                  Preview
                </SecondaryButton>

                <SecondaryButton onClick={() => void onDownload(a.id)}>
                  Letöltés
                </SecondaryButton>

                {unlink ? (
                  <DangerButton onClick={() => void onUnlink(a.id)}>
                    {unlinkLabel}
                  </DangerButton>
                ) : null}
              </div>
            </div>
          ))}

          {!items.length ? (
            <div className="text-sm text-slate-600">Nincs csatolmány.</div>
          ) : null}
        </div>
      </CardBody>
    </Card>
  );
}
