"use client";

import { useEffect, useState } from "react";
import toast from "react-hot-toast";
import { v4 as uuidv4 } from "uuid";
import {
  Card,
  CardBody,
  ToolbarRow,
  Button,
  SecondaryButton,
  TextArea,
  Divider,
  Chip,
} from "@/components/ui/CardBits";

export type ChecklistItem = { id: string; text: string; done: boolean };

export type NoteComment = {
  id: string;
  author_name?: string | null;
  body: string;
  is_public: boolean;
  checklist: ChecklistItem[];
  created_at: string;
};

function fmt(iso?: string | null) {
  if (!iso) return "-";
  try {
    return new Date(iso).toLocaleString();
  } catch {
    return iso ?? "-";
  }
}

type CreatePayload = {
  body: string;
  isPublic: boolean;
  checklist: ChecklistItem[];
};

type UpdatePayload = Partial<{
  body: string;
  isPublic: boolean;
  checklist: ChecklistItem[];
}>;

export default function NotesPanel(props: {
  title?: string;
  load: () => Promise<NoteComment[]>;
  create: (payload: CreatePayload) => Promise<any>;
  update: (commentId: string, payload: UpdatePayload) => Promise<any>;
}) {
  const { title = "Jegyzetek", load, create, update } = props;

  const [items, setItems] = useState<NoteComment[]>([]);
  const [loading, setLoading] = useState(false);

  // create form
  const [body, setBody] = useState("");
  const [isPublic, setIsPublic] = useState(false);
  const [checklist, setChecklist] = useState<ChecklistItem[]>([]);
  const [saving, setSaving] = useState(false);

  // edit form
  const [editingId, setEditingId] = useState<string | null>(null);
  const [editBody, setEditBody] = useState("");
  const [editIsPublic, setEditIsPublic] = useState(false);
  const [editChecklist, setEditChecklist] = useState<ChecklistItem[]>([]);
  const [updating, setUpdating] = useState(false);

  async function reload() {
    setLoading(true);
    try {
      setItems(await load());
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? "Jegyzetek betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  }

  useEffect(() => {
    void reload();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  async function onCreate() {
    if (!body.trim()) return;
    setSaving(true);
    try {
      await create({ body: body.trim(), isPublic, checklist });
      setBody("");
      setChecklist([]);
      setIsPublic(false);
      toast.success("Mentve");
      await reload();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Mentés sikertelen");
    } finally {
      setSaving(false);
    }
  }

  function startEdit(c: NoteComment) {
    setEditingId(c.id);
    setEditBody(c.body ?? "");
    setEditIsPublic(!!c.is_public);
    setEditChecklist((c.checklist ?? []).map((x) => ({ ...x })));
  }

  function cancelEdit() {
    setEditingId(null);
    setEditBody("");
    setEditIsPublic(false);
    setEditChecklist([]);
  }

  async function saveEdit() {
    if (!editingId) return;
    if (!editBody.trim()) return;

    setUpdating(true);
    try {
      await update(editingId, {
        body: editBody.trim(),
        isPublic: editIsPublic,
        checklist: editChecklist,
      });
      toast.success("Frissítve");
      cancelEdit();
      await reload();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Frissítés sikertelen");
    } finally {
      setUpdating(false);
    }
  }

  return (
    <Card>
      <CardBody className="grid gap-3">
        <div className="text-sm font-semibold text-slate-800">{title}</div>

        {/* CREATE */}
        <div className="grid gap-2">
          <TextArea
            value={body}
            onChange={(e) => setBody(e.target.value)}
            placeholder="Saját jegyzet… (opció: publikus, hogy mások is lássák)"
          />

          <ToolbarRow>
            <label className="flex items-center gap-2 text-sm text-slate-700">
              <input
                type="checkbox"
                checked={isPublic}
                onChange={(e) => setIsPublic(e.target.checked)}
              />
              Publikus (a résztvevők is látják)
            </label>

            <SecondaryButton
              onClick={() =>
                setChecklist((x) => [
                  ...x,
                  { id: uuidv4(), text: "", done: false },
                ])
              }>
              + Checklist item
            </SecondaryButton>

            <Button
              onClick={() => void onCreate()}
              disabled={saving || !body.trim()}>
              {saving ? "..." : "Mentés"}
            </Button>
          </ToolbarRow>

          {checklist.length ? (
            <div className="grid gap-2 rounded-xl border border-slate-200 p-3">
              {checklist.map((it) => (
                <div key={it.id} className="flex items-center gap-2">
                  <input
                    type="checkbox"
                    checked={it.done}
                    onChange={(e) =>
                      setChecklist((xs) =>
                        xs.map((x) =>
                          x.id === it.id ? { ...x, done: e.target.checked } : x
                        )
                      )
                    }
                  />
                  <input
                    className="w-full rounded-lg border border-slate-200 px-2 py-1 text-sm"
                    value={it.text}
                    onChange={(e) =>
                      setChecklist((xs) =>
                        xs.map((x) =>
                          x.id === it.id ? { ...x, text: e.target.value } : x
                        )
                      )
                    }
                    placeholder="Checklist szöveg…"
                  />
                  <SecondaryButton
                    onClick={() =>
                      setChecklist((xs) => xs.filter((x) => x.id !== it.id))
                    }>
                    töröl
                  </SecondaryButton>
                </div>
              ))}
            </div>
          ) : null}
        </div>

        <Divider />

        {/* LIST */}
        <div className="grid gap-2">
          {items.map((c) => {
            const isEditing = editingId === c.id;

            return (
              <div
                key={c.id}
                className="rounded-xl border border-slate-200 p-3">
                <div className="flex items-center justify-between gap-2">
                  <div className="text-xs text-slate-600">
                    {c.author_name ?? "—"} · {fmt(c.created_at)}
                  </div>

                  <div className="flex items-center gap-2">
                    <Chip tone={c.is_public ? "info" : "warn"}>
                      {c.is_public ? "public" : "private"}
                    </Chip>

                    {!isEditing ? (
                      <SecondaryButton onClick={() => startEdit(c)}>
                        Szerkesztés
                      </SecondaryButton>
                    ) : null}
                  </div>
                </div>

                {!isEditing ? (
                  <>
                    <div className="mt-2 whitespace-pre-wrap text-sm text-slate-800">
                      {c.body}
                    </div>

                    {c.checklist?.length ? (
                      <div className="mt-2 grid gap-1">
                        {c.checklist.map((x) => (
                          <div
                            key={x.id}
                            className="flex items-center gap-2 text-sm text-slate-700">
                            <input type="checkbox" checked={x.done} readOnly />
                            <div
                              className={
                                x.done ? "line-through opacity-70" : ""
                              }>
                              {x.text}
                            </div>
                          </div>
                        ))}
                      </div>
                    ) : null}
                  </>
                ) : (
                  <>
                    <div className="mt-2 grid gap-2">
                      <TextArea
                        value={editBody}
                        onChange={(e) => setEditBody(e.target.value)}
                      />

                      <ToolbarRow>
                        <label className="flex items-center gap-2 text-sm text-slate-700">
                          <input
                            type="checkbox"
                            checked={editIsPublic}
                            onChange={(e) => setEditIsPublic(e.target.checked)}
                          />
                          Publikus
                        </label>

                        <SecondaryButton
                          onClick={() =>
                            setEditChecklist((xs) => [
                              ...xs,
                              { id: uuidv4(), text: "", done: false },
                            ])
                          }>
                          + Checklist item
                        </SecondaryButton>

                        <SecondaryButton
                          onClick={cancelEdit}
                          disabled={updating}>
                          Mégse
                        </SecondaryButton>

                        <Button
                          onClick={() => void saveEdit()}
                          disabled={updating || !editBody.trim()}>
                          {updating ? "..." : "Mentés"}
                        </Button>
                      </ToolbarRow>

                      {editChecklist.length ? (
                        <div className="grid gap-2 rounded-xl border border-slate-200 p-3">
                          {editChecklist.map((it) => (
                            <div
                              key={it.id}
                              className="flex items-center gap-2">
                              <input
                                type="checkbox"
                                checked={it.done}
                                onChange={(e) =>
                                  setEditChecklist((xs) =>
                                    xs.map((x) =>
                                      x.id === it.id
                                        ? { ...x, done: e.target.checked }
                                        : x
                                    )
                                  )
                                }
                              />
                              <input
                                className="w-full rounded-lg border border-slate-200 px-2 py-1 text-sm"
                                value={it.text}
                                onChange={(e) =>
                                  setEditChecklist((xs) =>
                                    xs.map((x) =>
                                      x.id === it.id
                                        ? { ...x, text: e.target.value }
                                        : x
                                    )
                                  )
                                }
                                placeholder="Checklist szöveg…"
                              />
                              <SecondaryButton
                                onClick={() =>
                                  setEditChecklist((xs) =>
                                    xs.filter((x) => x.id !== it.id)
                                  )
                                }>
                                töröl
                              </SecondaryButton>
                            </div>
                          ))}
                        </div>
                      ) : null}
                    </div>
                  </>
                )}
              </div>
            );
          })}

          {!items.length && !loading ? (
            <div className="text-sm text-slate-600">Nincs jegyzet.</div>
          ) : null}
        </div>
      </CardBody>
    </Card>
  );
}
