"use client";

import dynamic from "next/dynamic";
import { useEffect, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

const SunEditor = dynamic(() => import("suneditor-react"), { ssr: false });
import "suneditor/dist/css/suneditor.min.css";

type Faq = {
  id?: string;
  title: string;
  content_html: string;
  header_template_id?: string | null;
  footer_template_id?: string | null;
};

type DocTemplate = {
  id: string;
  name: string;
  kind: string;
  is_active?: boolean;
  preview_image?: string | null;
};

export function FaqEditor({ id }: { id?: string }) {
  const [faq, setFaq] = useState<Faq>({
    title: "",
    content_html: "",
    header_template_id: null,
    footer_template_id: null,
  });
  const [loading, setLoading] = useState<boolean>(!!id);
  const [templates, setTemplates] = useState<DocTemplate[]>([]);
  const [templatesLoading, setTemplatesLoading] = useState(false);

  // LOAD EXISTING FAQ
  useEffect(() => {
    if (!id) {
      setLoading(false);
      return;
    }

    (async () => {
      try {
        const res = await ApiCall.faqs.getItem(id);
        if (res.status !== 200) throw new Error("Failed to load FAQ");
        setFaq(res.data);
      } catch (e) {
        console.error(e);
        toast.error("FAQ betöltése sikertelen");
      } finally {
        setLoading(false);
      }
    })();
  }, [id]);

  // LOAD DOC TEMPLATES (kind = 'faq' or generic)
  useEffect(() => {
    (async () => {
      try {
        setTemplatesLoading(true);
        const res = await ApiCall.docTemplates.getList({ kind: "faq" });
        setTemplates(res.data ?? []);
      } catch (e) {
        console.error(e);
      } finally {
        setTemplatesLoading(false);
      }
    })();
  }, []);

  const handleSave = async () => {
    try {
      const res = await ApiCall.faqs.upsertItem(faq);
      if (res.status !== 200) throw new Error("Failed to save");
      const saved = res.data;
      setFaq(saved);
      toast.success("FAQ mentve");
    } catch (e) {
      console.error(e);
      toast.error("Mentés sikertelen");
    }
  };

  const handleDownloadDocx = async () => {
    const faqId = faq.id ?? id;
    if (!faqId) {
      toast.error("Először mentsd el a FAQ-ot");
      return;
    }

    try {
      const res = await ApiCall.faqs.download(faqId);
      const blob = new Blob([res.data], {
        type: "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
      });

      const url = window.URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = `faq-${faqId}.docx`;
      document.body.appendChild(a);
      a.click();
      a.remove();
      window.URL.revokeObjectURL(url);
    } catch (e) {
      console.error(e);
      toast.error("DOCX letöltés sikertelen");
    }
  };

  const headerTemplate = templates.find((t) => t.id === faq.header_template_id);
  const footerTemplate = templates.find((t) => t.id === faq.footer_template_id);
  const headerPreviewUrl = headerTemplate?.preview_image
    ? ApiCall.mediaStoreApi.getResized(
        "/12/3/0",
        headerTemplate.preview_image,
        {
          format: "webp",
        }
      )
    : null;

  const footerPreviewUrl = footerTemplate?.preview_image
    ? ApiCall.mediaStoreApi.getResized(
        "/12/3/0",
        footerTemplate.preview_image,
        {
          format: "webp",
        }
      )
    : null;

  if (loading) return <p>Betöltés…</p>;

  return (
    <div className="max-w-3xl mx-auto space-y-4 py-4">
      {/* Template selectors */}
      <div className="flex flex-wrap gap-4 items-end">
        <div className="flex flex-col text-xs text-gray-600 w-full">
          <span className="mb-1">Fejléc sablon</span>
          <select
            className="border rounded px-2 py-1 text-sm min-w-[220px]"
            value={faq.header_template_id ?? ""}
            onChange={(e) =>
              setFaq((prev) => ({
                ...prev,
                header_template_id: e.target.value || null,
              }))
            }>
            <option value="">— Nincs —</option>
            {templates.map((t) => (
              <option key={t.id} value={t.id}>
                {t.name}
              </option>
            ))}
          </select>
          {headerPreviewUrl && (
            <div className="mt-2 border rounded overflow-hidden bg-gray-50">
              <img
                src={headerPreviewUrl}
                alt={`Fejléc sablon: ${headerTemplate?.name}`}
                className="w-full h-auto block"
              />
            </div>
          )}
        </div>

        <div className="flex flex-col text-xs text-gray-600 w-full">
          <span className="mb-1">Lábléc sablon</span>
          <select
            className="border rounded px-2 py-1 text-sm min-w-[220px]"
            value={faq.footer_template_id ?? ""}
            onChange={(e) =>
              setFaq((prev) => ({
                ...prev,
                footer_template_id: e.target.value || null,
              }))
            }>
            <option value="">— Nincs —</option>
            {templates.map((t) => (
              <option key={t.id} value={t.id}>
                {t.name}
              </option>
            ))}
          </select>
          {footerPreviewUrl && (
            <div className="mt-2 border rounded overflow-hidden bg-gray-50">
              <img
                src={footerPreviewUrl}
                alt={`Lábléc sablon: ${footerTemplate?.name}`}
                className="w-full h-auto block"
              />
            </div>
          )}
        </div>

        {templatesLoading && (
          <span className="text-xs text-gray-500">Sablonok betöltése…</span>
        )}
      </div>

      {/* Title */}
      <input
        type="text"
        value={faq.title}
        onChange={(e) => setFaq({ ...faq, title: e.target.value })}
        placeholder="FAQ cím (pl. VPN hibaelhárítás)"
        className="w-full px-3 py-2 border rounded text-lg"
      />

      {/* Editor */}
      <div className="border rounded">
        <SunEditor
          setContents={faq.content_html}
          onChange={(content: string) =>
            setFaq((prev) => ({ ...prev, content_html: content }))
          }
          setOptions={{
            height: "300px",
            font: [
              "Arial",
              "Calibri",
              "Times New Roman",
              "Courier New",
              "Verdana",
              "Roboto",
            ],
            fontSize: [8, 10, 12, 14, 16, 18, 24, 32],

            buttonList: [
              ["undo", "redo"],
              ["bold", "italic", "underline", "strike"],
              ["fontSize", "formatBlock"],
              ["fontColor", "hiliteColor"],
              ["align", "list", "lineHeight"],
              ["table"],
              ["link", "image"],
              ["removeFormat"],
              ["fullScreen", "showBlocks", "codeView"],
            ],
          }}
        />
      </div>

      {/* Actions */}
      <div className="flex gap-2">
        <button
          onClick={handleSave}
          className="px-4 py-2 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white text-sm">
          Mentés
        </button>
        <button
          onClick={handleDownloadDocx}
          className="px-4 py-2 rounded border border-gray-300 text-sm hover:bg-gray-50">
          DOCX letöltés
        </button>
      </div>
    </div>
  );
}
