"use client";

import dynamic from "next/dynamic";
import { useEffect, useRef, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import { slugifyTitle, timestampForFilename } from "@/utils/exportFileSlugify";
import apiClient from "@/lib/apiClient";
import plugins from "suneditor/src/plugins";
import compactTablePlugin from "./suneditorCompactTablePlugin";
import Select from "react-select";

const allBuiltInPlugins = Object.values(plugins);
const SunEditor = dynamic(() => import("suneditor-react"), { ssr: false });
import "suneditor/dist/css/suneditor.min.css";

type TemplateExportSettings = {
  pageSize?: "A4" | "A3" | "A5";
  orientation?: "portrait" | "landscape";
  font?: string;
  fontSizePt?: number;
  lang?: string;
  margins?: {
    topCm?: number;
    bottomCm?: number;
    leftCm?: number;
    rightCm?: number;
    headerCm?: number;
    footerCm?: number;
  };
  table?: {
    cantSplitRows?: boolean;
    addSpacingAfter?: boolean;
  };
};

type TemplateDocument = {
  id?: string;
  title: string;
  content_html: string;
  header_template_id?: string | null;
  footer_template_id?: string | null;
  template_type_id?: string;
  is_active?: boolean;
  export_settings?: TemplateExportSettings | null;
  form_id?: string | null; // ⬅️ NEW
  form_schema?: any; // resolved by backend, read-only
};

type DocTemplate = {
  id: string;
  name: string;
  kind: string;
  is_active?: boolean;
  header_image_name?: string | null;
  footer_image_name?: string | null;
};

type DocTemplateKind = {
  id: string;
  code: string;
  label: string;
};

export function TemplateEditor({
  id,
  templateKind,
  onSaved,
}: {
  id?: string;
  templateKind?: string;
  onSaved?: () => void;
}) {
  const [templateDocument, setTemplateDocument] = useState<TemplateDocument>({
    title: "",
    content_html: "",
    header_template_id: null,
    footer_template_id: null,
    template_type_id: undefined,
    is_active: true,
  });

  const [loading, setLoading] = useState<boolean>(!!id);

  const [templates, setTemplates] = useState<DocTemplate[]>([]);
  const [templatesLoading, setTemplatesLoading] = useState(false);

  const [kinds, setKinds] = useState<DocTemplateKind[]>([]);
  const [kindsLoading, setKindsLoading] = useState(false);

  const [forms, setForms] = useState<
    { id: string; code: string; name: string }[]
  >([]);
  const [formsLoading, setFormsLoading] = useState(false);

  useEffect(() => {
    (async () => {
      try {
        setFormsLoading(true);
        const res = await ApiCall.templateDocuments.getFormEditorList({
          onlyActive: "yes",
        });
        setForms(res.data ?? []);
      } catch (e) {
        console.error(e);
        toast.error("Űrlap sablonok betöltése sikertelen");
      } finally {
        setFormsLoading(false);
      }
    })();
  }, []);

  // store the suneditor instance (shape can differ, so we treat as any)
  const editorRef = useRef<any>(null);

  // LOAD TEMPLATE DOCUMENT (for edit)
  useEffect(() => {
    if (!id) {
      setLoading(false);
      return;
    }

    (async () => {
      try {
        const res = await ApiCall.templateDocuments.getItem(id);
        if (res.status !== 200) throw new Error("Failed to load template");
        setTemplateDocument(res.data);
      } catch (e) {
        console.error(e);
        toast.error("Sablon betöltése sikertelen");
      } finally {
        setLoading(false);
      }
    })();
  }, [id]);

  // LOAD DOC TEMPLATES (header/footer images)
  useEffect(() => {
    (async () => {
      try {
        setTemplatesLoading(true);
        const res = await ApiCall.docTemplates.getList({
          kind: templateKind,
        });
        setTemplates(res.data ?? []);
      } catch (e) {
        console.error(e);
      } finally {
        setTemplatesLoading(false);
      }
    })();
  }, [templateKind]);

  // LOAD DOC TEMPLATE KINDS (for react-select)
  useEffect(() => {
    (async () => {
      try {
        setKindsLoading(true);
        const res = await ApiCall.docTemplateKinds.getList({
          onlyActive: "yes",
        });
        setKinds(res.data ?? []);
      } catch (e) {
        console.error(e);
        toast.error("Sablon típusok betöltése sikertelen");
      } finally {
        setKindsLoading(false);
      }
    })();
  }, []);

  // const handleSave = async () => {
  //   try {
  //     // For new docs, enforce selecting a type
  //     if (!templateDocument.id && !templateDocument.template_type_id) {
  //       toast.error("Válaszd ki a sablon típusát");
  //       return;
  //     }

  //     const editor = editorRef.current;

  //     // Try a couple of shapes that suneditor-react might pass:
  //     let liveHtml: string | undefined;

  //     if (editor?.getContents) {
  //       liveHtml = editor.getContents(true);
  //     } else if (editor?.core?.getContents) {
  //       liveHtml = editor.core.getContents(true);
  //     }

  //     // Fallback to state if editor instance is missing for some reason
  //     const finalHtml = liveHtml ?? templateDocument.content_html;

  //     console.log("Saving template HTML:", finalHtml);

  //     const payload: any = {
  //       ...templateDocument,
  //       content_html: finalHtml,
  //     };

  //     // ✅ No more hard-coded kind_code = "generic" here.
  //     // Backend will use template_type_id.
  //     try {
  //       const res = await ApiCall.templateDocuments.upsertItem(payload);
  //       const saved = res.data;
  //       setTemplateDocument(saved);
  //       toast.success("Sablon mentve");
  //     } catch (error) {
  //       console.log("error", error);
  //       toast.error(error.response.data.error);
  //     }
  //   } catch (e) {
  //     console.error(e);
  //     toast.error("Mentés sikertelen");
  //   }
  // };
  const handleSave = async () => {
    try {
      if (!templateDocument.id && !templateDocument.template_type_id) {
        toast.error("Válaszd ki a sablon típusát");
        return;
      }

      const editor = editorRef.current;
      let liveHtml: string | undefined;

      if (editor?.getContents) {
        liveHtml = editor.getContents(true);
      } else if (editor?.core?.getContents) {
        liveHtml = editor.core.getContents(true);
      }

      const finalHtml = liveHtml ?? templateDocument.content_html;

      // 🚫 Don't send form_schema back from TemplateEditor
      const { form_schema, ...rest } = templateDocument;

      const payload: any = {
        ...rest,
        content_html: finalHtml,
      };

      try {
        const res = await ApiCall.templateDocuments.upsertItem(payload);
        const saved = res.data;
        setTemplateDocument(saved); // backend will still return effective form_schema
        toast.success("Sablon mentve");

        onSaved?.();
      } catch (error: any) {
        console.log("error", error);
        toast.error(error.response?.data?.error ?? "Mentés sikertelen");
      }
    } catch (e) {
      console.error(e);
      toast.error("Mentés sikertelen");
    }
  };

  const handleDownloadDocx = async () => {
    const faqId = templateDocument.id ?? id;
    if (!faqId) {
      toast.error("Először mentsd el a sablont");
      return;
    }

    try {
      const res = await ApiCall.templateDocuments.exportDocx(faqId);
      const blob = new Blob([res.data], {
        type: "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
      });

      const url = window.URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;

      const slug = slugifyTitle(templateDocument.title);
      const ts = timestampForFilename();
      a.download = `${slug}-${ts}.docx`;

      document.body.appendChild(a);
      a.click();
      a.remove();
      window.URL.revokeObjectURL(url);
    } catch (e) {
      console.error(e);
      toast.error("DOCX letöltés sikertelen");
    }
  };

  const headerTemplate = templates.find(
    (t) => t.id === templateDocument.header_template_id
  );
  const footerTemplate = templates.find(
    (t) => t.id === templateDocument.footer_template_id
  );

  const headerPreviewUrl = headerTemplate?.header_image_name
    ? ApiCall.mediaStore.getResized(
        "/12/3/0",
        headerTemplate.header_image_name,
        {
          format: "webp",
        }
      )
    : null;

  const footerPreviewUrl = footerTemplate?.footer_image_name
    ? ApiCall.mediaStore.getResized(
        "/12/3/0",
        footerTemplate.footer_image_name,
        {
          format: "webp",
        }
      )
    : null;

  const handlePrint = () => {
    const templateId = templateDocument.id ?? id;
    if (!templateId) {
      toast.error("Először mentsd el a sablont");
      return;
    }

    const baseURL = apiClient.defaults.baseURL?.replace(/\/$/, "") ?? "";
    const url = `${baseURL}/api/admin/template-documents/${templateId}/print`;
    window.open(url, "_blank");
  };

  if (loading) return <p>Betöltés…</p>;

  // react-select options for doc_template_kinds
  const kindOptions = kinds.map((k) => ({
    value: k.id,
    label: `${k.label} (${k.code})`,
  }));

  const selectedKindOption =
    kindOptions.find(
      (opt) => opt.value === templateDocument.template_type_id
    ) ?? null;
  console.log("templateDocument", templateDocument);
  return (
    <div className="max-w-7xl mx-auto space-y-4 py-4 px-2 bg-white">
      {/* Title + Type row */}
      <div className="flex flex-wrap gap-4 w-full bg-white p-2">
        <div className="flex-1 min-w-[260px] flex flex-col text-xs text-gray-600">
          <span className="mb-1">Sablon elnevezése</span>
          <input
            type="text"
            value={templateDocument.title}
            onChange={(e) =>
              setTemplateDocument({
                ...templateDocument,
                title: e.target.value,
              })
            }
            placeholder="Sablon cím (pl. Megrendelő, VPN hibaelhárítás stb.)"
            className="w-full px-3 py-2 border rounded text-lg"
          />
        </div>

        <div className="flex-1 min-w-[260px] flex flex-col text-xs text-gray-600">
          <span className="mb-1">Sablon típus</span>
          <Select
            classNamePrefix="react-select"
            className="z-50"
            isClearable
            isSearchable
            placeholder="Válassz sablon típust…"
            options={kindOptions}
            value={selectedKindOption}
            isLoading={kindsLoading}
            onChange={(opt: any) =>
              setTemplateDocument((prev) => ({
                ...prev,
                template_type_id: opt?.value,
              }))
            }
          />
        </div>

        <div className="flex-1 min-w-[260px] flex flex-col text-xs text-gray-600">
          <span className="mb-1">Űrlap sablon (opcionális)</span>
          <select
            className="border rounded px-3 py-2 text-sm"
            value={templateDocument.form_id ?? ""}
            onChange={(e) =>
              setTemplateDocument((prev) => ({
                ...prev,
                form_id: e.target.value || null,
              }))
            }>
            <option value="">— Nincs hozzárendelve —</option>
            {forms.map((f) => (
              <option key={f.id} value={f.id}>
                {f.name} ({f.code})
              </option>
            ))}
          </select>
          {formsLoading && (
            <span className="text-[11px] text-gray-500 mt-1">
              Űrlapok betöltése…
            </span>
          )}
        </div>
      </div>

      {/* Template selectors (header/footer) */}
      <div className="w-full flex flex-wrap gap-1 justify-between items-start bg-white p-2">
        <div className="flex flex-col text-xs text-gray-600 w-full max-w-[49%]">
          <span className="mb-1">Fejléc sablon</span>
          <select
            className="border border-gray-300 rounded px-2 py-1 text-sm min-w-[220px]"
            value={templateDocument.header_template_id ?? ""}
            onChange={(e) =>
              setTemplateDocument((prev) => ({
                ...prev,
                header_template_id: e.target.value || null,
              }))
            }>
            <option value="">— Nincs —</option>
            {templates.map((t) => (
              <option key={t.id} value={t.id}>
                {t.name}
              </option>
            ))}
          </select>
          {headerPreviewUrl && (
            <div className="mt-2 border border-gray-300 rounded overflow-hidden bg-gray-50">
              <img
                src={headerPreviewUrl}
                alt={`Fejléc sablon: ${headerTemplate?.name}`}
                className="w-full h-auto block"
              />
            </div>
          )}
        </div>

        <div className="flex flex-col text-xs text-gray-600 w-full max-w-[49%]">
          <span className="mb-1">Lábléc sablon</span>
          <select
            className="border border-gray-300 rounded px-2 py-1 text-sm min-w-[220px]"
            value={templateDocument.footer_template_id ?? ""}
            onChange={(e) =>
              setTemplateDocument((prev) => ({
                ...prev,
                footer_template_id: e.target.value || null,
              }))
            }>
            <option value="">— Nincs —</option>
            {templates.map((t) => (
              <option key={t.id} value={t.id}>
                {t.name}
              </option>
            ))}
          </select>
          {footerPreviewUrl && (
            <div className="mt-2 border border-gray-300 rounded overflow-hidden bg-gray-50">
              <img
                src={footerPreviewUrl}
                alt={`Lábléc sablon: ${footerTemplate?.name}`}
                className="w-full h-auto block"
              />
            </div>
          )}
        </div>

        {templatesLoading && (
          <span className="text-xs text-gray-500">Sablonok betöltése…</span>
        )}
      </div>
      {templateDocument.form_schema &&
        Array.isArray(templateDocument.form_schema) && (
          <div className="mt-3 border border-dashed border-gray-300 rounded p-2 text-xs">
            <p className="font-semibold mb-1">Elérhető mezők az űrlapból:</p>
            <ul className="space-y-1">
              {templateDocument.form_schema.map((field: any) => (
                <li key={field.name}>
                  <span className="font-mono bg-white px-1 rounded">
                    {`{{${field.name}}}`}
                  </span>{" "}
                  – {field.label} ({field.type})
                  {field.type === "checkbox" && (
                    <>
                      {" "}
                      · jelölőnégyzet token:{" "}
                      <span className="font-mono bg-white px-1 rounded">
                        {`{{check ${field.name}}}`}
                      </span>
                    </>
                  )}
                </li>
              ))}
            </ul>
          </div>
        )}

      <div className="w-full flex flex-wrap justify-between gap-1">
        {/* DOCX settings */}
        <div className="border border-gray-300 rounded p-3 max-md:w-full max-w-[20%] w-full bg-white">
          <h3 className="text-sm font-semibold">DOCX beállítások</h3>

          <div className="grid grid-cols-1 md:grid-cols-1 gap-3 text-sm">
            <div className="flex flex-col">
              <label className="mb-1">Papírméret</label>
              <select
                className="border rounded px-2 py-1"
                value={templateDocument.export_settings?.pageSize ?? "A4"}
                onChange={(e) =>
                  setTemplateDocument((prev) => ({
                    ...prev,
                    export_settings: {
                      ...prev.export_settings,
                      pageSize: e.target.value as "A4" | "A3" | "A5",
                    },
                  }))
                }>
                <option value="A3">A3</option>
                <option value="A4">A4</option>
                <option value="A5">A5</option>
              </select>
            </div>

            <div className="flex flex-col">
              <label className="mb-1">Tájolás</label>
              <select
                className="border rounded px-2 py-1"
                value={
                  templateDocument.export_settings?.orientation ?? "portrait"
                }
                onChange={(e) =>
                  setTemplateDocument((prev) => ({
                    ...prev,
                    export_settings: {
                      ...prev.export_settings,
                      orientation: e.target.value as "portrait" | "landscape",
                    },
                  }))
                }>
                <option value="portrait">Álló</option>
                <option value="landscape">Fekvő</option>
              </select>
            </div>

            <div className="flex flex-col">
              <label className="mb-1">Felső margó (cm)</label>
              <input
                type="number"
                step="0.1"
                className="border rounded px-2 py-1"
                value={templateDocument.export_settings?.margins?.topCm ?? 4}
                onChange={(e) =>
                  setTemplateDocument((prev) => ({
                    ...prev,
                    export_settings: {
                      ...prev.export_settings,
                      margins: {
                        ...prev.export_settings?.margins,
                        topCm: parseFloat(e.target.value),
                      },
                    },
                  }))
                }
              />
            </div>

            <div className="flex flex-col">
              <label className="mb-1">Alsó margó (cm)</label>
              <input
                type="number"
                step="0.1"
                className="border rounded px-2 py-1"
                value={
                  templateDocument.export_settings?.margins?.bottomCm ?? 2.25
                }
                onChange={(e) =>
                  setTemplateDocument((prev) => ({
                    ...prev,
                    export_settings: {
                      ...prev.export_settings,
                      margins: {
                        ...prev.export_settings?.margins,
                        bottomCm: parseFloat(e.target.value),
                      },
                    },
                  }))
                }
              />
            </div>
          </div>
        </div>

        {/* Editor */}
        <div className="border border-gray-300 p-2 rounded max-md:w-full max-w-[78%] min-h-[600px] w-full bg-white">
          <SunEditor
            setDefaultStyle="min-height: 600px"
            getSunEditorInstance={(editor: any) => {
              editorRef.current = editor;
            }}
            setContents={templateDocument.content_html}
            onChange={(content: string) =>
              setTemplateDocument((prev) => ({
                ...prev,
                content_html: content,
              }))
            }
            setOptions={{
              height: "300px",
              font: [
                "Arial",
                "Calibri",
                "Times New Roman",
                "Courier New",
                "Verdana",
                "Roboto",
              ],
              fontSize: [8, 10, 12, 14, 16, 18, 24, 32],
              attributesWhitelist: {
                all: "style|data-compact-.+",
              },
              plugins: [...allBuiltInPlugins, compactTablePlugin as any],
              buttonList: [
                ["undo", "redo"],
                ["bold", "italic", "underline", "strike"],
                ["fontSize", "formatBlock"],
                ["fontColor", "hiliteColor"],
                ["align", "list", "lineHeight"],
                ["table", "compactTable"],
                ["link", "image"],
                ["removeFormat"],
                ["fullScreen", "showBlocks", "codeView"],
              ],
            }}
          />
        </div>
      </div>

      {/* Actions */}
      <div className="w-full flex gap-2 justify-end">
        <label className="flex items-center gap-2 text-sm text-gray-700 mr-auto">
          <input
            type="checkbox"
            className="w-[1.25rem] h-[1.25rem]"
            checked={templateDocument.is_active ?? true}
            onChange={(e) =>
              setTemplateDocument((prev) => ({
                ...prev,
                is_active: e.target.checked,
              }))
            }
          />
          Aktív sablon
        </label>
        <button
          onClick={handleSave}
          className="px-4 py-2 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white text-sm cursor-pointer">
          Mentés
        </button>
        <button
          onClick={handleDownloadDocx}
          className="px-4 py-2 rounded border border-gray-300 text-sm hover:bg-gray-50 cursor-pointer">
          DOCX letöltés
        </button>
        <button
          onClick={handlePrint}
          className="px-4 py-2 rounded border border-gray-300 text-sm hover:bg-gray-50 cursor-pointer">
          Nyomtatás
        </button>
      </div>
    </div>
  );
}
