// components/shared/texteditor/suneditorCompactTablePlugin.ts

const compactTablePlugin = {
  name: "compactTable",
  display: "submenu",
  title: "Kompakt táblázat (Word export)",
  innerHTML: "<span>Tbl</span>",

  _submenu: null as HTMLElement | null,

  add(core: any, targetElement: HTMLElement) {
    const util = core.util;

    const submenu = util.createElement("div");
    submenu.className = "se-submenu se-list-layer";
    submenu.innerHTML = `
      <div class="se-list-inner">
        <div class="se-list-basic">

          <div class="se-list-title">Kompakt rács</div>

          <div class="se-list-item"
               data-preset="compact-tight"
               data-row-height="0.5cm"
               data-border-width="1px"
               data-border-color="#000000"
               data-padding="0 0.5pt">
            0.5cm | 1px fekete
          </div>

          <div class="se-list-item"
               data-preset="compact-thin"
               data-row-height="0.6cm"
               data-border-width="0.75pt"
               data-border-color="#000000"
               data-padding="0 0.5pt">
            0.6cm | 0.75pt fekete
          </div>

          <div class="se-list-item"
               data-preset="compact-grey"
               data-row-height="0.7cm"
               data-border-width="0.5pt"
               data-border-color="#444444"
               data-padding="0 0.75pt">
            0.7cm | 0.5pt szürke
          </div>

          <div class="se-list-title">Speciális</div>

          <div class="se-list-item"
               data-preset="no-border"
               data-row-height="0.6cm"
               data-padding="0 0.75pt">
            Nincs keret (láthatatlan táblázat)
          </div>

          <div class="se-list-item"
               data-preset="reset">
            Stílusok törlése (alaphelyzet)
          </div>

        </div>
      </div>
    `;

    core.initMenuTarget(this.name, targetElement, submenu);
    this._submenu = submenu;

    submenu.addEventListener("click", (e: Event) => {
      const target = e.target as HTMLElement | null;
      if (!target) return;

      const item = target.closest("div.se-list-item") as HTMLElement | null;
      if (!item) return;

      const preset = item.getAttribute("data-preset") || "compact-tight";

      if (preset === "reset") {
        this.resetTableStyles(core);
      } else {
        const rowHeight = item.getAttribute("data-row-height") || "0.6cm";
        const borderWidth = item.getAttribute("data-border-width") || "";
        const borderColor = item.getAttribute("data-border-color") || "";
        const padding = item.getAttribute("data-padding") || "0 0.5pt";

        this.applyTableStyles(core, {
          mode: preset === "no-border" ? "noBorder" : "grid",
          rowHeight,
          borderWidth,
          borderColor,
          padding,
        });
      }

      core.submenuOff();
    });
  },

  action(this: any) {
    this.submenuOn(this);
  },

  resetTableStyles(core: any) {
    const util = core.util;
    const selectionNode = core.getSelectionNode();
    const table = util.getParentElement(
      selectionNode,
      "table"
    ) as HTMLTableElement | null;

    if (!table) {
      alert("Nincs kijelölt táblázat.");
      return;
    }

    // remove our markers
    table.removeAttribute("data-compact-mode");
    table.removeAttribute("data-compact-row-height");
    table.removeAttribute("data-compact-border-width");
    table.removeAttribute("data-compact-border-color");
    table.removeAttribute("data-compact-padding");

    table.removeAttribute("style");
    table.querySelectorAll("tr").forEach((tr) => tr.removeAttribute("style"));
    table
      .querySelectorAll("th, td")
      .forEach((cell) => (cell as HTMLElement).removeAttribute("style"));

    if (core?.history && typeof core.history.push === "function") {
      core.history.push(false);
    }
  },

  applyTableStyles(
    core: any,
    opts: {
      mode: "grid" | "noBorder";
      rowHeight: string;
      borderWidth?: string;
      borderColor?: string;
      padding: string;
    }
  ) {
    const util = core.util;

    const selectionNode = core.getSelectionNode();
    const table = util.getParentElement(
      selectionNode,
      "table"
    ) as HTMLTableElement | null;

    if (!table) {
      alert("Nincs kijelölt táblázat.");
      return;
    }

    const { mode, rowHeight, borderWidth, borderColor, padding } = opts;

    // 🔹 persist config in data-* so backend can read it
    table.setAttribute("data-compact-mode", mode);
    table.setAttribute("data-compact-row-height", rowHeight);
    table.setAttribute("data-compact-padding", padding);
    if (borderWidth) {
      table.setAttribute("data-compact-border-width", borderWidth);
    } else {
      table.removeAttribute("data-compact-border-width");
    }
    if (borderColor) {
      table.setAttribute("data-compact-border-color", borderColor);
    } else {
      table.removeAttribute("data-compact-border-color");
    }

    // 🔹 still apply visual styles in the editor for user feedback
    const baseTableStyles = [
      "border-collapse:collapse",
      "border-spacing:0",
      "width:100%",
      "margin-top:0",
      "margin-bottom:0",
    ];
    table.setAttribute("style", baseTableStyles.join(";"));

    const rows = Array.from(table.querySelectorAll("tr"));
    rows.forEach((tr) => {
      (tr as HTMLElement).setAttribute("style", `height:${rowHeight}`);
    });

    const cells = Array.from(table.querySelectorAll("th, td"));
    cells.forEach((cell) => {
      const el = cell as HTMLElement;

      el.querySelectorAll("br").forEach((br) => br.remove());

      el.querySelectorAll("div").forEach((div) => {
        const txt = div.textContent?.trim() ?? "";
        const hasBlocks = div.querySelector(
          "table, p, div, ul, ol, h1, h2, h3, h4, h5, h6"
        );

        if (!hasBlocks && txt === "") {
          div.remove();
        } else if (!hasBlocks) {
          while (div.firstChild) {
            div.parentNode?.insertBefore(div.firstChild, div);
          }
          div.remove();
        }
      });

      Array.from(el.childNodes).forEach((node) => {
        if (node.nodeType === Node.TEXT_NODE) {
          if (!node.nodeValue || node.nodeValue.trim() === "") {
            el.removeChild(node);
          }
        }
      });

      const styleParts = [
        "vertical-align:top",
        `padding:${padding}`,
        "line-height:1",
      ];

      if (mode === "grid") {
        styleParts.push(`border:${borderWidth} solid ${borderColor}`);
      } else if (mode === "noBorder") {
        styleParts.push("border:none");
      }

      el.setAttribute("style", styleParts.join(";"));
    });

    if (core?.history && typeof core.history.push === "function") {
      core.history.push(false);
    }
  },
};

export default compactTablePlugin;
