"use client";

import React, { useEffect, useRef } from "react";
import * as jsondiffpatch from "jsondiffpatch";
import * as htmlFormatter from "jsondiffpatch/formatters/html";

// 🧠 Generate consistent hash to detect object changes
const objectHash = (obj) =>
  obj?._id || obj?.id || obj?.value || JSON.stringify(obj).slice(0, 50);

// 🧹 Normalize populated arrays: remove duplicate IDs if full object is present
function normalizeArray(arr) {
  if (!Array.isArray(arr)) return arr;
  const ids = new Set(
    arr.filter((x) => typeof x === "object" && x?._id).map((x) => x._id)
  );
  return arr.filter((x) => typeof x === "object" || !ids.has(x));
}

// 🧼 Clean nested data
function deepClean(obj) {
  if (Array.isArray(obj)) return normalizeArray(obj.map(deepClean));
  if (obj && typeof obj === "object") {
    const clean = {};
    for (const key in obj) {
      clean[key] = deepClean(obj[key]);
    }
    return clean;
  }
  return obj;
}

const diffpatcher = jsondiffpatch.create({
  objectHash,
  arrays: { detectMove: false },
  propertyFilter: (name) => !["__v", "updatedAt", "createdAt"].includes(name), // Optional filter
});

export default function DiffViewer({ oldData, newData }) {
  const ref = useRef(null);

  useEffect(() => {
    if (!ref.current || !oldData || !newData) return;

    const left = deepClean(oldData);
    const right = deepClean(newData);
    const delta = diffpatcher.diff(left, right);

    if (delta) {
      const html = htmlFormatter.format(delta, left, {
        showUnchanged: false,
        objectHash,
      });

      ref.current.innerHTML = html;
    } else {
      ref.current.innerHTML = "<p>Nem található eltérés.</p>";
    }
  }, [oldData, newData]);

  return (
    <div
      ref={ref}
      className="text-sm jsondiffpatch-delta jsondiffpatch-delta-formatters-side-by-side diff-viewer-wrapper"
    />
  );
}
