// src/components/forms/FormTemplateEditor.tsx
"use client";

import { useEffect, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

type FormField = {
  name: string;
  label: string;
  type: "text" | "textarea" | "checkbox" | "date" | "separator";
  required?: boolean;
  defaultValue?: any;
  description?: string; // 🔹 mező / szekció leírás
};

type DocForm = {
  id?: string;
  code: string;
  name: string;
  description?: string | null;
  schema: FormField[];
  is_active?: boolean;
};

export function FormTemplateEditor({ id }: { id?: string }) {
  const [form, setForm] = useState<DocForm>({
    code: "",
    name: "",
    description: "",
    schema: [],
    is_active: true,
  });

  const [loading, setLoading] = useState<boolean>(!!id);
  const [saving, setSaving] = useState(false);

  useEffect(() => {
    if (!id) {
      setLoading(false);
      return;
    }

    (async () => {
      try {
        setLoading(true);
        const res = await ApiCall.templateDocuments.getFormEditorItem(id);
        const data = res.data as DocForm;
        data.schema = (data.schema ?? []) as FormField[];
        setForm(data);
      } catch (e: any) {
        console.error(e);
        toast.error(e?.response?.data?.error ?? "Űrlap betöltése sikertelen");
      } finally {
        setLoading(false);
      }
    })();
  }, [id]);

  const addField = () => {
    setForm((prev) => ({
      ...prev,
      schema: [
        ...prev.schema,
        {
          name: `field_${prev.schema.length + 1}`,
          label: "Új mező",
          type: "text",
          required: false,
        },
      ],
    }));
  };

  const addSeparator = () => {
    setForm((prev) => ({
      ...prev,
      schema: [
        ...prev.schema,
        {
          name: `separator_${prev.schema.length + 1}`,
          label: "Szekció címe",
          type: "separator",
        },
      ],
    }));
  };

  const cloneField = (index: number) => {
    setForm((prev) => {
      const next = [...prev.schema];
      const original = next[index];
      if (!original) return prev;

      const baseName = original.name || `field_${index + 1}`;

      const clone: FormField = {
        ...original,
        name: `${baseName}_copy`,
      };

      next.splice(index + 1, 0, clone);

      return { ...prev, schema: next };
    });
  };

  const updateField = (index: number, patch: Partial<FormField>) => {
    setForm((prev) => {
      const next = [...prev.schema];
      next[index] = { ...next[index], ...patch };
      return { ...prev, schema: next };
    });
  };

  const removeField = (index: number) => {
    setForm((prev) => {
      const next = [...prev.schema];
      next.splice(index, 1);
      return { ...prev, schema: next };
    });
  };

  const moveField = (index: number, direction: "up" | "down") => {
    setForm((prev) => {
      const next = [...prev.schema];
      const targetIndex = direction === "up" ? index - 1 : index + 1;
      if (targetIndex < 0 || targetIndex >= next.length) return prev;
      const temp = next[index];
      next[index] = next[targetIndex];
      next[targetIndex] = temp;
      return { ...prev, schema: next };
    });
  };

  const handleSave = async () => {
    try {
      if (!form.code.trim() || !form.name.trim()) {
        toast.error("Kód és név megadása kötelező");
        return;
      }

      setSaving(true);
      const payload = { ...form };
      const res = await ApiCall.templateDocuments.upsertFormEditorItem(payload);
      setForm(res.data);
      toast.success("Űrlap sablon mentve");
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Mentés sikertelen");
    } finally {
      setSaving(false);
    }
  };

  if (loading) return <p className="p-4">Betöltés…</p>;

  return (
    <main className="max-w-7xl mx-auto p-4 space-y-4 bg-white">
      <header className="space-y-1">
        <h1 className="text-xl font-semibold">Űrlap sablon szerkesztése</h1>
        <p className="text-sm text-gray-600">
          Itt tudod összerakni a menthető / újrahasználható űrlap mezőket
          (form_schema).
        </p>
      </header>

      {/* Basic info */}
      <section className="border border-gray-200 rounded p-3 space-y-3">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">
          <div className="flex flex-col">
            <label className="mb-1 text-xs text-gray-600">
              Kód (egyedi azonosító)
            </label>
            <input
              type="text"
              className="border rounded px-2 py-1"
              value={form.code}
              onChange={(e) =>
                setForm((prev) => ({ ...prev, code: e.target.value }))
              }
            />
          </div>
          <div className="flex flex-col">
            <label className="mb-1 text-xs text-gray-600">
              Név (admin felület)
            </label>
            <input
              type="text"
              className="border rounded px-2 py-1"
              value={form.name}
              onChange={(e) =>
                setForm((prev) => ({ ...prev, name: e.target.value }))
              }
            />
          </div>
        </div>

        <div className="flex flex-col text-xs text-gray-600">
          <label className="mb-1">Leírás (opcionális)</label>
          <textarea
            className="border rounded px-2 py-1 text-sm"
            value={form.description ?? ""}
            onChange={(e) =>
              setForm((prev) => ({ ...prev, description: e.target.value }))
            }
          />
        </div>

        <label className="flex items-center gap-2 text-sm text-gray-700">
          <input
            type="checkbox"
            className="w-4 h-4"
            checked={form.is_active ?? true}
            onChange={(e) =>
              setForm((prev) => ({ ...prev, is_active: e.target.checked }))
            }
          />
          Aktív űrlap
        </label>
      </section>

      {/* Fields editor */}
      <section className="border border-gray-200 rounded p-3 space-y-3">
        <div className="flex justify-between items-center gap-2">
          <h2 className="text-sm font-semibold">Mezők</h2>
          <div className="flex gap-2">
            <button
              type="button"
              onClick={addSeparator}
              className="px-3 py-1 rounded bg-slate-500/80 text-white text-xs hover:bg-slate-600">
              + Új szekció / elválasztó
            </button>
            <button
              type="button"
              onClick={addField}
              className="px-3 py-1 rounded bg-emerald-600/80 text-white text-xs hover:bg-emerald-600">
              + Új mező
            </button>
          </div>
        </div>

        {!form.schema.length && (
          <p className="text-xs text-gray-500">
            Még nincs mező. Kattints a <strong>+ Új mező</strong> vagy{" "}
            <strong>+ Új szekció</strong> gombra.
          </p>
        )}

        <div className="space-y-2">
          {form.schema.map((field, idx) => (
            <div
              key={idx}
              className={`border border-gray-200 rounded px-3 py-2 flex flex-col gap-2 ${
                field.type === "separator" ? "bg-amber-50" : "bg-gray-50"
              }`}>
              <div className="flex justify-between items-center text-xs">
                <span className="font-semibold">
                  #{idx + 1} –{" "}
                  {field.type === "separator"
                    ? (field.label || "Szekció / elválasztó") + " (separator)"
                    : field.name || "névtelen mező"}
                </span>
                <div className="flex gap-2">
                  <button
                    type="button"
                    onClick={() => moveField(idx, "up")}
                    className="px-2 py-1 text-[11px] border rounded">
                    ↑
                  </button>
                  <button
                    type="button"
                    onClick={() => moveField(idx, "down")}
                    className="px-2 py-1 text-[11px] border rounded">
                    ↓
                  </button>
                  <button
                    type="button"
                    onClick={() => removeField(idx)}
                    className="px-2 py-1 text-[11px] border border-red-300 text-red-700 rounded">
                    Törlés
                  </button>
                </div>
              </div>

              {/* Separator editor */}
              {field.type === "separator" ? (
                <>
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-2 text-xs">
                    <div className="flex flex-col">
                      <label className="mb-1">Szekció címe (label)</label>
                      <input
                        type="text"
                        className="border rounded px-2 py-1 text-sm"
                        value={field.label}
                        onChange={(e) =>
                          updateField(idx, { label: e.target.value })
                        }
                        placeholder="pl. Személyes adatok"
                      />
                    </div>
                    <div className="flex flex-col">
                      <label className="mb-1">Belső név (opcionális)</label>
                      <input
                        type="text"
                        className="border rounded px-2 py-1 text-sm"
                        value={field.name}
                        onChange={(e) =>
                          updateField(idx, { name: e.target.value.trim() })
                        }
                        placeholder="pl. section_personal"
                      />
                    </div>
                    <div className="flex flex-col items-start md:items-end justify-between text-[11px] text-gray-500 gap-1">
                      <span>
                        Ez a sor csak vizuális elválasztó, nem generál mezőt.
                      </span>
                      <button
                        type="button"
                        onClick={() => cloneField(idx)}
                        className="mt-1 inline-flex items-center px-2 py-1 border rounded text-[11px] text-slate-700 hover:bg-slate-100">
                        Duplikálás
                      </button>
                    </div>
                  </div>

                  {/* 🔹 Separator description */}
                  <div className="flex flex-col text-xs text-gray-700">
                    <label className="mb-1">
                      Szekció leírása (kitöltéskor súgó)
                    </label>
                    <textarea
                      className="border rounded px-2 py-1 text-sm min-h-[40px]"
                      value={field.description ?? ""}
                      onChange={(e) =>
                        updateField(idx, { description: e.target.value })
                      }
                      placeholder="pl. Ide kerülnek a dolgozó személyes adatai."
                    />
                  </div>
                </>
              ) : (
                <>
                  {/* Normal field editor */}
                  <div className="grid grid-cols-1 md:grid-cols-4 gap-2 text-xs">
                    <div className="flex flex-col">
                      <label className="mb-1">Mező neve (token)</label>
                      <input
                        type="text"
                        className="border rounded px-2 py-1 text-sm"
                        value={field.name}
                        onChange={(e) =>
                          updateField(idx, { name: e.target.value.trim() })
                        }
                        placeholder="pl. employee_name"
                      />
                    </div>
                    <div className="flex flex-col">
                      <label className="mb-1">Felirat (label)</label>
                      <input
                        type="text"
                        className="border rounded px-2 py-1 text-sm"
                        value={field.label}
                        onChange={(e) =>
                          updateField(idx, { label: e.target.value })
                        }
                        placeholder="pl. Dolgozó neve"
                      />
                    </div>
                    <div className="flex flex-col">
                      <label className="mb-1">Típus</label>
                      <select
                        className="border rounded px-2 py-1 text-sm"
                        value={field.type}
                        onChange={(e) =>
                          updateField(idx, {
                            type: e.target.value as FormField["type"],
                          })
                        }>
                        <option value="text">Szöveg</option>
                        <option value="textarea">Többsoros szöveg</option>
                        <option value="checkbox">Jelölőnégyzet</option>
                        <option value="date">Dátum</option>
                        <option value="separator">Elválasztó / szekció</option>
                      </select>
                    </div>
                    <div className="flex flex-col">
                      <label className="mb-1">Alapértelmezés / Kötelező</label>
                      <div className="flex items-center gap-2">
                        {field.type === "checkbox" ? (
                          <input
                            type="checkbox"
                            className="w-4 h-4"
                            checked={!!field.defaultValue}
                            onChange={(e) =>
                              updateField(idx, {
                                defaultValue: e.target.checked,
                              })
                            }
                          />
                        ) : (
                          <input
                            type="text"
                            className="border rounded px-2 py-1 text-sm flex-1"
                            value={field.defaultValue ?? ""}
                            onChange={(e) =>
                              updateField(idx, {
                                defaultValue: e.target.value,
                              })
                            }
                          />
                        )}
                        <label className="inline-flex items-center gap-1 text-xs">
                          <input
                            type="checkbox"
                            className="w-4 h-4"
                            checked={!!field.required}
                            onChange={(e) =>
                              updateField(idx, { required: e.target.checked })
                            }
                          />
                          Kötelező
                        </label>
                      </div>
                    </div>
                  </div>

                  {/* 🔹 Field description */}
                  <div className="flex flex-col text-xs text-gray-700">
                    <label className="mb-1">
                      Mező leírása (kitöltéskor súgó)
                    </label>
                    <textarea
                      className="border rounded px-2 py-1 text-sm min-h-[40px]"
                      value={field.description ?? ""}
                      onChange={(e) =>
                        updateField(idx, { description: e.target.value })
                      }
                      placeholder="pl. Írd be a dolgozó teljes nevét."
                    />
                  </div>

                  <p className="text-[11px] text-gray-500">
                    Használat sablonban:{" "}
                    <code className="bg-gray-200 px-1 rounded">
                      {"{{" + field.name + "}}"}
                    </code>
                    {field.type === "checkbox" && (
                      <>
                        {" "}
                        vagy{" "}
                        <code className="bg-gray-200 px-1 rounded">
                          {"{{check " + field.name + "}}"}
                        </code>
                      </>
                    )}
                  </p>
                </>
              )}
            </div>
          ))}
        </div>

        <div className="w-full flex justify-end gap-2">
          <button
            type="button"
            onClick={addSeparator}
            className="flex px-3 py-1 rounded bg-slate-500/80 text-white text-xs hover:bg-slate-600 self-end">
            + Új szekció / elválasztó
          </button>
          <button
            type="button"
            onClick={addField}
            className="flex px-3 py-1 rounded bg-emerald-600/80 text-white text-xs hover:bg-emerald-600 self-end">
            + Új mező
          </button>
        </div>
      </section>

      <div className="flex justify-end">
        <button
          type="button"
          onClick={handleSave}
          disabled={saving}
          className="px-4 py-2 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white text-sm disabled:bg-emerald-300">
          {saving ? "Mentés…" : "Űrlap sablon mentése"}
        </button>
      </div>
    </main>
  );
}
