// src/components/forms/FormTemplateEditor.tsx
"use client";

import { useEffect, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";

type FormField = {
  name: string;
  label: string;
  type: "text" | "textarea" | "checkbox" | "date";
  required?: boolean;
  defaultValue?: any;
};

type DocForm = {
  id?: string;
  code: string;
  name: string;
  description?: string | null;
  schema: FormField[];
  is_active?: boolean;
};

type Props = {
  id?: string;
  onSaved?: (doc: DocForm) => void; // <<< NEW
};

export default function HrFormTemplateEditor({ id, onSaved }: Props) {
  const [form, setForm] = useState<DocForm>({
    code: "",
    name: "",
    description: "",
    schema: [],
    is_active: true,
  });

  const [loading, setLoading] = useState<boolean>(!!id);
  const [saving, setSaving] = useState(false);

  useEffect(() => {
    if (!id) {
      setForm({
        code: "",
        name: "",
        description: "",
        schema: [],
        is_active: true,
      });
      setLoading(false);
      return;
    }

    (async () => {
      try {
        setLoading(true);
        const res = await ApiCall.templateDocuments.getFormEditorItem(id);
        const data = res.data as DocForm;
        data.schema = (data.schema ?? []) as FormField[];
        setForm(data);
      } catch (e: any) {
        console.error(e);
        toast.error(e?.response?.data?.error ?? "Űrlap betöltése sikertelen");
      } finally {
        setLoading(false);
      }
    })();
  }, [id]);

  const addField = () => {
    setForm((prev) => ({
      ...prev,
      schema: [
        ...prev.schema,
        {
          name: `field_${prev.schema.length + 1}`,
          label: "Új mező",
          type: "text",
          required: false,
        },
      ],
    }));
  };

  const updateField = (index: number, patch: Partial<FormField>) => {
    setForm((prev) => {
      const next = [...prev.schema];
      next[index] = { ...next[index], ...patch };
      return { ...prev, schema: next };
    });
  };

  const removeField = (index: number) => {
    setForm((prev) => {
      const next = [...prev.schema];
      next.splice(index, 1);
      return { ...prev, schema: next };
    });
  };

  const moveField = (index: number, direction: "up" | "down") => {
    setForm((prev) => {
      const next = [...prev.schema];
      const targetIndex = direction === "up" ? index - 1 : index + 1;
      if (targetIndex < 0 || targetIndex >= next.length) return prev;
      const temp = next[index];
      next[index] = next[targetIndex];
      next[targetIndex] = temp;
      return { ...prev, schema: next };
    });
  };

  const handleSave = async () => {
    try {
      if (!form.code.trim() || !form.name.trim()) {
        toast.error("Kód és név megadása kötelező");
        return;
      }

      setSaving(true);
      const payload = { ...form };
      const res = await ApiCall.templateDocuments.upsertFormEditorItem(payload);
      const saved = res.data as DocForm;
      setForm(saved);
      onSaved?.(saved); // <<< notify parent
      toast.success("Űrlap sablon mentve");
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Mentés sikertelen");
    } finally {
      setSaving(false);
    }
  };

  if (loading) return <p className="p-4">Betöltés…</p>;

  return (
    <main className="max-w-7xl mx-auto p-4 space-y-4 bg-white">
      {/* ... everything else unchanged ... */}
      {/* fields editor etc. */}
    </main>
  );
}
