// src/components/template-documents/TemplateFillAndSend.tsx
"use client";

import { useEffect, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import Select from "react-select/async";
import { useMyApprovals } from "@/hooks/useMyApprovalsBadge";

type TemplateExportSettings = {
  pageSize?: "A4" | "A3" | "A5";
  orientation?: "portrait" | "landscape";
  font?: string;
  fontSizePt?: number;
  lang?: string;
  margins?: {
    topCm?: number;
    bottomCm?: number;
    leftCm?: number;
    rightCm?: number;
    headerCm?: number;
    footerCm?: number;
  };
  table?: {
    cantSplitRows?: boolean;
    addSpacingAfter?: boolean;
  };
};

type FormField = {
  name: string;
  label: string;
  type: "text" | "textarea" | "checkbox" | "date" | "separator";
  required?: boolean;
  defaultValue?: any;
  description?: string;
};

type TemplateDocumentWithForm = {
  id: string;
  title: string;
  content_html: string;
  form_schema?: FormField[] | null;
  export_settings?: TemplateExportSettings | null;
};

type RecipientOption = {
  value: string; // recipient id
  label: string; // "Name <email>"
};

type ApproverOption = {
  value: string;
  label: string;
  name: string;
  role?: string | null;
};

type TemplateFillAndSendProps = {
  id: string;
  personId?: string;
  companyId?: string | null;
};

type ApprovalMode = "send_immediately" | "request_approval";

// helper: build initial values from schema
function buildInitialValues(
  schema: FormField[],
  mode: "defaults" | "empty"
): Record<string, any> {
  const values: Record<string, any> = {};

  schema.forEach((field) => {
    if (field.type === "separator") return;

    if (mode === "defaults" && field.defaultValue !== undefined) {
      values[field.name] = field.defaultValue;
      return;
    }

    if (field.type === "checkbox") {
      values[field.name] = false;
    } else {
      values[field.name] = "";
    }
  });

  return values;
}

export function TemplateFillAndSend({
  id,
  personId,
  companyId,
}: TemplateFillAndSendProps) {
  const [template, setTemplate] = useState<TemplateDocumentWithForm | null>(
    null
  );
  const [loading, setLoading] = useState(true);
  const [sending, setSending] = useState(false);

  const [formValues, setFormValues] = useState<Record<string, any>>({});
  const [selectedRecipients, setSelectedRecipients] = useState<
    RecipientOption[]
  >([]);
  const [selectedApprovers, setSelectedApprovers] = useState<ApproverOption[]>(
    []
  );
  const { refreshSummary, loadList } = useMyApprovals();
  const [emailSubject, setEmailSubject] = useState("");
  const [emailBody, setEmailBody] = useState("");

  const [mode, setMode] = useState<ApprovalMode>("send_immediately");
  const [sendEmailOnApproval, setSendEmailOnApproval] = useState(true);

  // load template
  useEffect(() => {
    (async () => {
      try {
        setLoading(true);
        const res = await ApiCall.templateDocuments.getItem(id);
        if (res.status !== 200) throw new Error("Failed to load template");

        const data = res.data as TemplateDocumentWithForm;
        setTemplate(data);

        setEmailSubject(data.title ?? "");
        setEmailBody("Csatolva találod a dokumentumot.");

        const schema = (data.form_schema ?? []) as FormField[];
        const initial = buildInitialValues(schema, "defaults");
        setFormValues(initial);
      } catch (e: any) {
        console.error(e);
        toast.error(
          e?.response?.data?.error ?? "Sablon betöltése sikertelen (Kitöltés)"
        );
      } finally {
        setLoading(false);
      }
    })();
  }, [id]);

  const schema: FormField[] = (template?.form_schema as any) ?? [];

  const handleFieldChange = (name: string, value: any) => {
    setFormValues((prev) => ({ ...prev, [name]: value }));
  };

  const clearToEmpty = () => {
    const emptyValues = buildInitialValues(schema, "empty");
    setFormValues(emptyValues);
  };

  const reapplyDefaults = () => {
    const defaults = buildInitialValues(schema, "defaults");
    setFormValues(defaults);
  };

  // recipients async loader (you already had this)
  const loadRecipients = async (
    inputValue: string
  ): Promise<RecipientOption[]> => {
    try {
      const res = await ApiCall.emailRegistry.emailRecipients({
        search: inputValue,
      });
      const data = res.data ?? [];
      return data.map((r: any) => ({
        value: r.id,
        label: `${r.name} <${r.email}>`,
      }));
    } catch (e) {
      console.error(e);
      return [];
    }
  };

  // approvers async loader – adjust to your actual endpoint
  const loadApprovers = async (
    inputValue: string
  ): Promise<ApproverOption[]> => {
    try {
      // TODO: wire this to your actual backend route
      const res = await ApiCall.hr.searchApprovers({ search: inputValue });
      const data = res.data ?? [];
      return data.map((u: any) => ({
        value: u.id,
        label: u.role ? `${u.name} (${u.role})` : u.name,
        name: u.name, // <-- add raw name
        role: u.role ?? null,
      }));
    } catch (e) {
      console.error(e);
      return [];
    }
  };

  const handleSubmit = async () => {
    if (!template) return;

    if (!schema.length) {
      const cont = confirm(
        "Ehhez a sablonhoz még nincs űrlap (form_schema). Biztosan folytatod?"
      );
      if (!cont) return;
    }

    // we only *require* recipients if an email will be sent in this flow
    const emailWillBeSent =
      mode === "send_immediately" || sendEmailOnApproval === true;

    if (emailWillBeSent && !selectedRecipients.length) {
      toast.error("Válaszd ki legalább egy címzettet");
      return;
    }

    if (emailWillBeSent) {
      if (!emailSubject.trim() || !emailBody.trim()) {
        toast.error("Add meg az email tárgyát és szövegét");
        return;
      }
    }

    if (mode === "request_approval" && !selectedApprovers.length) {
      toast.error("Válassz legalább egy jóváhagyót");
      return;
    }

    setSending(true);
    try {
      const recipient_ids = selectedRecipients.map((o) => o.value);

      const payload: any = {
        values: formValues,
        recipient_ids,
        email_subject: emailSubject,
        email_body: emailBody,
        mode, // "send_immediately" | "request_approval"
        send_email_on_approval: sendEmailOnApproval,
        approvers: selectedApprovers.map((a) => ({
          user_id: a.value,
          name: a.name, // <-- use raw name, not label
          role: a.role ?? null,
        })),
      };

      if (personId) payload.person_id = personId;
      if (companyId) payload.company_id = companyId;

      const res = await ApiCall.templateDocuments.fillAndSend(id, payload);

      if (res.status === 200) {
        if (mode === "send_immediately") {
          toast.success("Dokumentum elküldve");
        } else {
          await Promise.all([refreshSummary(), loadList()]);
          toast.success("Dokumentum létrejött és jóváhagyásra vár");
        }
      } else {
        toast.error("Küldés sikertelen");
      }
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Küldés közben hiba történt");
    } finally {
      setSending(false);
    }
  };

  if (loading) return <p className="p-4">Betöltés…</p>;
  if (!template)
    return <p className="p-4 text-red-600">Sablon nem található</p>;

  return (
    <main className="max-w-5xl mx-auto p-4 space-y-6 bg-white">
      <header className="space-y-1">
        <h1 className="text-xl font-semibold">
          {template.title} – Kitöltés és küldés
        </h1>
        <p className="text-sm text-gray-600">
          Töltsd ki az alábbi mezőket, válaszd ki a címzetteket, majd döntsd el,
          hogy azonnal küldöd-e emailben, vagy előbb jóváhagyásra küldöd.
        </p>
      </header>

      {/* Mode switch */}
      <section className="space-y-2 border border-gray-200 rounded p-3">
        <h2 className="text-sm font-semibold">Küldési mód</h2>
        <div className="flex flex-col gap-2 text-xs text-gray-700">
          <label className="flex items-center gap-2">
            <input
              type="radio"
              className="w-4 h-4"
              checked={mode === "send_immediately"}
              onChange={() => setMode("send_immediately")}
            />
            <div className="flex flex-col">
              <span className="font-medium text-sm">
                Azonnali küldés emailben
              </span>
              <span className="text-[11px] text-gray-500">
                A dokumentum jóváhagyottnak számít, és azonnal kiküldjük a
                címzetteknek.
              </span>
            </div>
          </label>

          <label className="flex items-center gap-2">
            <input
              type="radio"
              className="w-4 h-4"
              checked={mode === "request_approval"}
              onChange={() => setMode("request_approval")}
            />
            <div className="flex flex-col">
              <span className="font-medium text-sm">
                Jóváhagyásra küldés (belső folyamat)
              </span>
              <span className="text-[11px] text-gray-500">
                A dokumentum csak jóváhagyás után lesz végleges. Opcionálisan
                jóváhagyás után automatikus emailt is küldhetünk.
              </span>
            </div>
          </label>
        </div>

        {mode === "request_approval" && (
          <label className="flex items-center gap-2 mt-2 text-xs text-gray-700">
            <input
              type="checkbox"
              className="w-4 h-4"
              checked={sendEmailOnApproval}
              onChange={(e) => setSendEmailOnApproval(e.target.checked)}
            />
            Jóváhagyás után automatikus email küldése a címzetteknek
          </label>
        )}
      </section>

      {/* Approvers (only for request_approval) */}
      {mode === "request_approval" && (
        <section className="space-y-2 border border-gray-200 rounded p-3">
          <h2 className="text-sm font-semibold">Jóváhagyók</h2>
          <p className="text-xs text-gray-500">
            Válaszd ki, kiknek kell jóváhagyniuk a dokumentumot (pl. CEO,
            közvetlen vezető). Csak akkor lesz &quot;jóváhagyott&quot;, ha
            minden jóváhagyó elfogadta.
          </p>
          <Select
            cacheOptions
            defaultOptions
            isMulti
            loadOptions={loadApprovers}
            value={selectedApprovers}
            onChange={(value) =>
              setSelectedApprovers(value as ApproverOption[])
            }
            placeholder="Jóváhagyók keresése név alapján…"
            classNamePrefix="react-select"
          />
        </section>
      )}

      {/* Recipients */}
      <section className="space-y-2 border border-gray-200 rounded p-3">
        <h2 className="text-sm font-semibold">Címzettek</h2>
        <p className="text-xs text-gray-500">
          Válassz egy vagy több címzettet az adatbázisból.{" "}
          {mode === "request_approval" && !sendEmailOnApproval && (
            <span className="text-[11px] text-gray-500">
              (Ha nem kérsz automatikus küldést jóváhagyás után, a címzettek
              opcionálisak.)
            </span>
          )}
        </p>
        <Select
          cacheOptions
          defaultOptions
          isMulti
          loadOptions={loadRecipients}
          value={selectedRecipients}
          onChange={(value) =>
            setSelectedRecipients(value as RecipientOption[])
          }
          placeholder="Címzettek keresése név vagy email alapján…"
          classNamePrefix="react-select"
        />
      </section>

      {/* Email content */}
      <section className="space-y-3 border border-gray-200 rounded p-3">
        <h2 className="text-sm font-semibold">Email tartalma</h2>
        <p className="text-xs text-gray-500">
          Add meg az email tárgyát és szövegét, amit a címzettek megkapnak a
          csatolt DOCX dokumentummal együtt. Csak akkor kötelező, ha email
          küldés is történik.
        </p>

        <div className="flex flex-col gap-2">
          <div className="flex flex-col text-xs text-gray-700">
            <span className="mb-1">Email tárgya</span>
            <input
              type="text"
              className="border rounded px-2 py-1 text-sm"
              value={emailSubject}
              onChange={(e) => setEmailSubject(e.target.value)}
            />
          </div>

          <div className="flex flex-col text-xs text-gray-700">
            <span className="mb-1">Email szövege</span>
            <textarea
              className="border rounded px-2 py-1 text-sm min-h-[80px]"
              value={emailBody}
              onChange={(e) => setEmailBody(e.target.value)}
            />
          </div>
        </div>
      </section>

      {/* Form fields */}
      <section className="space-y-3 border border-gray-200 rounded p-3">
        <div className="flex items-center justify-between gap-2">
          <h2 className="text-sm font-semibold">Űrlap adatok</h2>

          {schema.length > 0 && (
            <div className="flex gap-2">
              <button
                type="button"
                onClick={clearToEmpty}
                className="px-3 py-1 rounded border text-xs text-gray-700 hover:bg-gray-50">
                Alapértelmezett értékek törlése
              </button>
              <button
                type="button"
                onClick={reapplyDefaults}
                className="px-3 py-1 rounded border border-emerald-500 text-xs text-emerald-700 hover:bg-emerald-50">
                Alapértelmezések visszaállítása
              </button>
            </div>
          )}
        </div>

        {!schema.length && (
          <p className="text-xs text-orange-600">
            Ehhez a sablonhoz még nincs beállított <code>form_schema</code>. A
            dokumentum a placeholder-ek alapértelmezett értékeivel fog
            generálódni (ha vannak).
          </p>
        )}

        <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
          {schema.map((field, index) => {
            const key = field.name || `field_${index}`;
            const value = formValues[field.name];

            if (field.type === "separator") {
              return (
                <div
                  key={key}
                  className="md:col-span-2 pt-2 mt-2 border-t border-gray-200">
                  <h3 className="text-sm font-semibold text-gray-800">
                    {field.label || "Szekció"}
                  </h3>
                  {field.description && (
                    <p className="text-[11px] text-gray-500 mt-0.5">
                      {field.description}
                    </p>
                  )}
                </div>
              );
            }

            if (field.type === "checkbox") {
              return (
                <div key={key} className="flex flex-col text-xs text-gray-700">
                  <label className="flex items-center gap-2 text-sm border rounded px-3 py-2 bg-gray-50">
                    <input
                      type="checkbox"
                      className="w-4 h-4"
                      checked={!!value}
                      onChange={(e) =>
                        handleFieldChange(field.name, e.target.checked)
                      }
                    />
                    <span>{field.label}</span>
                  </label>
                  {field.description && (
                    <p className="text-[11px] text-gray-500 mt-1">
                      {field.description}
                    </p>
                  )}
                </div>
              );
            }

            if (field.type === "textarea") {
              return (
                <div key={key} className="flex flex-col text-xs text-gray-700">
                  <span className="mb-1">
                    {field.label}
                    {field.required && <span className="text-red-500"> *</span>}
                  </span>
                  <textarea
                    className="border rounded px-2 py-1 text-sm min-h-[80px]"
                    value={value ?? ""}
                    onChange={(e) =>
                      handleFieldChange(field.name, e.target.value)
                    }
                  />
                  {field.description && (
                    <p className="text-[11px] text-gray-500 mt-1">
                      {field.description}
                    </p>
                  )}
                </div>
              );
            }

            // text / date
            return (
              <div key={key} className="flex flex-col text-xs text-gray-700">
                <span className="mb-1">
                  {field.label}
                  {field.required && <span className="text-red-500"> *</span>}
                </span>
                <input
                  type={field.type === "date" ? "date" : "text"}
                  className="border rounded px-2 py-1 text-sm"
                  value={value ?? ""}
                  onChange={(e) =>
                    handleFieldChange(field.name, e.target.value)
                  }
                />
                {field.description && (
                  <p className="text-[11px] text-gray-500 mt-1">
                    {field.description}
                  </p>
                )}
              </div>
            );
          })}
        </div>
      </section>

      {/* Actions */}
      <div className="flex justify-end gap-2">
        <button
          onClick={handleSubmit}
          disabled={sending}
          className="px-4 py-2 rounded bg-emerald-600/80 hover:bg-emerald-600 text-white text-sm disabled:bg-emerald-300">
          {sending
            ? "Feldolgozás…"
            : mode === "send_immediately"
            ? "Kitöltés + DOCX azonnali küldése"
            : "Kitöltés + dokumentum jóváhagyásra küldése"}
        </button>
      </div>
    </main>
  );
}
