"use client";

import React from "react";
import {
  Search,
  Plus,
  MoreHorizontal,
  ChevronRight,
  Upload,
  X,
  AlertTriangle,
} from "lucide-react";

/**
 * Zoho-ish, CRM-style primitives.
 * - Tailwind-based (no inline styles)
 * - Subtle borders, soft shadows, compact density
 * - Sticky header, page chrome, toolbars
 * - Consistent focus rings and hover states
 */

function cn(...classes: Array<string | false | null | undefined>) {
  return classes.filter(Boolean).join(" ");
}

// ---------- Page ----------

export function Page({
  title,
  subtitle,
  actions,
  breadcrumb,
  children,
}: {
  title: string;
  subtitle?: string;
  actions?: React.ReactNode;
  breadcrumb?: Array<{ label: string; href?: string }>;
  children: React.ReactNode;
}) {
  return (
    <div className="min-h-screen bg-slate-50 text-slate-900">
      {/* Top App Bar (Zoho-ish chrome) */}
      {/* <div className="sticky top-0 z-40 border-b border-slate-200/80 bg-white/80 backdrop-blur">
        <div className="mx-auto flex max-w-6xl items-center gap-3 px-4 py-3">
          <div className="flex items-center gap-2">
            <div className="h-8 w-8 rounded-xl bg-gradient-to-br from-blue-600 to-indigo-600 shadow-sm" />
            <div className="font-semibold tracking-tight">CRM</div>
          </div>

          <div className="ml-auto flex items-center gap-2">
            <div className="hidden sm:block">
              <SearchInput placeholder="Search records, contacts, deals..." />
            </div>
            {actions}
            <IconButton aria-label="More" title="More">
              <MoreHorizontal className="h-4 w-4" />
            </IconButton>
          </div>
        </div>
      </div> */}

      <div className="mx-auto max-w-8xl px-4 py-5">
        {breadcrumb?.length ? (
          <Breadcrumb items={breadcrumb} className="mb-3" />
        ) : null}

        <div className="mb-4 flex flex-wrap items-start justify-between gap-3">
          <div>
            <h1 className="text-xl font-semibold tracking-tight">{title}</h1>
            {subtitle ? (
              <div className="mt-1 text-sm text-slate-600">{subtitle}</div>
            ) : null}
          </div>
          {actions ? <div className="flex gap-2">{actions}</div> : null}
        </div>
        {children}
      </div>
    </div>
  );
}

export function Breadcrumb({
  items,
  className,
}: {
  items: Array<{ label: string; href?: string }>;
  className?: string;
}) {
  return (
    <nav
      className={cn(
        "flex items-center gap-1 text-sm text-slate-600",
        className
      )}>
      {items.map((it, idx) => {
        const isLast = idx === items.length - 1;
        const Node = it.href && !isLast ? "a" : "span";
        return (
          <React.Fragment key={`${it.label}-${idx}`}>
            <Node
              {...(Node === "a" ? { href: it.href } : {})}
              className={cn(
                "rounded-md px-1.5 py-0.5",
                Node === "a" &&
                  "hover:bg-slate-100 hover:text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-600/20"
              )}>
              {it.label}
            </Node>
            {!isLast ? (
              <ChevronRight className="h-4 w-4 text-slate-400" />
            ) : null}
          </React.Fragment>
        );
      })}
    </nav>
  );
}

// ---------- Cards ----------

// export function Card({
//   className,
//   ...props
// }: React.HTMLAttributes<HTMLDivElement>) {
//   return (
//     <div
//       {...props}
//       className={cn(
//         "rounded-xl border border-slate-200 bg-white shadow-sm p-2",
//         "hover:shadow-md hover:shadow-slate-900/[0.04]",
//         "transition-shadow",
//         className
//       )}
//     />
//   );
// }
export const Card = React.forwardRef<
  HTMLDivElement,
  React.HTMLAttributes<HTMLDivElement>
>(({ className, ...props }, ref) => {
  return (
    <div
      ref={ref}
      {...props}
      className={cn(
        "rounded-xl border border-slate-200 bg-white shadow-sm p-2",
        "hover:shadow-md hover:shadow-slate-900/[0.04]",
        "transition-shadow",
        className
      )}
    />
  );
});
Card.displayName = "Card";
export function CardHeader({
  className,
  ...props
}: React.HTMLAttributes<HTMLDivElement>) {
  return (
    <div
      {...props}
      className={cn(
        "flex items-start justify-between gap-3 border-b border-slate-100 px-4 py-3",
        className
      )}
    />
  );
}

export function CardBody({
  className,
  ...props
}: React.HTMLAttributes<HTMLDivElement>) {
  return <div {...props} className={cn("px-4 py-3", className)} />;
}
export function CardTitle({
  children,
  className,
}: {
  children: React.ReactNode;
  className?: string;
}) {
  return (
    <div className={cn("text-sm font-semibold text-slate-900", className)}>
      {children}
    </div>
  );
}

export function CardMeta({
  children,
  className,
}: {
  children: React.ReactNode;
  className?: string;
}) {
  return (
    <div className={cn("mt-0.5 text-xs text-slate-600", className)}>
      {children}
    </div>
  );
}

// export function CardTitle({ children }: { children: React.ReactNode }) {
//   return <div className="text-sm font-semibold text-slate-900">{children}</div>;
// }

// export function CardMeta({ children }: { children: React.ReactNode }) {
//   return <div className="mt-0.5 text-xs text-slate-600">{children}</div>;
// }

// ---------- Chips ----------

const chipTone: Record<
  NonNullable<ChipProps["tone"]>,
  { base: string; dot: string }
> = {
  neutral: {
    base: "bg-slate-50 text-slate-700 border-slate-200",
    dot: "bg-slate-400",
  },
  good: {
    base: "bg-emerald-50 text-emerald-800 border-emerald-200",
    dot: "bg-emerald-500",
  },
  warn: {
    base: "bg-amber-50 text-amber-900 border-amber-200",
    dot: "bg-amber-500",
  },
  bad: { base: "bg-rose-50 text-rose-900 border-rose-200", dot: "bg-rose-500" },
  info: {
    base: "bg-blue-50 text-blue-900 border-blue-200",
    dot: "bg-blue-500",
  },
};

type ChipProps = {
  children: React.ReactNode;
  tone?: "neutral" | "good" | "warn" | "bad" | "info";
  dot?: boolean;
  className?: string;
};

export function Chip({
  children,
  tone = "neutral",
  dot = true,
  className,
}: ChipProps) {
  const t = chipTone[tone];
  return (
    <span
      className={cn(
        "inline-flex items-center gap-2 rounded-full border px-2.5 py-1 text-xs font-medium",
        t.base,
        className
      )}>
      {dot ? <span className={cn("h-1.5 w-1.5 rounded-full", t.dot)} /> : null}
      <span className="whitespace-nowrap">{children}</span>
    </span>
  );
}

// ---------- Toolbar & layout helpers ----------

export function ToolbarRow({
  className,
  children,
}: {
  className?: string;
  children: React.ReactNode;
}) {
  return (
    <div className={cn("flex flex-wrap items-center gap-2", className)}>
      {children}
    </div>
  );
}

export function GridCards({ children }: { children: React.ReactNode }) {
  return (
    <div className="grid gap-3 sm:grid-cols-2 lg:grid-cols-3">{children}</div>
  );
}

export function SectionTitle({ children }: { children: React.ReactNode }) {
  return (
    <h2 className="my-3 text-sm font-semibold text-slate-900">{children}</h2>
  );
}

export function Divider() {
  return <div className="my-4 h-px bg-slate-200/70" />;
}

// ---------- Inputs ----------

const inputBase =
  "w-full rounded-xl border border-slate-200 bg-white px-3 py-2 text-sm shadow-sm " +
  "placeholder:text-slate-400 focus:outline-none focus:ring-2 focus:ring-blue-600/20 focus:border-blue-600/40";

export function TextInput({
  className,
  ...props
}: React.InputHTMLAttributes<HTMLInputElement>) {
  return <input {...props} className={cn(inputBase, className)} />;
}

export function TextArea({
  className,
  ...props
}: React.TextareaHTMLAttributes<HTMLTextAreaElement>) {
  return (
    <textarea
      {...props}
      className={cn(inputBase, "min-h-[100px] resize-y", className)}
    />
  );
}

export function Select({
  className,
  ...props
}: React.SelectHTMLAttributes<HTMLSelectElement>) {
  return <select {...props} className={cn(inputBase, className)} />;
}

export function Field({
  label,
  hint,
  error,
  children,
}: {
  label: string;
  hint?: string;
  error?: string;
  children: React.ReactNode;
}) {
  return (
    <div className="space-y-1">
      <div className="flex items-baseline justify-between gap-2">
        <label className="text-xs font-medium text-slate-700">{label}</label>
        {hint ? <div className="text-xs text-slate-500">{hint}</div> : null}
      </div>
      {children}
      {error ? (
        <div className="flex items-center gap-1 text-xs text-rose-700">
          <AlertTriangle className="h-3.5 w-3.5" />
          {error}
        </div>
      ) : null}
    </div>
  );
}

export function SearchInput({
  className,
  ...props
}: React.InputHTMLAttributes<HTMLInputElement>) {
  return (
    <div
      className={cn(
        "flex items-center gap-2 rounded-xl border border-slate-200 bg-white px-3 py-2 text-sm shadow-sm",
        "focus-within:ring-2 focus-within:ring-blue-600/20 focus-within:border-blue-600/40",
        className
      )}>
      <Search className="h-4 w-4 text-slate-400" />
      <input
        {...props}
        className="w-72 bg-transparent text-sm outline-none placeholder:text-slate-400"
      />
      {props.value ? (
        <button
          type="button"
          className="rounded-md p-1 text-slate-400 hover:bg-slate-100 hover:text-slate-700"
          onClick={() => {
            // Only clears if parent is controlling value and passed onChange.
            const ev = {
              target: { value: "" },
            } as unknown as React.ChangeEvent<HTMLInputElement>;
            props.onChange?.(ev);
          }}
          aria-label="Clear"
          title="Clear">
          <X className="h-4 w-4" />
        </button>
      ) : null}
    </div>
  );
}

// ---------- Buttons ----------

const btn =
  "inline-flex items-center justify-center gap-2 rounded-xl px-3 py-2 text-sm font-medium " +
  "transition active:scale-[0.99] disabled:opacity-60 disabled:cursor-not-allowed " +
  "focus:outline-none focus:ring-2 focus:ring-blue-600/20";

export function Button({
  className,
  ...props
}: React.ButtonHTMLAttributes<HTMLButtonElement>) {
  return (
    <button
      {...props}
      className={cn(
        btn,
        "bg-slate-900 text-white shadow-sm hover:bg-slate-800",
        className
      )}
    />
  );
}

export function SecondaryButton({
  className,
  ...props
}: React.ButtonHTMLAttributes<HTMLButtonElement>) {
  return (
    <button
      {...props}
      className={cn(
        btn,
        "border border-slate-200 bg-white text-slate-900 shadow-sm hover:bg-slate-200 cursor-pointer",
        className
      )}
    />
  );
}

export function DangerButton({
  className,
  ...props
}: React.ButtonHTMLAttributes<HTMLButtonElement>) {
  return (
    <button
      {...props}
      className={cn(
        btn,
        "border border-rose-200 bg-rose-600 text-white shadow-sm hover:bg-rose-700",
        className
      )}
    />
  );
}

export function IconButton({
  className,
  ...props
}: React.ButtonHTMLAttributes<HTMLButtonElement> & { title?: string }) {
  return (
    <button
      {...props}
      className={cn(
        "inline-flex h-10 w-10 items-center justify-center rounded-xl border border-slate-200 bg-white text-slate-700 shadow-sm",
        "transition hover:bg-slate-50 active:scale-[0.99]",
        "focus:outline-none focus:ring-2 focus:ring-blue-600/20",
        "disabled:opacity-60 disabled:cursor-not-allowed",
        className
      )}
    />
  );
}

export function PrimaryAction({
  children,
  ...props
}: React.ButtonHTMLAttributes<HTMLButtonElement>) {
  return (
    <Button {...props}>
      <Plus className="h-4 w-4" />
      {children}
    </Button>
  );
}

// ---------- Empty State ----------

export function EmptyState({
  title,
  hint,
  action,
}: {
  title: string;
  hint?: string;
  action?: React.ReactNode;
}) {
  return (
    <div className="rounded-2xl border border-dashed border-slate-300 bg-white p-5">
      <div className="text-sm font-semibold text-slate-900">{title}</div>
      {hint ? <div className="mt-1 text-sm text-slate-600">{hint}</div> : null}
      {action ? <div className="mt-3">{action}</div> : null}
    </div>
  );
}

// ---------- Dropzone ----------

export function Dropzone({
  onFile,
  disabled,
  label = "Drop a file here, or click to browse",
}: {
  onFile: (file: File) => void | Promise<void>;
  disabled?: boolean;
  label?: string;
}) {
  return (
    <label
      className={cn(
        "block rounded-2xl border border-dashed border-slate-300 bg-white p-4",
        "transition",
        disabled
          ? "cursor-not-allowed bg-slate-50"
          : "cursor-pointer hover:bg-slate-50"
      )}
      onDragOver={(e) => {
        e.preventDefault();
        e.stopPropagation();
      }}
      onDrop={(e) => {
        e.preventDefault();
        e.stopPropagation();
        if (disabled) return;
        const f = e.dataTransfer.files?.[0];
        if (f) void onFile(f);
      }}>
      <div className="flex items-center gap-2 text-sm text-slate-700">
        <Upload className="h-4 w-4 text-slate-500" />
        <span className="opacity-90">{label}</span>
      </div>
      <input
        type="file"
        disabled={disabled}
        className="hidden"
        onChange={(e) => {
          const f = e.target.files?.[0];
          if (f) void onFile(f);
        }}
      />
    </label>
  );
}

// ---------- Example usage (optional) ----------

export function DemoZohoPage() {
  return (
    <Page
      title="Deals"
      subtitle="Pipeline overview and recent activity"
      breadcrumb={[{ label: "CRM", href: "#" }, { label: "Deals" }]}
      actions={
        <>
          <SecondaryButton>Import</SecondaryButton>
          <PrimaryAction>New Deal</PrimaryAction>
        </>
      }>
      <ToolbarRow className="mb-4">
        <div className="w-full sm:w-auto">
          <SearchInput placeholder="Filter deals…" />
        </div>
        <Select className="w-44">
          <option>Owner: Anyone</option>
          <option>Owner: Me</option>
        </Select>
        <Select className="w-44">
          <option>Stage: Any</option>
          <option>Qualified</option>
          <option>Proposal</option>
        </Select>
        <div className="ml-auto flex gap-2">
          <SecondaryButton>Export</SecondaryButton>
          <IconButton aria-label="More">
            {" "}
            <MoreHorizontal className="h-4 w-4" />{" "}
          </IconButton>
        </div>
      </ToolbarRow>

      <GridCards>
        <Card>
          <CardHeader>
            <div>
              <CardTitle>Acme Renewal</CardTitle>
              <CardMeta>Expected close • Feb 12</CardMeta>
            </div>
            <Chip tone="good">Qualified</Chip>
          </CardHeader>
          <CardBody>
            <div className="flex items-center justify-between text-sm">
              <div className="text-slate-600">Amount</div>
              <div className="font-semibold">$24,000</div>
            </div>
            <Divider />
            <div className="flex items-center justify-between text-sm">
              <div className="text-slate-600">Owner</div>
              <div className="font-medium">You</div>
            </div>
          </CardBody>
        </Card>

        <Card>
          <CardHeader>
            <div>
              <CardTitle>Globex Expansion</CardTitle>
              <CardMeta>Last activity • 2 days ago</CardMeta>
            </div>
            <Chip tone="warn">Proposal</Chip>
          </CardHeader>
          <CardBody>
            <EmptyState
              title="No next step"
              hint="Add a task or call to keep this deal moving."
              action={<Button>Add Task</Button>}
            />
          </CardBody>
        </Card>

        <Card>
          <CardHeader>
            <div>
              <CardTitle>Umbrella Corp</CardTitle>
              <CardMeta>High risk</CardMeta>
            </div>
            <Chip tone="bad">At Risk</Chip>
          </CardHeader>
          <CardBody>
            <Dropzone
              label="Attach a proposal PDF"
              onFile={async (file) => {
                // eslint-disable-next-line no-console
                console.log("uploaded", file.name);
              }}
            />
          </CardBody>
        </Card>
      </GridCards>
    </Page>
  );
}

export function ProgressBar({ value }: { value: number }) {
  const v = Math.max(0, Math.min(100, Number(value) || 0));
  return (
    <div className="w-full">
      <div className="h-2 w-full rounded-full bg-slate-100 border border-slate-200 overflow-hidden">
        <div className="h-full bg-slate-900" style={{ width: `${v}%` }} />
      </div>
    </div>
  );
}

// "use client";

// import React from "react";

// const baseBorder = "1px solid #e5e7eb";

// export function Page({
//   title,
//   subtitle,
//   actions,
//   children,
// }: {
//   title: string;
//   subtitle?: string;
//   actions?: React.ReactNode;
//   children: React.ReactNode;
// }) {
//   return (
//     <div style={{ background: "#f8fafc", minHeight: "100vh" }}>
//       <div style={{ maxWidth: 1200, margin: "0 auto", padding: 16 }}>
//         <div
//           style={{
//             display: "flex",
//             alignItems: "flex-start",
//             justifyContent: "space-between",
//             gap: 12,
//             flexWrap: "wrap",
//             marginBottom: 12,
//           }}>
//           <div>
//             <h1 style={{ margin: 0, fontSize: 20, fontWeight: 900 }}>
//               {title}
//             </h1>
//             {subtitle ? (
//               <div style={{ fontSize: 12, opacity: 0.75, marginTop: 4 }}>
//                 {subtitle}
//               </div>
//             ) : null}
//           </div>
//           {actions ? (
//             <div style={{ display: "flex", gap: 8 }}>{actions}</div>
//           ) : null}
//         </div>

//         {children}
//       </div>
//     </div>
//   );
// }

// export function Card(props: React.HTMLAttributes<HTMLDivElement>) {
//   return (
//     <div
//       {...props}
//       style={{
//         border: baseBorder,
//         borderRadius: 14,
//         padding: 14,
//         background: "#fff",
//         boxShadow: "0 1px 0 rgba(17,24,39,0.04)",
//         ...props.style,
//       }}
//     />
//   );
// }

// export function CardTitle({ children }: { children: React.ReactNode }) {
//   return <div style={{ fontWeight: 900, fontSize: 14 }}>{children}</div>;
// }

// export function CardMeta({ children }: { children: React.ReactNode }) {
//   return (
//     <div style={{ fontSize: 12, opacity: 0.75, marginTop: 4 }}>{children}</div>
//   );
// }

// export function Chip({
//   children,
//   tone = "neutral",
// }: {
//   children: React.ReactNode;
//   tone?: "neutral" | "good" | "warn" | "bad" | "info";
// }) {
//   const toneMap: Record<string, { bg: string; border: string; color: string }> =
//     {
//       neutral: { bg: "#fff", border: "#e5e7eb", color: "#111827" },
//       good: { bg: "#ecfdf5", border: "#a7f3d0", color: "#065f46" },
//       warn: { bg: "#fffbeb", border: "#fcd34d", color: "#92400e" },
//       bad: { bg: "#fef2f2", border: "#fecaca", color: "#7f1d1d" },
//       info: { bg: "#eff6ff", border: "#bfdbfe", color: "#1d4ed8" },
//     };

//   const t = toneMap[tone] ?? toneMap.neutral;

//   return (
//     <span
//       style={{
//         display: "inline-flex",
//         alignItems: "center",
//         gap: 6,
//         padding: "2px 8px",
//         borderRadius: 999,
//         border: `1px solid ${t.border}`,
//         background: t.bg,
//         color: t.color,
//         fontSize: 12,
//         whiteSpace: "nowrap",
//       }}>
//       {children}
//     </span>
//   );
// }

// export function ToolbarRow({ children }: { children: React.ReactNode }) {
//   return (
//     <div
//       style={{
//         display: "flex",
//         gap: 8,
//         alignItems: "center",
//         flexWrap: "wrap",
//       }}>
//       {children}
//     </div>
//   );
// }

// export function TextInput(props: React.InputHTMLAttributes<HTMLInputElement>) {
//   return (
//     <input
//       {...props}
//       style={{
//         padding: "10px 12px",
//         borderRadius: 10,
//         border: baseBorder,
//         minWidth: 220,
//         outline: "none",
//         ...props.style,
//       }}
//     />
//   );
// }

// export function TextArea(
//   props: React.TextareaHTMLAttributes<HTMLTextAreaElement>
// ) {
//   return (
//     <textarea
//       {...props}
//       style={{
//         padding: "10px 12px",
//         borderRadius: 10,
//         border: baseBorder,
//         width: "100%",
//         outline: "none",
//         resize: "vertical",
//         ...props.style,
//       }}
//     />
//   );
// }

// export function Select(props: React.SelectHTMLAttributes<HTMLSelectElement>) {
//   return (
//     <select
//       {...props}
//       style={{
//         padding: "10px 12px",
//         borderRadius: 10,
//         border: baseBorder,
//         outline: "none",
//         ...props.style,
//       }}
//     />
//   );
// }

// function btnBase(disabled?: boolean) {
//   return {
//     padding: "10px 12px",
//     borderRadius: 10,
//     border: baseBorder,
//     cursor: disabled ? "not-allowed" : "pointer",
//     transition: "transform 0.05s ease, box-shadow 0.15s ease",
//   } as const;
// }

// export function Button(props: React.ButtonHTMLAttributes<HTMLButtonElement>) {
//   const disabled = !!props.disabled;
//   return (
//     <button
//       {...props}
//       style={{
//         ...btnBase(disabled),
//         background: disabled ? "#f3f4f6" : "#111827",
//         color: disabled ? "#6b7280" : "#fff",
//         boxShadow: disabled ? "none" : "0 1px 0 rgba(17,24,39,0.06)",
//         ...props.style,
//       }}
//       onMouseDown={(e) => {
//         if (!disabled) e.currentTarget.style.transform = "scale(0.99)";
//         props.onMouseDown?.(e);
//       }}
//       onMouseUp={(e) => {
//         e.currentTarget.style.transform = "scale(1)";
//         props.onMouseUp?.(e);
//       }}
//     />
//   );
// }

// export function SecondaryButton(
//   props: React.ButtonHTMLAttributes<HTMLButtonElement>
// ) {
//   const disabled = !!props.disabled;
//   return (
//     <button
//       {...props}
//       style={{
//         ...btnBase(disabled),
//         background: "#fff",
//         color: "#111827",
//         ...props.style,
//       }}
//     />
//   );
// }

// export function DangerButton(
//   props: React.ButtonHTMLAttributes<HTMLButtonElement>
// ) {
//   const disabled = !!props.disabled;
//   return (
//     <button
//       {...props}
//       style={{
//         ...btnBase(disabled),
//         background: disabled ? "#fef2f2" : "#991b1b",
//         color: disabled ? "#7f1d1d" : "#fff",
//         border: `1px solid ${disabled ? "#fecaca" : "#7f1d1d"}`,
//         ...props.style,
//       }}
//     />
//   );
// }

// export function IconButton(
//   props: React.ButtonHTMLAttributes<HTMLButtonElement> & { title?: string }
// ) {
//   const disabled = !!props.disabled;
//   return (
//     <button
//       {...props}
//       style={{
//         ...btnBase(disabled),
//         padding: "8px 10px",
//         lineHeight: 1,
//         background: "#fff",
//         ...props.style,
//       }}
//     />
//   );
// }

// export function GridCards({ children }: { children: React.ReactNode }) {
//   return (
//     <div
//       style={{
//         display: "grid",
//         gridTemplateColumns: "repeat(auto-fill, minmax(320px, 1fr))",
//         gap: 12,
//       }}>
//       {children}
//     </div>
//   );
// }

// export function SectionTitle({ children }: { children: React.ReactNode }) {
//   return <h2 style={{ margin: "10px 0 10px", fontSize: 16 }}>{children}</h2>;
// }

// export function Divider() {
//   return <div style={{ height: 1, background: "#eef2f7", margin: "12px 0" }} />;
// }

// export function EmptyState({
//   title,
//   hint,
//   action,
// }: {
//   title: string;
//   hint?: string;
//   action?: React.ReactNode;
// }) {
//   return (
//     <div
//       style={{
//         border: "1px dashed #cbd5e1",
//         borderRadius: 14,
//         padding: 16,
//         background: "#fff",
//       }}>
//       <div style={{ fontWeight: 900 }}>{title}</div>
//       {hint ? (
//         <div style={{ fontSize: 12, opacity: 0.75, marginTop: 4 }}>{hint}</div>
//       ) : null}
//       {action ? <div style={{ marginTop: 10 }}>{action}</div> : null}
//     </div>
//   );
// }

// export function Dropzone({
//   onFile,
//   disabled,
//   label = "Dobd ide a fájlt, vagy kattints kiválasztáshoz",
// }: {
//   onFile: (file: File) => void | Promise<void>;
//   disabled?: boolean;
//   label?: string;
// }) {
//   return (
//     <label
//       style={{
//         display: "block",
//         border: "1px dashed #cbd5e1",
//         borderRadius: 14,
//         padding: 14,
//         background: disabled ? "#f8fafc" : "#fff",
//         cursor: disabled ? "not-allowed" : "pointer",
//       }}
//       onDragOver={(e) => {
//         e.preventDefault();
//         e.stopPropagation();
//       }}
//       onDrop={(e) => {
//         e.preventDefault();
//         e.stopPropagation();
//         if (disabled) return;
//         const f = e.dataTransfer.files?.[0];
//         if (f) void onFile(f);
//       }}>
//       <div style={{ fontSize: 12, opacity: 0.8 }}>{label}</div>
//       <input
//         type="file"
//         disabled={disabled}
//         style={{ display: "none" }}
//         onChange={(e) => {
//           const f = e.target.files?.[0];
//           if (f) void onFile(f);
//         }}
//       />
//     </label>
//   );
// }

// // "use client";

// // import React from "react";

// // export function Card(props: React.HTMLAttributes<HTMLDivElement>) {
// //   return (
// //     <div
// //       {...props}
// //       style={{
// //         border: "1px solid #e5e7eb",
// //         borderRadius: 12,
// //         padding: 12,
// //         background: "#fff",
// //         ...props.style,
// //       }}
// //     />
// //   );
// // }

// // export function CardTitle({ children }: { children: React.ReactNode }) {
// //   return <div style={{ fontWeight: 800, fontSize: 14 }}>{children}</div>;
// // }

// // export function CardMeta({ children }: { children: React.ReactNode }) {
// //   return (
// //     <div style={{ fontSize: 12, opacity: 0.75, marginTop: 4 }}>{children}</div>
// //   );
// // }

// // export function Chip({ children }: { children: React.ReactNode }) {
// //   return (
// //     <span
// //       style={{
// //         display: "inline-flex",
// //         alignItems: "center",
// //         gap: 6,
// //         padding: "2px 8px",
// //         borderRadius: 999,
// //         border: "1px solid #e5e7eb",
// //         fontSize: 12,
// //         whiteSpace: "nowrap",
// //       }}>
// //       {children}
// //     </span>
// //   );
// // }

// // export function ToolbarRow({ children }: { children: React.ReactNode }) {
// //   return (
// //     <div
// //       style={{
// //         display: "flex",
// //         gap: 8,
// //         alignItems: "center",
// //         flexWrap: "wrap",
// //       }}>
// //       {children}
// //     </div>
// //   );
// // }

// // export function TextInput(props: React.InputHTMLAttributes<HTMLInputElement>) {
// //   return (
// //     <input
// //       {...props}
// //       style={{
// //         padding: "10px 12px",
// //         borderRadius: 10,
// //         border: "1px solid #e5e7eb",
// //         minWidth: 220,
// //         ...props.style,
// //       }}
// //     />
// //   );
// // }

// // export function Select(props: React.SelectHTMLAttributes<HTMLSelectElement>) {
// //   return (
// //     <select
// //       {...props}
// //       style={{
// //         padding: "10px 12px",
// //         borderRadius: 10,
// //         border: "1px solid #e5e7eb",
// //         ...props.style,
// //       }}
// //     />
// //   );
// // }

// // export function Button(props: React.ButtonHTMLAttributes<HTMLButtonElement>) {
// //   return (
// //     <button
// //       {...props}
// //       style={{
// //         padding: "10px 12px",
// //         borderRadius: 10,
// //         border: "1px solid #e5e7eb",
// //         background: props.disabled ? "#f3f4f6" : "#111827",
// //         color: props.disabled ? "#6b7280" : "#fff",
// //         cursor: props.disabled ? "not-allowed" : "pointer",
// //         ...props.style,
// //       }}
// //     />
// //   );
// // }

// // export function SecondaryButton(
// //   props: React.ButtonHTMLAttributes<HTMLButtonElement>
// // ) {
// //   return (
// //     <button
// //       {...props}
// //       style={{
// //         padding: "10px 12px",
// //         borderRadius: 10,
// //         border: "1px solid #e5e7eb",
// //         background: "#fff",
// //         color: "#111827",
// //         cursor: props.disabled ? "not-allowed" : "pointer",
// //         ...props.style,
// //       }}
// //     />
// //   );
// // }

// // export function GridCards({ children }: { children: React.ReactNode }) {
// //   return (
// //     <div
// //       style={{
// //         display: "grid",
// //         gridTemplateColumns: "repeat(auto-fill, minmax(320px, 1fr))",
// //         gap: 12,
// //       }}>
// //       {children}
// //     </div>
// //   );
// // }

// // export function SectionTitle({ children }: { children: React.ReactNode }) {
// //   return <h2 style={{ margin: "8px 0 10px", fontSize: 16 }}>{children}</h2>;
// // }
