// context/AuthContext.js
"use client";
import { createContext, useContext, useEffect, useState } from "react";
import Cookies from "js-cookie";
import { useRouter } from "next/navigation";
import ApiCall from "@/api/api-calls";
import apiClient from "@/lib/apiClient";
import useSocketAuth from "@/hooks/useSocketAuth"; // 🔥 NEW HOOK
import SocketContextProvider from "@/context/SocketProvider";
import jwt from "jsonwebtoken";

const AuthContext = createContext(null);

export const AuthProvider = ({ children }) => {
  const [user, setUser] = useState(null);
  const [loading, setLoading] = useState(true);
  const [isAuthenticated, setIsAuthenticated] = useState(false);
  const router = useRouter();

  // 🔌 Initialize socket using new modular hook
  // onLogout from socket side: local-only logout (no broadcast)
  const { socket, reconnect } = useSocketAuth({
    onLogout: () => {
      console.log("[Auth] onLogout from useSocketAuth");
      localLogout(); // defined below
    },
  });

  // ⚙️ Local logout logic used by:
  // - explicit logout()
  // - socket forced logout (onLogout)
  // - BroadcastChannel LOGOUT messages from other tabs
  function localLogout() {
    try {
      Cookies.remove("edSId");
      setUser(null);
      setIsAuthenticated(false);

      try {
        socket?.disconnect();
      } catch (e) {
        console.error("[Auth] Error disconnecting socket on logout", e);
      }

      router.push("/");
    } catch (e) {
      console.error("[Auth] localLogout error:", e);
    }
  }

  useEffect(() => {
    const initAuth = async () => {
      const token = Cookies.get("edSId");
      if (!token) {
        setLoading(false);
        return;
      }

      apiClient.defaults.headers.common.Authorization = `Bearer ${token}`;
      try {
        const res = await ApiCall.auth.getAdmin();
        setUser(res.user || res.data?.user);
        setIsAuthenticated(true);
      } catch (err) {
        console.warn("getAdmin failed. Trying refresh...", err);
        try {
          const refreshRes = await ApiCall.auth.refreshToken();
          const newToken = refreshRes.data?.accessToken;
          const userData = refreshRes.data?.user;

          if (!newToken || !userData)
            throw new Error("Invalid refresh response");

          Cookies.set("edSId", newToken, { expires: 1 });
          apiClient.defaults.headers.common[
            "Authorization"
          ] = `Bearer ${newToken}`;

          setUser(userData);
          setIsAuthenticated(true);
          window.dispatchEvent(
            new CustomEvent("token-refreshed", { detail: newToken })
          );
        } catch (refreshErr) {
          console.error("Refresh failed → Logging out...", refreshErr);
          Cookies.remove("edSId");
          setUser(null);
          setIsAuthenticated(false);
        }
      } finally {
        setLoading(false);
      }
    };

    initAuth();
  }, []);

  useEffect(() => {
    const handleAuthFailed = () => {
      logout(); // will broadcast + local logout
    };
    window.addEventListener("auth-failed", handleAuthFailed);
    return () => window.removeEventListener("auth-failed", handleAuthFailed);
  }, []);

  // 🛰 Listen for LOGOUT from *other* tabs & do local logout
  useEffect(() => {
    if (typeof window === "undefined") return;

    const channel = new BroadcastChannel("socket-tabs");

    const handleMessage = (event) => {
      if (event.data?.type === "LOGOUT") {
        console.log("[Auth] LOGOUT received from another tab");
        localLogout();
      }
    };

    channel.addEventListener("message", handleMessage);

    return () => {
      channel.removeEventListener("message", handleMessage);
      channel.close();
    };
  }, [socket]); // socket dependency so localLogout sees latest socket

  const login = async (username, password) => {
    try {
      const res = await ApiCall.auth.postLogin({ username, password });
      const data = res.data || res;
      const token = data.accessToken;
      const userData = data.user;

      if (!token || !userData)
        return { success: false, message: "Invalid server response" };

      Cookies.set("edSId", token, { expires: 1 });
      apiClient.defaults.headers.common["Authorization"] = `Bearer ${token}`;
      setUser(userData);
      setIsAuthenticated(true);

      return { success: true };
    } catch (err) {
      console.error("Login error:", err);
      return { success: false, message: "Login failed" };
    }
  };

  // 🌐 Public logout: API + broadcast + local logout
  const logout = async () => {
    try {
      await ApiCall.auth.postLogout();
    } catch (err) {
      console.error("Logout error:", err);
    } finally {
      // 🔊 Tell other tabs
      try {
        if (typeof window !== "undefined") {
          const channel = new BroadcastChannel("socket-tabs");
          channel.postMessage({ type: "LOGOUT" });
          channel.close();
        }
      } catch (e) {
        console.error("[Auth] BroadcastChannel LOGOUT error:", e);
      }

      // 🧹 Local logout effects
      localLogout();
    }
  };

  return (
    <AuthContext.Provider
      value={{
        user,
        isAuthenticated,
        loading,
        login,
        logout,
        socket,
        reconnectSocket: reconnect,
      }}>
      <SocketContextProvider socket={socket}>{children}</SocketContextProvider>
    </AuthContext.Provider>
  );
};

export const useAuth = () => useContext(AuthContext);

// // context/AuthContext.js
// "use client";
// import { createContext, useContext, useEffect, useState } from "react";
// import Cookies from "js-cookie";
// import { useRouter } from "next/navigation";
// import ApiCall from "@/api/api-calls";
// import apiClient from "@/lib/apiClient";
// import useSocketAuth from "@/hooks/useSocketAuth"; // 🔥 NEW HOOK
// import SocketContextProvider from "@/context/SocketProvider";
// import jwt from "jsonwebtoken";

// const AuthContext = createContext(null);

// export const AuthProvider = ({ children }) => {
//   const [user, setUser] = useState(null);
//   const [loading, setLoading] = useState(true);
//   const [isAuthenticated, setIsAuthenticated] = useState(false);
//   const router = useRouter();

//   useEffect(() => {
//     const initAuth = async () => {
//       const token = Cookies.get("edSId");
//       if (!token) {
//         setLoading(false);
//         return;
//       }
//       apiClient.defaults.headers.common.Authorization = `Bearer ${token}`;
//       try {
//         const res = await ApiCall.auth.getAdmin();
//         setUser(res.user || res.data?.user);
//         setIsAuthenticated(true);
//       } catch (err) {
//         console.warn("getAdmin failed. Trying refresh...", err);
//         try {
//           const refreshRes = await ApiCall.auth.refreshToken();
//           const newToken = refreshRes.data?.accessToken;
//           const userData = refreshRes.data?.user;

//           if (!newToken || !userData)
//             throw new Error("Invalid refresh response");

//           Cookies.set("edSId", newToken, { expires: 1 });
//           apiClient.defaults.headers.common[
//             "Authorization"
//           ] = `Bearer ${newToken}`;

//           setUser(userData);
//           setIsAuthenticated(true);
//           window.dispatchEvent(
//             new CustomEvent("token-refreshed", { detail: newToken })
//           );
//         } catch (refreshErr) {
//           console.error("Refresh failed → Logging out...", refreshErr);
//           Cookies.remove("edSId");
//           setUser(null);
//           setIsAuthenticated(false);
//         }
//       } finally {
//         setLoading(false);
//       }
//     };

//     initAuth();
//   }, []);

//   useEffect(() => {
//     const handleAuthFailed = () => {
//       logout();
//     };
//     window.addEventListener("auth-failed", handleAuthFailed);
//     return () => window.removeEventListener("auth-failed", handleAuthFailed);
//   }, []);

//   const login = async (username, password) => {
//     try {
//       const res = await ApiCall.auth.postLogin({ username, password });
//       const data = res.data || res;
//       const token = data.accessToken;
//       const userData = data.user;

//       if (!token || !userData)
//         return { success: false, message: "Invalid server response" };

//       Cookies.set("edSId", token, { expires: 1 });
//       apiClient.defaults.headers.common["Authorization"] = `Bearer ${token}`;
//       setUser(userData);
//       setIsAuthenticated(true);

//       return { success: true };
//     } catch (err) {
//       console.error("Login error:", err);
//       return { success: false, message: "Login failed" };
//     }
//   };

//   const logout = async () => {
//     try {
//       await ApiCall.auth.postLogout();
//     } catch (err) {
//       console.error("Logout error:", err);
//     } finally {
//       Cookies.remove("edSId");
//       setUser(null);
//       setIsAuthenticated(false);
//       socket?.disconnect();
//       router.push("/");
//     }
//   };

//   // 🔌 Initialize socket using new modular hook
//   const { socket, reconnect } = useSocketAuth({ onLogout: logout });

//   return (
//     <AuthContext.Provider
//       value={{
//         user,
//         isAuthenticated,
//         loading,
//         login,
//         logout,
//         socket,
//         reconnectSocket: reconnect,
//       }}>
//       <SocketContextProvider socket={socket}>{children}</SocketContextProvider>
//     </AuthContext.Provider>
//   );
// };

// export const useAuth = () => useContext(AuthContext);
