// context/SocketProvider.js
"use client";
import React, { createContext, useEffect, useState } from "react";

export const SocketContext = createContext(null);
export const AdminUsersListContext = createContext([]);
export const SocketStatusContext = createContext("disconnected");

// 🔔 Notifications context (used by <SocketNotifications />)
export const SocketNotificationsContext = createContext([]);

type ChatUnreadContextType = {
  unread: number;
  setUnread: React.Dispatch<React.SetStateAction<number>>;
};

export const ChatUnreadContext = createContext<ChatUnreadContextType>({
  unread: 0,
  setUnread: () => {},
});

const SocketContextProvider = ({ children, socket }) => {
  const [adminUsersList, setAdminUsersList] = useState([]);
  const [socketStatus, setSocketStatus] = useState("disconnected");
  const [notifications, setNotifications] = useState([]);
  const [chatUnread, setChatUnread] = useState(0);

  const pushNotification = (notif) => {
    setNotifications((prev) => [
      ...prev,
      {
        id: Date.now() + Math.random(),
        ...notif,
      },
    ]);
  };

  useEffect(() => {
    if (!socket) {
      setSocketStatus("disconnected");
      setAdminUsersList([]);
      return;
    }

    const handleConnect = () => {
      console.log("✅ Socket connected");
      setSocketStatus("connected");
      pushNotification({
        type: "info",
        message: "Socket connected",
      });
    };

    const handleDisconnect = (reason) => {
      console.warn("⚠ Socket disconnected:", reason);
      setSocketStatus("disconnected");
      pushNotification({
        type: "warning",
        message: `Socket disconnected: ${reason}`,
      });
    };

    const handleConnectError = (err) => {
      console.error("❌ Socket connection error:", err.message);
      setSocketStatus("error");
      pushNotification({
        type: "error",
        message: `Socket error: ${err.message}`,
      });
    };

    const handleUpdateAdminUsersList = (list) => {
      setAdminUsersList(list || []);
    };

    const handleUserConnected = ({ username }) => {
      pushNotification({
        type: "info",
        message: `${username} connected`,
      });
    };

    const handleUserDisconnected = ({ username }) => {
      pushNotification({
        type: "warning",
        message: `${username} disconnected`,
      });
    };

    // ✅ Handle connection lifecycle
    socket.on("connect", handleConnect);
    socket.on("disconnect", handleDisconnect);
    socket.on("connect_error", handleConnectError);

    // ✅ Admin user list updates
    socket.on("updateAdminUsersList", handleUpdateAdminUsersList);

    // ✅ Presence notifications (emitted by the server)
    socket.on("user:connected", handleUserConnected);
    socket.on("user:disconnected", handleUserDisconnected);

    return () => {
      socket.off("connect", handleConnect);
      socket.off("disconnect", handleDisconnect);
      socket.off("connect_error", handleConnectError);
      socket.off("updateAdminUsersList", handleUpdateAdminUsersList);
      socket.off("user:connected", handleUserConnected);
      socket.off("user:disconnected", handleUserDisconnected);

      try {
        socket.disconnect();
      } catch (e) {
        console.error(
          "[SocketProvider] error disconnecting socket on unmount",
          e
        );
      }
    };
  }, [socket]);

  return (
    <SocketContext.Provider value={socket}>
      <SocketStatusContext.Provider value={socketStatus}>
        <AdminUsersListContext.Provider value={adminUsersList}>
          <SocketNotificationsContext.Provider value={notifications}>
            <ChatUnreadContext.Provider
              value={{ unread: chatUnread, setUnread: setChatUnread }}>
              {children}
            </ChatUnreadContext.Provider>
          </SocketNotificationsContext.Provider>
        </AdminUsersListContext.Provider>
      </SocketStatusContext.Provider>
    </SocketContext.Provider>
  );
};

export default SocketContextProvider;

// "use client";
// import { createContext, useEffect, useState, useCallback } from "react";

// export const SocketContext = createContext(null);
// export const AdminUsersListContext = createContext([]);
// export const SocketStatusContext = createContext("disconnected");

// // 🔔 NEW
// export const SocketNotificationsContext = createContext([]);

// // One channel across module
// const channel = new BroadcastChannel("socket-tabs");

// const SocketContextProvider = ({ children, socket }) => {
//   const [adminUsersList, setAdminUsersList] = useState([]);
//   const [socketStatus, setSocketStatus] = useState("disconnected");
//   const [notifications, setNotifications] = useState([]); // NEW

//   const handleLogoutBroadcast = useCallback(() => {
//     channel.postMessage({ type: "LOGOUT" });
//   }, []);

//   useEffect(() => {
//     if (!socket) return;

//     const handleConnect = () => {
//       console.log("✅ Socket connected");
//       setSocketStatus("connected");
//     };

//     const handleDisconnect = (reason) => {
//       console.warn("⚠ Socket disconnected:", reason);
//       setSocketStatus("disconnected");
//     };

//     const handleConnectError = (err) => {
//       console.error("❌ Socket connection error:", err.message);
//       setSocketStatus("error");
//     };

//     const handleUpdateAdminUsersList = (list) => {
//       setAdminUsersList(list || []);
//     };

//     const pushNotification = (notif) => {
//       setNotifications((prev) => [
//         ...prev,
//         { id: Date.now() + Math.random(), ...notif },
//       ]);
//     };

//     const handleUserConnected = ({ username }) => {
//       pushNotification({
//         type: "info",
//         message: `${username} connected`,
//       });
//     };

//     const handleUserDisconnected = ({ username }) => {
//       pushNotification({
//         type: "warning",
//         message: `${username} disconnected`,
//       });
//     };

//     socket.on("connect", handleConnect);
//     socket.on("disconnect", handleDisconnect);
//     socket.on("connect_error", handleConnectError);
//     socket.on("updateAdminUsersList", handleUpdateAdminUsersList);

//     // NEW: presence events
//     socket.on("user:connected", handleUserConnected);
//     socket.on("user:disconnected", handleUserDisconnected);

//     // ✅ Listen for broadcast logout across tabs
//     channel.onmessage = (event) => {
//       if (event.data.type === "LOGOUT") {
//         console.log("📢 Logout event received via BroadcastChannel");
//         socket.disconnect();
//         setSocketStatus("disconnected");
//       }
//     };

//     return () => {
//       socket.off("connect", handleConnect);
//       socket.off("disconnect", handleDisconnect);
//       socket.off("connect_error", handleConnectError);
//       socket.off("updateAdminUsersList", handleUpdateAdminUsersList);
//       socket.off("user:connected", handleUserConnected);
//       socket.off("user:disconnected", handleUserDisconnected);
//       channel.close();
//       socket.disconnect();
//     };
//   }, [socket]);

//   return (
//     <SocketContext.Provider value={socket}>
//       <SocketStatusContext.Provider value={socketStatus}>
//         <AdminUsersListContext.Provider value={adminUsersList}>
//           <SocketNotificationsContext.Provider value={notifications}>
//             {children}
//           </SocketNotificationsContext.Provider>
//         </AdminUsersListContext.Provider>
//       </SocketStatusContext.Provider>
//     </SocketContext.Provider>
//   );
// };

// export default SocketContextProvider;
