"use client";

import React, {
  createContext,
  useCallback,
  useContext,
  useEffect,
  useState,
} from "react";
import ApiCall from "@/api/api-calls";
import { toast } from "react-hot-toast";
import { useAuth } from "@/context/AuthContext";

export type AdminMenuItem = {
  id: string | number;
  adminmenutitle: string;
  adminmenulink?: string | null;
  icon?: string | null;
  children?: AdminMenuItem[];
};

type AdminMenuContextValue = {
  menuData: AdminMenuItem[];
  loadingMenu: boolean;
  refreshMenu: () => Promise<void>;
};

const AdminMenuContext = createContext<AdminMenuContextValue | null>(null);

export function AdminMenuProvider({ children }: { children: React.ReactNode }) {
  const { isAuthenticated } = useAuth();
  const [menuData, setMenuData] = useState<AdminMenuItem[]>([]);
  const [loadingMenu, setLoadingMenu] = useState(false);

  const fetchNavigation = useCallback(async () => {
    if (!isAuthenticated) {
      setMenuData([]);
      return;
    }
    try {
      setLoadingMenu(true);
      const response = await ApiCall.adminMenu.getAdminMenuTree();
      if (response?.status === 200) {
        setMenuData(response.data?.body || []);
      } else {
        toast.error("A menü betöltése nem sikerült");
      }
    } catch (err) {
      console.error("Sidebar fetch error:", err);
      toast.error("Hiba történt a menü betöltésekor");
    } finally {
      setLoadingMenu(false);
    }
  }, [isAuthenticated]);

  useEffect(() => {
    fetchNavigation();
  }, [fetchNavigation]);

  const value: AdminMenuContextValue = {
    menuData,
    loadingMenu,
    refreshMenu: fetchNavigation,
  };

  return (
    <AdminMenuContext.Provider value={value}>
      {children}
    </AdminMenuContext.Provider>
  );
}

export function useAdminMenu() {
  const ctx = useContext(AdminMenuContext);
  if (!ctx) {
    throw new Error("useAdminMenu must be used within an AdminMenuProvider");
  }
  return ctx;
}
