"use client";

import { useContext, useEffect, useState } from "react";
import { SocketContext } from "@/context/SocketProvider";

export default function useDocumentLock(docId) {
  const socket = useContext(SocketContext);
  const [lock, setLock] = useState({
    status: "unknown", // "unknown" | "granted" | "denied"
    owner: null,
  });

  useEffect(() => {
    if (!socket || !docId) return;

    const handleGranted = ({ docId: grantedDocId, owner }) => {
      if (grantedDocId !== docId) return;
      setLock({ status: "granted", owner });
    };

    const handleDenied = ({ docId: deniedDocId, owner }) => {
      if (deniedDocId !== docId) return;
      setLock({ status: "denied", owner });
    };

    const handleReleased = ({ docId: releasedDocId }) => {
      if (releasedDocId !== docId) return;
      setLock({ status: "unknown", owner: null });
    };

    socket.on("doc:lock:granted", handleGranted);
    socket.on("doc:lock:denied", handleDenied);
    socket.on("doc:lock:released", handleReleased);

    // ask for the lock when we mount
    socket.emit("doc:lock", { docId });

    return () => {
      socket.emit("doc:unlock", { docId });
      socket.off("doc:lock:granted", handleGranted);
      socket.off("doc:lock:denied", handleDenied);
      socket.off("doc:lock:released", handleReleased);
    };
  }, [socket, docId]);

  return lock;
}
