"use client";
import { useCallback, useEffect, useState, useRef } from "react";

const useInfiniteFetch = ({
  fetchItems, // (page, limit) => Promise<{ items, total }>
  limit = 20,
  initialPage = 1,
  autoLoadFirstPage = true,
  onSessionExpired = null,
  onError = null,
}) => {
  const [items, setItems] = useState([]);
  const [page, setPage] = useState(initialPage);
  const [loading, setLoading] = useState(false);
  const [hasMore, setHasMore] = useState(true);
  const [error, setError] = useState(null);

  const totalRef = useRef(null);

  const loadNext = useCallback(async () => {
    if (loading || !hasMore) return;

    setLoading(true);
    setError(null);
    try {
      const { items: newItems = [], total = 0 } = await fetchItems({
        page,
        limit,
      });

      setItems((prev) => [...prev, ...newItems]);
      setPage((prev) => prev + 1);
      totalRef.current = total;

      if (items.length + newItems.length >= total) {
        setHasMore(false);
      }
    } catch (err) {
      console.error("❌ useInfiniteFetch error:", err);
      setError(err);

      const isAuthError =
        err?.message?.toLowerCase().includes("session expired") ||
        err?.type === "auth";

      if (isAuthError && typeof onSessionExpired === "function") {
        onSessionExpired();
        return;
      }

      if (typeof onError === "function") onError(err);
    } finally {
      setLoading(false);
    }
  }, [page, loading, hasMore]);

  // Auto-initialize
  useEffect(() => {
    if (autoLoadFirstPage) loadNext();
  }, []);

  return {
    items,
    loadNext,
    loading,
    hasMore,
    error,
    reset: () => {
      setItems([]);
      setPage(initialPage);
      setHasMore(true);
      totalRef.current = null;
    },
  };
};

export default useInfiniteFetch;
