// src/hooks/useMyApprovalsBadge.tsx
"use client";

import {
  createContext,
  useCallback,
  useContext,
  useEffect,
  useState,
} from "react";
import ApiCall from "@/api/api-calls";
import { useAuth } from "@/context/AuthContext";

export type MyApprovalItem = {
  document_id: string;
  title: string;
  status: string;
  created_at: string;
  updated_at: string | null;
  person_name: string | null;
  approver_id: string;
  approver_status: string;
  approver_role: string | null;
  decided_at: string | null;
};

type MyApprovalsContextValue = {
  pendingCount: number;
  items: MyApprovalItem[];
  loadingSummary: boolean;
  loadingList: boolean;
  refreshSummary: () => Promise<void>;
  loadList: () => Promise<void>;
};

const MyApprovalsContext = createContext<MyApprovalsContextValue | null>(null);

export function MyApprovalsProvider({
  children,
}: {
  children: React.ReactNode;
}) {
  const { isAuthenticated } = useAuth();
  const [pendingCount, setPendingCount] = useState(0);
  const [items, setItems] = useState<MyApprovalItem[]>([]);
  const [loadingSummary, setLoadingSummary] = useState(false);
  const [loadingList, setLoadingList] = useState(false);

  const fetchSummary = useCallback(async () => {
    if (!isAuthenticated) {
      setPendingCount(0);
      return;
    }
    try {
      setLoadingSummary(true);
      const res = await ApiCall.hr.getMyApprovalsSummary();
      setPendingCount(res.data?.pendingCount ?? 0);
    } catch (e) {
      console.error("Failed to load approvals summary", e);
    } finally {
      setLoadingSummary(false);
    }
  }, [isAuthenticated]);

  const loadList = useCallback(async () => {
    if (!isAuthenticated) {
      setItems([]);
      setPendingCount(0); // also reset the badge
      return;
    }
    try {
      setLoadingList(true);
      const res = await ApiCall.hr.getMyApprovalsList();
      const list = (res.data?.items ?? []) as MyApprovalItem[];
      setItems(list);

      // 🔹 derive badge count from the list
      const pending = list.filter(
        (i) =>
          i.approver_status === "pending" && i.status === "pending_approval"
      ).length;
      setPendingCount(pending);
    } catch (e) {
      console.error("Failed to load approvals list", e);
    } finally {
      setLoadingList(false);
    }
  }, [isAuthenticated]);

  // polling
  useEffect(() => {
    fetchSummary();
    const interval = setInterval(fetchSummary, 60_000);
    return () => clearInterval(interval);
  }, [fetchSummary]);

  const value: MyApprovalsContextValue = {
    pendingCount,
    items,
    loadingSummary,
    loadingList,
    refreshSummary: fetchSummary,
    loadList,
  };

  return (
    <MyApprovalsContext.Provider value={value}>
      {children}
    </MyApprovalsContext.Provider>
  );
}

// hook used by components
export function useMyApprovals() {
  const ctx = useContext(MyApprovalsContext);
  if (!ctx) {
    throw new Error("useMyApprovals must be used within a MyApprovalsProvider");
  }
  return ctx;
}
