// hooks/useSocketAuth.js
"use client";
import { useEffect, useRef, useState } from "react";
import { io } from "socket.io-client";
import Cookies from "js-cookie";
import apiClient from "@/lib/apiClient";

const useSocketAuth = ({ onLogout }) => {
  const [socket, setSocket] = useState(null);
  const [loading, setLoading] = useState(false);
  const reconnectRef = useRef(null);

  const initializeSocket = async () => {
    try {
      setLoading(true);
      const accessToken = Cookies.get("edSId");
      if (!accessToken) return;

      const socketTokenRes = await apiClient.get(
        "/api/admin/auth/socket-token",
        {
          headers: { Authorization: `Bearer ${accessToken}` },
        }
      );

      const socketToken = socketTokenRes.data?.socketToken;
      if (!socketToken) throw new Error("Missing socket token");

      const newSocket = io(process.env.NEXT_PUBLIC_SOCKET_SERVER_URL, {
        path: "/socket.io",
        transports: ["websocket", "polling"],
        withCredentials: true,
        query: { token: socketToken },
      });

      newSocket.on("connect", () => {
        console.log("✅ Socket connected:", newSocket.id);
      });

      newSocket.on("connect_error", (err) => {
        console.error("❌ Socket connection error:", err.message);
      });

      newSocket.on("disconnect", async (reason) => {
        console.warn("⚠ Socket disconnected:", reason);
        if (reason === "io server disconnect") {
          await reconnectRef.current?.();
        }
      });

      reconnectRef.current = initializeSocket;
      setSocket(newSocket);
    } catch (err) {
      console.error("❌ Socket initialization error:", err);
      if (err?.response?.status === 401 && typeof onLogout === "function") {
        onLogout();
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    initializeSocket();
    return () => {
      socket?.disconnect();
    };
  }, []);

  return { socket, loading, reconnect: initializeSocket };
};

export default useSocketAuth;
