// lib/data-utils.js
import { ensureTokenFresh } from "@/lib/ensureTokenFresh";

/**
 * Wraps any API function with token freshness check (simple form).
 * Used for methods like `mediaStoreApi.getAll()` etc.
 */
export const withTokenRefresh = (apiCallFn) => {
  if (apiCallFn?.isWrapped) return apiCallFn;

  const wrappedFn = async (...args) => {
    const tokenFresh = await ensureTokenFresh();
    if (!tokenFresh.success) {
      throw new Error(tokenFresh.message || "Session expired.");
    }
    return apiCallFn(...args);
  };

  wrappedFn.isWrapped = true;
  return wrappedFn;
};

/**
 * Automatically applies `withTokenRefresh()` to specific methods in an API object.
 * Useful for wrapping selected API methods only.
 */
export const wrapApiWithTokenRefresh = (apiObject, methodsToWrap = []) => {
  const wrappedApi = {};

  Object.entries(apiObject).forEach(([key, fn]) => {
    if (methodsToWrap.includes(key) && typeof fn === "function") {
      wrappedApi[key] = withTokenRefresh(fn);
    } else {
      wrappedApi[key] = fn;
    }
  });

  return wrappedApi;
};

/**
 * Wraps any fetchItems-like function for infinite scroll with token check.
 * Used for custom data fetches, not tied to an API object.
 */
export const wrapFetchItemsWithRefresh = (fetchFn) => {
  if (fetchFn?.isWrapped) return fetchFn;

  const wrappedFetch = async (...args) => {
    const tokenStatus = await ensureTokenFresh();
    if (!tokenStatus.success) {
      const error = new Error(tokenStatus.message || "Session expired.");
      error.type = "auth";
      throw error;
    }
    return fetchFn(...args);
  };

  wrappedFetch.isWrapped = true;
  return wrappedFetch;
};

/**
 * Wraps paginated fetch functions that accept parameter objects: { page, limit, ... }
 * Example: wrapPaginationFetcher(({ page, limit, filters }) => apiCall(...))
 */
export const wrapPaginationFetcher = (fetchFn) => {
  if (fetchFn?.isWrapped) return fetchFn;

  const wrappedFetch = async (params = {}) => {
    const tokenStatus = await ensureTokenFresh();
    if (!tokenStatus.success) {
      const error = new Error(tokenStatus.message || "Session expired.");
      error.type = "auth";
      throw error;
    }
    return fetchFn(params);
  };

  wrappedFetch.isWrapped = true;
  return wrappedFetch;
};
