// lib/ensureTokenFresh.ts
import Cookies from "js-cookie";
import axios from "axios";
import apiClient from "@/lib/apiClient";

function decodeJwt(token: string): { exp?: number } | null {
  try {
    const [, payload] = token.split(".");
    return JSON.parse(atob(payload.replace(/-/g, "+").replace(/_/g, "/")));
  } catch {
    return null;
  }
}

const SKEW_SEC = 60;

export async function ensureTokenFresh(): Promise<{
  success: boolean;
  message?: string;
}> {
  try {
    const token = Cookies.get("edSId");

    if (!token) {
      const ok = await refreshAccessToken();
      return ok
        ? { success: true }
        : { success: false, message: "Session expired. Please log in again." };
    }

    const decoded = decodeJwt(token);
    const now = Math.floor(Date.now() / 1000);
    const exp = decoded?.exp ?? 0;

    if (exp <= now + SKEW_SEC) {
      const ok = await refreshAccessToken();
      return ok
        ? { success: true }
        : { success: false, message: "Session expired. Please log in again." };
    }

    return { success: true };
  } catch {
    return { success: false, message: "Session expired. Please log in again." };
  }
}

async function refreshAccessToken(): Promise<boolean> {
  try {
    // Build a safe URL for browser & SSR
    const base = apiClient.defaults.baseURL || ""; // use relative path on client
    const url = `${base}/api/admin/auth/refresh-token`;

    const { data } = await axios.get(url, { withCredentials: true });
    const newToken = data?.accessToken;
    if (!newToken) return false;

    Cookies.set("edSId", newToken, { expires: 1 });
    apiClient.defaults.headers.common["Authorization"] = `Bearer ${newToken}`;

    if (typeof window !== "undefined") {
      window.dispatchEvent(
        new CustomEvent("token-refreshed", { detail: newToken })
      );
    }
    return true;
  } catch (e) {
    Cookies.remove("edSId");
    return false;
  }
}
