import { getBaseServerUrl } from "./getBaseUrl";

// ✅ GET with cache / revalidate / tags (SSR safe)
export async function fetchWithCache(path = "", options: RequestInit = {}) {
  try {
    const url = `${getBaseServerUrl()}${path}`;
    const res = await fetch(url, { method: "GET", ...options });

    if (!res.ok) {
      const message = await res.text();
      throw new Error(`❌ ${res.status}: ${message}`);
    }

    const data = await res.json();
    if (process.env.NODE_ENV === "development") {
      console.log("✅ fetchWithCache:", url, data);
    }
    return data;
  } catch (err: any) {
    console.error("🛑 fetchWithCache:", err.message);
    return null;
  }
}

// ✅ POST with JSON body (no caching)
export async function fetchPostJson(
  path = "",
  payload: any = {},
  headers: HeadersInit = {}
) {
  try {
    const url = `${getBaseServerUrl()}${path}`;
    const res = await fetch(url, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        ...headers,
      },
      body: JSON.stringify(payload),
      cache: "no-store",
    });

    if (!res.ok) {
      const message = await res.text();
      throw new Error(`❌ ${res.status}: ${message}`);
    }

    const data = await res.json();
    return data;
  } catch (err: any) {
    console.error("🛑 fetchPostJson:", err.message);
    return null;
  }
}

// ✅ GET with query string builder
export function buildQueryParams(
  basePath: string,
  params: Record<string, any> = {}
) {
  const search = new URLSearchParams(
    Object.entries(params).reduce<Record<string, string>>((acc, [k, v]) => {
      if (v !== undefined && v !== null) acc[k] = String(v);
      return acc;
    }, {})
  ).toString();

  return `${basePath}${search ? "?" + search : ""}`;
}

export async function fetchWithQueryParams(
  path: string,
  params: Record<string, any> = {},
  options: RequestInit = {}
) {
  const url = buildQueryParams(path, params);
  return await fetchWithCache(url, options);
}

// ✅ Base auth fetch (takes a token)
export async function fetchWithAuth(
  path: string,
  token: string,
  options: RequestInit & {
    responseType?: "json" | "text" | "blob" | "arrayBuffer" | "raw";
  } = {}
) {
  const { responseType = "json", ...fetchOptions } = options;
  const url = `${getBaseServerUrl()}${path}`;

  if (!token) {
    throw new Error("❌ fetchWithAuth: missing auth token");
  }

  const res = await fetch(url, {
    ...fetchOptions,
    headers: {
      ...(fetchOptions.headers || {}),
      Authorization: `Bearer ${token}`,
    },
  });

  if (!res.ok) {
    let errorText: string;
    try {
      errorText = await res.text();
    } catch {
      errorText = "<no error body>";
    }
    throw new Error(`❌ Auth fetch failed: ${res.status} ${errorText}`);
  }

  switch (responseType) {
    case "text":
      return res.text();
    case "blob":
      return res.blob();
    case "arrayBuffer":
      return res.arrayBuffer();
    case "raw":
      return res; // caller handles Response directly
    case "json":
    default:
      return res.json();
  }
}
