// lib/withTokenRefresh.js
import { ensureTokenFresh } from "./ensureTokenFresh";

export const withTokenRefresh = (apiCallFn) => {
  if (apiCallFn?.isWrapped) return apiCallFn; // prevent wrapping again

  const wrappedFn = async (...args) => {
    const tokenFresh = await ensureTokenFresh();
    if (!tokenFresh.success) {
      throw new Error(tokenFresh.message || "Session expired.");
    }
    return apiCallFn(...args);
  };

  wrappedFn.isWrapped = true; // ✅ Mark as wrapped
  return wrappedFn;
};

// ✅ NEW: Automatically apply withTokenRefresh to selected methods in an API object
export const wrapApiWithTokenRefresh = (apiObject, methodsToWrap = []) => {
  const wrappedApi = {};

  Object.entries(apiObject).forEach(([key, fn]) => {
    if (methodsToWrap.includes(key) && typeof fn === "function") {
      wrappedApi[key] = withTokenRefresh(fn);
    } else {
      wrappedApi[key] = fn;
    }
  });

  return wrappedApi;
};
