// redux/slices/productGroupSlice.js
import { createSlice } from "@reduxjs/toolkit";
import {
  containerInitialState,
  getContainerReducers,
} from "../factory/containerSliceFactory";
import {
  columnInitialState,
  getColumnReducers,
} from "../factory/columnSliceFactory";
import {
  elementInitialState,
  getElementReducers,
} from "../factory/elementSliceFactory";
import {
  galleryInitialState,
  getGalleryReducers,
} from "../factory/gallerySliceFactory";

// Custom metadata fields
const customInitialState = {
  productGroupCategory: [],
  productGroupFooterCategory: [],
  productTagList: [],
  productColorsList: [],
  productSizesList: [],
  productGroupMark: [],
  connectedProductsList: [],
  current: [],
  productPriceType: [],
};

const initialState = {
  ...containerInitialState,
  ...columnInitialState,
  ...elementInitialState,
  ...galleryInitialState,
  ...customInitialState,
};
const clamp = (n, min, max) => Math.max(min, Math.min(max, n));

const moveItem = (arr, from, to) => {
  if (from === to) return arr;
  const item = arr.splice(from, 1)[0];
  arr.splice(to, 0, item);
  return arr;
};

// Remap innerId (column indexes) for all blocks in a container when a column moves
const remapInnerIdsOnColumnMove = (content, fromIdx, toIdx) => {
  for (const blk of content ?? []) {
    if (blk?.innerId === undefined) continue;
    const inner = Number(blk.innerId);
    if (inner === fromIdx) {
      blk.innerId = toIdx;
    } else if (fromIdx < toIdx && inner > fromIdx && inner <= toIdx) {
      blk.innerId = inner - 1;
    } else if (fromIdx > toIdx && inner >= toIdx && inner < fromIdx) {
      blk.innerId = inner + 1;
    }
  }
};

// Ensure innerElementIndex is continuous 0..n-1
const reindexElements = (block) => {
  (block?.content ?? []).forEach((el, i) => (el.innerElementIndex = i));
};

// Find or create a column block (by parentId/innerId)
const getOrCreateColumnBlock = (container, parentIdx, columnIdx) => {
  let blk = (container.content ?? []).find(
    (b) => Number(b.parentId) === parentIdx && Number(b.innerId) === columnIdx
  );
  if (!blk) {
    blk = { parentId: parentIdx, innerId: columnIdx, content: [] };
    if (!container.content) container.content = [];
    container.content.push(blk);
  }
  return blk;
};
const productGroupSlice = createSlice({
  name: "productGroup",
  initialState,
  reducers: {
    resetProductGroupState: () => initialState,
    // Custom product metadata
    setProductGroupCategory: (state, action) => {
      state.productGroupCategory = action.payload.category;
    },
    setProductGroupFooterCategories: (state, action) => {
      state.productGroupFooterCategory = action.payload.footerCategories || [];
    },
    setProductGroupTags: (state, action) => {
      state.productTagList = action.payload.tags;
    },
    setProductGroupColors: (state, action) => {
      state.productColorsList = action.payload.colors || [];
    },
    setProductGroupSizes: (state, action) => {
      state.productSizesList = action.payload.sizes || [];
    },
    setProductGroupMark: (state, action) => {
      state.productGroupMark = action.payload.mark || null; // Store only ID
    },
    setProductGroupPriceType: (state, action) => {
      state.productPriceType = action.payload.priceType; // Store only string
    },
    setProductGroupConnectedProducts: (state, action) => {
      state.connectedProductsList = action.payload.connectedProducts;
    },
    // Initial load data mapping
    loadProductGroupInitialData: (state, action) => {
      const { data, productProperties } = action.payload;

      if (data) {
        state.productGroupCategory =
          productProperties.selected.categories || [];
        state.productGroupFooterCategory =
          productProperties.selected.footerCategories || [];
        state.productTagList = productProperties.selected.tags || [];
        state.productColorsList = productProperties.selected.colors || [];
        state.productSizesList = productProperties.selected.sizes || [];
        state.productGroupMark = productProperties.selected.marks?.[0] || null;
        state.productPriceType = productProperties.selected.currencies || null;
        state.connectedProductsList =
          productProperties?.selected.connectedGroups || [];
        state.featuredImage = productProperties.selected.featuredImage;
      }
    },
    reorderContainersByPosition: (state, action) => {
      const { fromIndex, toPos1Based } = action.payload;
      const arr = state.containerData ?? [];
      if (!arr.length) return;
      const from = clamp(fromIndex, 0, arr.length - 1);
      const to = clamp((toPos1Based ?? 1) - 1, 0, arr.length - 1);
      moveItem(arr, from, to);

      // After reordering containers, fix every block.parentId to match its new container index
      arr.forEach((container, idx) => {
        (container.content ?? []).forEach((blk) => (blk.parentId = idx));
      });
    },

    // 2) COLUMNS: move a column within same container to position (1-based)
    reorderColumnsByPosition: (state, action) => {
      const { containerIndex, fromIndex, toPos1Based } = action.payload;
      const container = state.containerData?.[containerIndex];
      if (!container) return;

      const size = container.size ?? [];
      if (!size.length) return;

      const from = clamp(fromIndex, 0, size.length - 1);
      const to = clamp((toPos1Based ?? 1) - 1, 0, size.length - 1);

      // Move visual slot (widths)
      moveItem(size, from, to);

      // Remap blocks' innerId to follow the index change
      remapInnerIdsOnColumnMove(container.content ?? [], from, to);
    },

    // 3) ELEMENTS: move element anywhere by container/column/position (1-based position)
    moveElementByPosition: (state, action) => {
      const {
        fromContainerIndex,
        fromColumnIndex,
        fromElementIndex,
        toContainerIndex,
        toColumnIndex,
        toPos1Based,
      } = action.payload;

      const fromC = state.containerData?.[fromContainerIndex];
      const toC = state.containerData?.[toContainerIndex];
      if (!fromC || !toC) return;

      const fromBlock = getOrCreateColumnBlock(
        fromC,
        fromContainerIndex,
        fromColumnIndex
      );
      const toBlock = getOrCreateColumnBlock(
        toC,
        toContainerIndex,
        toColumnIndex
      );

      const fromArr = fromBlock.content ?? [];
      const toArr = toBlock.content ?? [];

      if (
        !fromArr.length ||
        fromElementIndex < 0 ||
        fromElementIndex >= fromArr.length
      )
        return;

      const [moved] = fromArr.splice(fromElementIndex, 1);
      // Update its parent/inner ids
      moved.parentId = toContainerIndex;
      moved.innerId = toColumnIndex;

      const insertAt = clamp(
        (toPos1Based ?? toArr.length + 1) - 1,
        0,
        toArr.length
      );
      toArr.splice(insertAt, 0, moved);

      // Reindex
      reindexElements(fromBlock);
      reindexElements(toBlock);
    },
    // Inject factory reducers
    ...getContainerReducers(),
    ...getColumnReducers(),
    ...getElementReducers(),
    ...getGalleryReducers(),
  },
});

export const {
  resetProductGroupState,
  setProductGroupCategory,
  setProductGroupFooterCategories,
  setProductGroupTags,
  setProductGroupColors,
  setProductGroupSizes,
  setProductGroupMark,
  setProductGroupPriceType,
  setProductGroupConnectedProducts,
  loadProductGroupInitialData,
  reorderContainersByPosition,
  reorderColumnsByPosition,
  moveElementByPosition,
  // Gallery-related actions injected via factory
  setFeaturedImage,
  resetFeaturedImageState,
  setSelectedGalleryImages,
  removeImageFromGallery,
  updateGalleryImage,
  updateGalleryImageOrder,
  resetGalleryState,
  selectedSliderImages,
  updateSliderImage,
  removeSingleSliderImage,
  updateSliderImageOrder,
  cloneSliderImage,
  resetSliderState,
  setSelectedImage,
  resetImageState,
  // Container-related actions injected via factory
  addContainer,
  setSelectedContainer,
  removeContainer,
  cleanCurrentlySelectedContent,
  setContainerList,
  updateContainerContent,
  selectedContainerOptions,
  setContainerTitle,
  resetContainerState,
  // Column-related actions injected via factory
  addColumn,
  setSelectedColumn,
  updateColumnOrder,
  updateColumnOptions,
  changeColumnSize,
  cloneColumn,
  removeColumn,
  // Element-related actions injected via factory
  selectedElement,
  cloneElement,
  updateColumnElementOrder,
  moveColumnElement,
  removeElement,
  // Common props actions injected via factory
  publishedPage,
  activeLayout,
  staticPage,
  hasParent,
  showInMenu,
  resetCommonPropsState,
} = productGroupSlice.actions;

export default productGroupSlice.reducer;
